package com.paypal.action;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.ArrayList;
import java.util.List;

import static de.hybris.platform.testframework.Assert.assertEquals;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalTakePaymentActionTest {

    private static final String TRANSITION_OK = "OK";
    private static final String TRANSITION_NOK = "NOK";

    @InjectMocks
    private PayPalTakePaymentAction payPalTakePaymentAction;

    @Mock
    private PaymentService paymentService;

    @Mock
    private ModelService modelService;

    @Mock
    private PaymentTransactionModel mockTransaction;
    @Mock
    private OrderProcessModel mockOrderProcess;
    @Mock
    private OrderModel mockOrder;
    @Mock
    private PayPalCreditCardPaymentInfoModel mockPaypalPaymentInfo;
    @Mock
    private CreditCardPaymentInfoModel mockCCPaymentInfo;
    @Mock
    private PaymentTransactionEntryModel mockPaymentTransactionEntry;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        ArrayList<PaymentTransactionModel> mockTransactionArr = new ArrayList<>(List.of(mockTransaction));
        when(mockOrderProcess.getOrder()).thenReturn(mockOrder);
        when(mockOrder.getPaymentTransactions()).thenReturn(mockTransactionArr);
        when(mockTransaction.getEntries()).thenReturn(List.of(mockPaymentTransactionEntry));
    }

    @Test
    public void shouldSucceedWhenTransactionEntryCapturedOrPartiallyCapturedForPaypal() {
        when(mockOrder.getPaymentInfo()).thenReturn(mockPaypalPaymentInfo);
        when(mockPaymentTransactionEntry.getType()).thenReturn(PaymentTransactionType.CAPTURE)
                .thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);

        String actualCapture = payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        assertEquals(TRANSITION_OK, actualCapture);

        String actualPartialCapture = payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        assertEquals(TRANSITION_OK, actualPartialCapture);
    }

    @Test
    public void shouldSucceedWhenPaymentIsHostedFieldsAndTransactionAccepted() {
        when(mockOrder.getPaymentInfo()).thenReturn(mockCCPaymentInfo);
        when(mockTransaction.getInfo()).thenReturn(mockCCPaymentInfo);
        when(paymentService.capture(mockTransaction)).thenReturn(mockPaymentTransactionEntry);
        when(mockPaymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        String actualTransactionAccepted = payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        verify(mockOrder).setStatus(OrderStatus.PAYMENT_CAPTURED);
        verify(modelService).save(mockOrder);

        assertEquals(TRANSITION_OK, actualTransactionAccepted);
    }

    @Test
    public void shouldProceedWhenPaymentIsHostedFieldsAndTransactionRejected() {
        when(mockOrder.getPaymentInfo()).thenReturn(mockCCPaymentInfo);
        when(mockTransaction.getInfo()).thenReturn(mockCCPaymentInfo);
        when(paymentService.capture(mockTransaction)).thenReturn(mockPaymentTransactionEntry);
        when(mockPaymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        verify(mockOrder).setStatus(OrderStatus.PAYMENT_NOT_CAPTURED);
        verify(modelService).save(mockOrder);
    }

    @Test
    public void shouldProceedWithPayPalPMWhenTransactionTypeAuthorizeAndTransactionRejected() {
        when(mockOrder.getPaymentInfo()).thenReturn(mockPaypalPaymentInfo);
        when(mockPaymentTransactionEntry.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(mockTransaction.getInfo()).thenReturn(mockPaypalPaymentInfo);
        when(paymentService.capture(mockTransaction)).thenReturn(mockPaymentTransactionEntry);
        when(mockPaymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        String actual = payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        verify(mockOrder).setStatus(OrderStatus.PAYMENT_NOT_CAPTURED);
        verify(modelService).save(mockOrder);

        assertEquals(TRANSITION_NOK, actual);
    }

    @Test
    public void shouldProceedWithPayPalPMWhenTransactionTypeAuthorizeAndTransactionAccepted() {
        when(mockOrder.getPaymentInfo()).thenReturn(mockPaypalPaymentInfo);
        when(mockPaymentTransactionEntry.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(mockTransaction.getInfo()).thenReturn(mockPaypalPaymentInfo);
        when(paymentService.capture(mockTransaction)).thenReturn(mockPaymentTransactionEntry);
        when(mockPaymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        String actual = payPalTakePaymentAction.executeAction(mockOrderProcess).toString();

        verify(mockOrder).setStatus(OrderStatus.PAYMENT_CAPTURED);
        verify(modelService).save(mockOrder);

        assertEquals(TRANSITION_OK, actual);
    }


}
