package com.paypal.action;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.ticket.enums.CsTicketCategory;
import de.hybris.platform.ticket.enums.CsTicketPriority;
import de.hybris.platform.ticket.events.model.CsCustomerEventModel;
import de.hybris.platform.ticket.model.CsTicketModel;
import de.hybris.platform.ticket.service.TicketBusinessService;
import org.assertj.core.util.Lists;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;

@UnitTest
public class PayPalCheckTransactionReviewStatusActionTest {

    private static final String ACCEPTED = "ACCEPTED";
    private static final String TRANSITION_OK = "OK";
    private static final String TRANSITION_NOK = "NOK";
    private static final String TRANSITION_WAIT = "WAIT";
    private static final String TICKET_TITLE = "ticketTitle";
    private static final String TICKET_KEY = "message.ticket.orderinreview.title";
    private static final String AUTHORIZE = "authorize";
    private static final String ORDER_CODE = "order code";
    private static final String SUBJECT = "subject";
    private static final String DESCRIPTION = "description";
    @InjectMocks
    @Spy
    private PayPalCheckTransactionReviewStatusAction payPalCheckTransactionReviewStatusAction;

    @Mock
    private OrderProcessModel mockOrderProcessModel;

    @Mock
    private ModelService modelService;

    @Mock
    private TicketBusinessService ticketBusinessService;

    @Mock
    private PaymentTransactionModel mockTransaction;

    @Mock
    private PaymentTransactionEntryModel mockPaymentTransactionEntryModel;

    @Mock
    private OrderModel mockOrderModel;

    @Mock
    private PaymentInfoModel mockPaymentInfoModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel mockPaypalPaymentInfoModel;

    @Mock
    private CsTicketCategory csTicketCategory;

    @Mock
    private CsTicketPriority csTicketPriority;

    @Mock
    private UserModel mockUserModel;

    @Mock
    private PayPalPaymentInfoService paymentInfoService;

    @Captor
    private ArgumentCaptor<CsTicketModel> csTicketModelArgumentCaptor;

    @Captor
    private ArgumentCaptor<CsCustomerEventModel> customerEventModelArgumentCaptor;


    private ArrayList<PaymentTransactionEntryModel> mockEntryTransactionArr;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        ArrayList<PaymentTransactionModel> mockTransactionArr = new ArrayList<>(List.of(mockTransaction));
        mockEntryTransactionArr = new ArrayList<>(List.of(mockPaymentTransactionEntryModel));

        when(mockOrderProcessModel.getOrder()).thenReturn(mockOrderModel);
        when(mockOrderModel.getPaymentTransactions()).thenReturn(mockTransactionArr);
    }

    @Test
    public void shouldSucceedWhenTransactionEntriesAreNotPresent() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(Lists.emptyList());

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);

        verify(ticketBusinessService, never()).createTicket(any(), any());
        verify(modelService, never()).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWhenReviewDecisionIsAccepted() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.REVIEW_DECISION);
        when(mockPaymentTransactionEntryModel.getTransactionStatus()).thenReturn(ACCEPTED);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);

        verify(mockOrderModel).setStatus(OrderStatus.PAYMENT_AUTHORIZED);
        verify(modelService).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWhenOrderIsNull() throws Exception {
        when(mockOrderProcessModel.getOrder()).thenReturn(null);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);

        verify(ticketBusinessService, never()).createTicket(any(), any());
        verify(modelService, never()).save(mockOrderModel);
    }

    @Test
    public void shouldFailWhenReviewIsNotAccepted() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.REVIEW_DECISION);
        when(mockPaymentTransactionEntryModel.getTransactionStatus()).thenReturn("NOTACCEPTED");

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_NOK, actual);

        verify(mockOrderModel).setStatus(OrderStatus.PAYMENT_NOT_AUTHORIZED);
        verify(modelService).save(mockOrderModel);
    }

    @Test
    public void shouldWaitWithAuthorizeEntryTypeForPaypal() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaypalPaymentInfoModel);
        when(mockPaypalPaymentInfoModel.getIntent()).thenReturn(AUTHORIZE);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_WAIT, actual);

        verify(mockOrderModel).setStatus(OrderStatus.PAYMENT_NOT_CAPTURED);
        verify(modelService).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWithAuthorizeEntryTypeForCC() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);
    }

    @Test
    public void shouldWaitWhenAuthorizationInReview() throws Exception {
        doReturn(TICKET_KEY).when(payPalCheckTransactionReviewStatusAction)
                .getLocalizedStringForTicketMessage(mockOrderModel);
        doReturn(TICKET_TITLE).when(payPalCheckTransactionReviewStatusAction).getLocalizedStringForTicketTitle();
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaypalPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REVIEW.name());
        when(mockPaypalPaymentInfoModel.getIntent()).thenReturn(AUTHORIZE);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(mockOrderModel.getCode()).thenReturn(ORDER_CODE);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_WAIT, actual);

        verify(modelService).attach(any());
        verify(ticketBusinessService).createTicket(any(), any());

        verify(mockOrderProcessModel, times(1)).getOrder();
        verify(mockPaymentTransactionEntryModel, times(1)).getTransactionStatus();
        verify(mockTransaction, times(1)).getInfo();
        verify(mockPaymentTransactionEntryModel, times(2)).getType();
        verify(mockOrderModel).setStatus(OrderStatus.SUSPENDED);
        verify(modelService).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWithAuthorizeEntryTypeWhenNotInReviewOrNotCC() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);
        verify(ticketBusinessService, never()).createTicket(any(), any());
        verify(modelService, never()).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWithCaptureEntryType() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);

        String actualCapture = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actualCapture);
        verify(ticketBusinessService, never()).createTicket(any(), any());
        verify(modelService, never()).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWithPartialCaptureEntryType() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);

        String actualPartialCapture = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actualPartialCapture);
        verify(ticketBusinessService, never()).createTicket(any(), any());
        verify(modelService, never()).save(mockOrderModel);
    }

    @Test
    public void shouldSucceedWithCancelEntryType() throws Exception {
        when(mockTransaction.getEntries()).thenReturn(mockEntryTransactionArr);
        when(mockTransaction.getInfo()).thenReturn(mockPaymentInfoModel);
        when(mockPaymentTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.CANCEL);

        String actual = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        assertEquals(TRANSITION_OK, actual);
    }

    @Test
    public void shouldReturnStringRepresentationOfTransitions() {
        Set<String> actual =  payPalCheckTransactionReviewStatusAction.getTransitions();

        assertEquals(actual, Set.of(TRANSITION_OK, TRANSITION_NOK, TRANSITION_WAIT));
    }

    @Test
    public void createTicketShouldReturnTicket() {
        when(mockOrderModel.getUser()).thenReturn(mockUserModel);

        payPalCheckTransactionReviewStatusAction.createTicket(SUBJECT, DESCRIPTION, mockOrderModel, csTicketCategory, csTicketPriority);

        verify(ticketBusinessService).createTicket(csTicketModelArgumentCaptor.capture(), customerEventModelArgumentCaptor.capture());

        CsTicketModel csTicketModel = csTicketModelArgumentCaptor.getValue();
        CsCustomerEventModel customerEventModel = customerEventModelArgumentCaptor.getValue();
        verify(modelService).attach(csTicketModel);

        assertEquals(DESCRIPTION, customerEventModel.getText());
        assertEquals(SUBJECT, csTicketModel.getHeadline());
        assertEquals(csTicketCategory, csTicketModel.getCategory());
        assertEquals(csTicketPriority, csTicketModel.getPriority());
        assertEquals(mockOrderModel, csTicketModel.getOrder());
        assertEquals(mockUserModel, csTicketModel.getCustomer());
    }

    @Test
    public void shouldSetPaymentNotAuthorizedStatusWhenSaveOrderFlowActive() throws Exception{
        when(mockOrderProcessModel.getOrder()).thenReturn(mockOrderModel);
        when(mockOrderModel.getPaymentInfo()).thenReturn(mockPaypalPaymentInfoModel);
        when(paymentInfoService.isPayPalSaveOrderFlowActive(mockPaypalPaymentInfoModel)).thenReturn(true);
        when(mockOrderModel.getPaymentTransactions()).thenReturn(Collections.emptyList());

        String transition = payPalCheckTransactionReviewStatusAction.execute(mockOrderProcessModel);

        verify(mockOrderModel).setStatus(OrderStatus.PAYMENT_NOT_AUTHORIZED);
        verify(modelService).save(mockOrderModel);
        assertEquals(TRANSITION_WAIT, transition);
    }
}
