package com.paypal.hybris.facade.strategy.payment.impl.breakdown;


import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.util.DiscountValue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

import java.math.BigDecimal;
import java.util.List;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.class)
public class BreakdownCalculationServiceTest {

	private static final BigDecimal ITEMS_TOTAL_AMOUNT = new BigDecimal("100.00");
	private static final BigDecimal ITEMS_DISCOUNT_AMOUNT = new BigDecimal("20.0");
	@Mock
	private AbstractOrderEntryModel entryModel;
	@Mock
	private AbstractOrderModel orderModel;
	@InjectMocks
	private BreakdownCalculationService unit;
	private DiscountValue dValue;
	private DiscountValue globalDValue;

	@Before
	public void setUp() {
		when(orderModel.getEntries()).thenReturn(List.of(entryModel));

		dValue = new DiscountValue("DCODE", 0, true, 10.0,"USD");
		globalDValue = new DiscountValue("NOTQuoteDiscount", 0, true, 10.0,"USD");

		when(entryModel.getDiscountValues()).thenReturn(List.of(dValue));
		when(orderModel.getGlobalDiscountValues()).thenReturn(List.of(globalDValue));
	}

	@Test
	public void shouldCalculateItemsTotal() {
		when(entryModel.getBasePrice()).thenReturn(10d);
		when(entryModel.getQuantity()).thenReturn(10L);

		BigDecimal itemsTotal = unit.calculateItemsAmount(orderModel);

		assertThat(itemsTotal).isEqualTo(ITEMS_TOTAL_AMOUNT);
	}

	@Test
	public void shouldCorrectlyCalculateTotalDiscountWhenItemsDiscountAndGlobalDiscountPresent() {
		when(entryModel.getDiscountValues()).thenReturn(List.of(dValue));
		when(orderModel.getGlobalDiscountValues()).thenReturn(List.of(globalDValue));

		BigDecimal itemsTotal = unit.calculateTotalDiscount(orderModel);

		assertThat(itemsTotal).isEqualByComparingTo(ITEMS_DISCOUNT_AMOUNT);
	}

	@Test
	public void shouldCorrectlyCalculateTotalDiscountWhenItemsDiscountsArePresentGlobalMissing() {
		when(orderModel.getGlobalDiscountValues()).thenReturn(null);

		BigDecimal itemsTotal = unit.calculateTotalDiscount(orderModel);

		assertThat(itemsTotal).isEqualByComparingTo(BigDecimal.TEN);
	}

	@Test
	public void shouldCorrectlyCalculateTotalDiscountWhenGlobalDiscountArePresentItemsMissing() {
		when(entryModel.getDiscountValues()).thenReturn(null);

		BigDecimal itemsTotal = unit.calculateTotalDiscount(orderModel);

		assertThat(itemsTotal).isEqualByComparingTo(BigDecimal.TEN);
	}

	@Test
	public void shouldNotAddGlobalDiscountsWithQuoteDiscountCode() {
		globalDValue = new DiscountValue("QuoteDiscount", 0, true, 10.0,"USD");
		when(orderModel.getGlobalDiscountValues()).thenReturn(List.of(globalDValue));

		BigDecimal itemsTotal = unit.calculateTotalDiscount(orderModel);

		assertThat(itemsTotal).isEqualByComparingTo(BigDecimal.TEN);
	}

	@Test
	public void shouldReturnOnlyGlobalDiscountsWhenNoEntries() {
		when(orderModel.getEntries()).thenReturn(null);

		BigDecimal itemsTotal = unit.calculateTotalDiscount(orderModel);

		assertThat(itemsTotal).isEqualByComparingTo(BigDecimal.TEN);
	}
}