package com.paypal.hybris.facade.strategy.payment.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.SetupTokenRequestData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalPopulatorStrategyTest {

    private static final String RETURN_URL = "https://example.com/returnUrl";
    private static final String CANCEL_URL = "https://example.com/cancelUrl";
    private static final String PAYMENT_METHOD_CODE = "79895";
    private static final String VAULT_CUSTOMER_ID = "123645";
    private static final String USAGE_TYPE = "MERCHANT";
    private static final String PAYER_ID = "payerId";

    @Mock
    private UserService userService;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private PayPalCustomerAccountService payPalCustomerAccountService;

    private PayPalSavePaymentForPurchaseLaterRequest purchaseLaterRequest;
    private SetupTokenRequestData setupTokenRequestData;


    @InjectMocks
    private DefaultPayPalPopulatorStrategy unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        purchaseLaterRequest = new PayPalSavePaymentForPurchaseLaterRequest();
        setupTokenRequestData = getSourceData();

        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(payPalCustomerAccountService.getCustomerModel(PAYER_ID)).thenReturn(customerModel);
    }

    @Test
    public void shouldPopulateData() {
        when(customerModel.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);

        unit.populate(setupTokenRequestData, purchaseLaterRequest);

        assertEquals(VAULT_CUSTOMER_ID, purchaseLaterRequest.getCustomer().getId());
        assertEquals(RETURN_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getReturnUrl());
        assertEquals(CANCEL_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getCancelUrl());
        assertEquals(USAGE_TYPE, purchaseLaterRequest.getPaymentSource().getPaypal().getUsageType());
    }

    @Test
    public void shouldPopulateDataWhenPayerIdIsNotPresentOnCustomer() {
        when(customerModel.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);

        unit.populate(setupTokenRequestData, purchaseLaterRequest);

        assertEquals(VAULT_CUSTOMER_ID, purchaseLaterRequest.getCustomer().getId());
        assertEquals(RETURN_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getReturnUrl());
        assertEquals(CANCEL_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getCancelUrl());
        assertEquals(USAGE_TYPE, purchaseLaterRequest.getPaymentSource().getPaypal().getUsageType());
    }

    @Test
    public void shouldPopulateDataWhenCustomerVaultIdIsEmpty() {
        when(customerModel.getVaultCustomerId()).thenReturn(null);

        unit.populate(setupTokenRequestData, purchaseLaterRequest);

        assertNull(purchaseLaterRequest.getCustomer().getId());
        assertEquals(RETURN_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getReturnUrl());
        assertEquals(CANCEL_URL, purchaseLaterRequest.getPaymentSource().getPaypal().getExperienceContext().getCancelUrl());
        assertEquals(USAGE_TYPE, purchaseLaterRequest.getPaymentSource().getPaypal().getUsageType());
    }

    @Test
    public void shouldReturnTrueWhenPaymentProviderIsPayPal() {
        boolean result = unit.test(PayPalPaymentProvider.PAYPAL);

        assertTrue(result);
    }

    private SetupTokenRequestData getSourceData() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentMethodCode(PAYMENT_METHOD_CODE);
        setupTokenRequestData.setPayerId(PAYER_ID);

        return setupTokenRequestData;
    }

}
