package com.paypal.hybris.facade.strategy.payment.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.enums.ThreeDsVerificationOnMyAccountType;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.SetupTokenRequestData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorservices.urlresolver.SiteBaseUrlResolutionService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultCreditCardPopulatorStrategyTest {

    private static final String NAME_ON_CARD = "Tim";
    private static final String EXPIRY_YEAR = "2030";
    private static final String EXPIRY_MONTH = "10";
    private static final String CARD_NUMBER = "4005519200000004";
    private static final String CVV = "123";
    private static final String ADDRESS_LINE1 = "addressLine1";
    private static final String ADDRESS_LINE2 = "addressLine2";
    private static final String POSTAL_CODE = "postalCode";
    private static final String COUNTRY_ISO_CODE = "countryIsoCode";
    private static final String REGION_ISO_CODE = "regionIsoCode";
    private static final String TOWN = "Sacramento";
    private static final String MY_ACCOUNT_PAYMENT_DETAILS_APPROVE_LINK = "/my-account/payment-details/approved?guid=%s";
    private static final String MY_ACCOUNT_PAYMENT_DETAILS_CANCEL_LINK = "/my-account/payment-details/cancelled?guid=%s";
    private static final String RETURN_URL = "/return";
    private static final String CANCEL_URL = "/cancel";
    private static final String VAULT_CUSTOMER_ID = "123645";
    private static final String PAYMENT_METHOD_CODE = "79895";


    @Mock
    private UserService userService;
    @Mock
    private PayPalConfigurationService payPalConfigurationService;
    @Mock
    private BaseSiteService baseSiteService;
    @Mock
    private SiteBaseUrlResolutionService siteBaseUrlResolutionService;
    @Mock
    private BaseSiteModel baseSiteModel;
    @Mock
    private CustomerModel customerModel;

    private SetupTokenRequestData setupTokenRequestData;

    @InjectMocks
    private DefaultCreditCardPopulatorStrategy unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        setupTokenRequestData = getSourceData();

        when(payPalConfigurationService.getPayPalVerifyCCOnMyAccount()).thenReturn(ThreeDsVerificationOnMyAccountType.SCA_WHEN_REQUIRED.getCode());
        when(baseSiteService.getCurrentBaseSite()).thenReturn(baseSiteModel);
        when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel, true,
                String.format(MY_ACCOUNT_PAYMENT_DETAILS_APPROVE_LINK, setupTokenRequestData.getPaymentMethodCode()))).thenReturn(RETURN_URL);
        when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel, true,
                String.format(MY_ACCOUNT_PAYMENT_DETAILS_CANCEL_LINK, setupTokenRequestData.getPaymentMethodCode()))).thenReturn(CANCEL_URL);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);
    }

    @Test
    public void shouldPopulateData() {
        PayPalSavePaymentForPurchaseLaterRequest purchaseLaterRequest = new PayPalSavePaymentForPurchaseLaterRequest();

        unit.populate(setupTokenRequestData, purchaseLaterRequest);

        CardData card = purchaseLaterRequest.getPaymentSource().getCard();

        assertEquals(VAULT_CUSTOMER_ID, purchaseLaterRequest.getCustomer().getId());

        assertEquals(NAME_ON_CARD, card.getName());
        assertEquals(EXPIRY_YEAR + "-" + EXPIRY_MONTH, card.getExpiry());
        assertEquals(CARD_NUMBER, card.getNumber());
        assertEquals(CVV, card.getCvv());

        assertEquals(ADDRESS_LINE1, card.getBillingAddress().getAddressLine1());
        assertEquals(ADDRESS_LINE2, card.getBillingAddress().getAddressLine2());
        assertEquals(REGION_ISO_CODE, card.getBillingAddress().getAdminArea1());
        assertEquals(TOWN, card.getBillingAddress().getAdminArea2());
        assertEquals(POSTAL_CODE, card.getBillingAddress().getPostalCode());
        assertEquals(COUNTRY_ISO_CODE, card.getBillingAddress().getCountryCode());

        assertEquals(RETURN_URL, card.getExperienceContext().getReturnUrl());
        assertEquals(CANCEL_URL, card.getExperienceContext().getCancelUrl());
    }

    @Test
    public void shouldPopulateDataWhenNoVaultCustomerIdAndNoVerification() {
        PayPalSavePaymentForPurchaseLaterRequest purchaseLaterRequest = new PayPalSavePaymentForPurchaseLaterRequest();

        when(payPalConfigurationService.getPayPalVerifyCCOnMyAccount()).thenReturn(ThreeDsVerificationOnMyAccountType.NO_VERIFICATION.getCode());
        when(customerModel.getVaultCustomerId()).thenReturn(null);

        unit.populate(setupTokenRequestData, purchaseLaterRequest);

        CardData card = purchaseLaterRequest.getPaymentSource().getCard();

        assertNull(purchaseLaterRequest.getCustomer().getId());

        assertEquals(NAME_ON_CARD, card.getName());
        assertEquals(EXPIRY_YEAR + "-" + EXPIRY_MONTH, card.getExpiry());
        assertEquals(CARD_NUMBER, card.getNumber());
        assertEquals(CVV, card.getCvv());

        assertEquals(ADDRESS_LINE1, card.getBillingAddress().getAddressLine1());
        assertEquals(ADDRESS_LINE2, card.getBillingAddress().getAddressLine2());
        assertEquals(REGION_ISO_CODE, card.getBillingAddress().getAdminArea1());
        assertEquals(TOWN, card.getBillingAddress().getAdminArea2());
        assertEquals(POSTAL_CODE, card.getBillingAddress().getPostalCode());
        assertEquals(COUNTRY_ISO_CODE, card.getBillingAddress().getCountryCode());

        assertNull(card.getExperienceContext());
    }

    @Test
    public void shouldReturnTrueWhenPaymentProviderIsHostedFields() {
        boolean result = unit.test(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        assertTrue(result);
    }

    private SetupTokenRequestData getSourceData() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentMethodCode(PAYMENT_METHOD_CODE);

        CreditCardAddressData creditCardAddressData = new CreditCardAddressData();
        creditCardAddressData.setNameOnCard(NAME_ON_CARD);
        creditCardAddressData.setExpiryYear(EXPIRY_YEAR);
        creditCardAddressData.setExpiryMonth(EXPIRY_MONTH);
        creditCardAddressData.setCardNumber(CARD_NUMBER);
        creditCardAddressData.setCvv(CVV);

        AddressData billingAddress = getBillingAddress();

        creditCardAddressData.setBillingAddress(billingAddress);
        setupTokenRequestData.setCreditCardData(creditCardAddressData);
        return setupTokenRequestData;
    }

    private static AddressData getBillingAddress() {
        AddressData billingAddress = new AddressData();
        billingAddress.setLine1(ADDRESS_LINE1);
        billingAddress.setLine2(ADDRESS_LINE2);
        billingAddress.setTown(TOWN);
        billingAddress.setPostalCode(POSTAL_CODE);

        RegionData regionData = new RegionData();
        regionData.setIsocode(REGION_ISO_CODE);
        billingAddress.setRegion(regionData);

        CountryData countryData = new CountryData();
        countryData.setIsocode(COUNTRY_ISO_CODE);
        billingAddress.setCountry(countryData);

        return billingAddress;
    }

}