package com.paypal.hybris.facade.populator;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.util.ExpiredDateValidatorUtil;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.PK;
import de.hybris.platform.core.model.ItemModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import org.apache.logging.log4j.util.Strings;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class PayPalCreditCardPaymentInfoPopulatorTest {

    private static final PK PK_VALUE = PK.BIG_PK;
    private static final String NUMBER_VALUE = "number";
    private static final String CC_OWNER_VALUE = "ccOwner";
    private static final String VALID_TO_MONTH_VALUE = "10";
    private static final String VALID_TO_YEAR_VALUE = "2050";
    private static final String VALID_FROM_MONTH_VALUE = "11";
    private static final String VALID_FROM_YEAR_VALUE = "2030";
    private static final String SUBSCRIPTION_ID_VALUE = "subscriptionId";
    private static final String PAYER_EMAIL_VALUE = "email";
    private static final String PAYER_ID_VALUE = "payerId";
    private static final String PAYPAL_ORDER_ID_VALUE = "orderId";
    private static final Long MONTH_TO_EXPIRATION = 242L;
    private static final Long MONTH_TO_EXPIRATION_EXPECTED = 243L;
    private static final String REGION = "test region";
    private static final String TOWN = "town";
    private static final String LINE_2 = "line 2";

    @Mock
    private ExpiredDateValidatorUtil expiredDateValidatorUtil;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private CreditCardPaymentInfoModel creditCardPaymentInfoModel;
    @Mock
    private ItemModel itemModel;

    private CCPaymentInfoData paymentInfoData;
    private final AddressData addressData = new AddressData();

    @InjectMocks
    private PayPalCreditCardPaymentInfoPopulator unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        paymentInfoData = new CCPaymentInfoData();

        when(paymentInfoModel.getPk()).thenReturn(PK_VALUE);
        when(paymentInfoModel.getNumber()).thenReturn(NUMBER_VALUE);
        when(paymentInfoModel.getType()).thenReturn(null);
        when(paymentInfoModel.getCcOwner()).thenReturn(CC_OWNER_VALUE);
        when(paymentInfoModel.getValidToMonth()).thenReturn(VALID_TO_MONTH_VALUE);
        when(paymentInfoModel.getValidToYear()).thenReturn(VALID_TO_YEAR_VALUE);
        when(paymentInfoModel.getValidFromMonth()).thenReturn(VALID_FROM_MONTH_VALUE);
        when(paymentInfoModel.getValidFromYear()).thenReturn(VALID_FROM_YEAR_VALUE);
        when(paymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID_VALUE);
        when(paymentInfoModel.getBillingAddress()).thenReturn(null);
        when(paymentInfoModel.getIssueNumber()).thenReturn(null);

        when(paymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL_VALUE);
        when(paymentInfoModel.getPayerId()).thenReturn(PAYER_ID_VALUE);
        when(paymentInfoModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID_VALUE);
        when(paymentInfoModel.getSubscriptionId()).thenReturn(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);
        when(paymentInfoModel.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRE_SOON);
        when(paymentInfoModel.getValidToMonth()).thenReturn(VALID_TO_MONTH_VALUE);
        when(paymentInfoModel.getValidToYear()).thenReturn(VALID_TO_YEAR_VALUE);
        when(expiredDateValidatorUtil.getMonthsToExpiration(VALID_TO_MONTH_VALUE, VALID_TO_YEAR_VALUE))
                .thenReturn(MONTH_TO_EXPIRATION);
        when(paymentInfoModel.getOriginal()).thenReturn(itemModel);
        when(itemModel.getPk()).thenReturn(PK_VALUE);
        when(paymentInfoModel.isShouldBeSaved()).thenReturn(true);
        paymentInfoData.setBillingAddress(addressData);
    }

    @Test
    public void shouldPopulatePayPalCreditCardPaymentInfo() {
        when(paymentInfoModel.getPaymentProvider())
                .thenReturn(PaymentProvider.PAYPAL)
                .thenReturn(PaymentProvider.PAYPAL);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(PAYER_EMAIL_VALUE, paymentInfoData.getPayerEmail());
        assertEquals(PAYER_ID_VALUE, paymentInfoData.getPayerId());
        assertEquals(PAYPAL_ORDER_ID_VALUE, paymentInfoData.getPayPalOrderId());
        assertEquals(PaymentProvider.PAYPAL.toString(), paymentInfoData.getPaymentProvider().toString());
        assertEquals(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER, paymentInfoData.getSubscriptionId());
        assertEquals(ExpirationStatus.EXPIRE_SOON.toString(), paymentInfoData.getExpirationStatus());
        assertEquals(MONTH_TO_EXPIRATION_EXPECTED.longValue(), paymentInfoData.getRemainderMonthToExpire());
        assertEquals(PK_VALUE.toString(), paymentInfoData.getOriginalId());
        assertFalse(paymentInfoData.getIsEnable());
        assertTrue(paymentInfoData.isShouldBeSaved());
        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getFirstName());
        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getLastName());
    }

    @Test
    public void shouldPopulateForPayPalBillingAddressFirstNameAndSurnameAsEmpty() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getFirstName());
        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getLastName());
    }

    @Test
    public void shouldPopulateForPayPalBillingAddressTownAndLine2AndRegionAsEmpty() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getTown());
        assertEquals(Strings.EMPTY, paymentInfoData.getBillingAddress().getLine2());
        assertNull(paymentInfoData.getBillingAddress().getRegion());
    }

    @Test
    public void shouldNotPopulateForCCBillingAddressTownAndLine2AndRegionAsEmpty() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        RegionData region = new RegionData();
        region.setName(REGION);
        addressData.setTown(TOWN);
        addressData.setLine2(LINE_2);
        addressData.setRegion(region);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(TOWN, paymentInfoData.getBillingAddress().getTown());
        assertEquals(LINE_2, paymentInfoData.getBillingAddress().getLine2());
        assertEquals(region, paymentInfoData.getBillingAddress().getRegion());
    }

    @Test
    public void shouldNotPopulateForCCBillingAddressFirstNameAndSurnameAsEmpty() {
        when(paymentInfoModel.getPaymentProvider())
                .thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertNull(paymentInfoData.getBillingAddress().getFirstName());
        assertNull(paymentInfoData.getBillingAddress().getLastName());
    }

    @Test
    public void shouldPopulatePayPalCreditCardPaymentInfoWhenPMIsHostedFields() {
        when(paymentInfoModel.getPaymentProvider())
                .thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(PaymentProvider.PAYPAL_HOSTED_FIELDS.toString(), paymentInfoData.getPaymentProvider().toString());
        assertFalse(paymentInfoData.getIsEnable());

        verify(defaultPayPalConfigurationService).isPayPalHostedFieldsVaultEnabled();
    }

    @Test
    public void shouldPopulatePayPalCreditCardPaymentInfoWhenPMIsApplePay() {
        when(paymentInfoModel.getPaymentProvider())
                .thenReturn(PaymentProvider.APPLEPAY);

        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(PaymentProvider.APPLEPAY.toString(), paymentInfoData.getPaymentProvider().toString());
        assertTrue(paymentInfoData.getIsEnable());
    }

    @Test
    public void shouldPopulatePayPalCreditCardPaymentInfoWhenPMIsNull() {
        unit.populate(paymentInfoModel, paymentInfoData);

        assertNull(paymentInfoData.getPaymentProvider());
        assertFalse(paymentInfoData.getIsEnable());
    }

    @Test
    public void shouldNotPopulatePayPalCreditCardPaymentInfoFieldsIfOtherType() {
        when(creditCardPaymentInfoModel.getPk()).thenReturn(PK_VALUE);
        when(creditCardPaymentInfoModel.getNumber()).thenReturn(NUMBER_VALUE);
        when(creditCardPaymentInfoModel.getType()).thenReturn(null);
        when(creditCardPaymentInfoModel.getCcOwner()).thenReturn(CC_OWNER_VALUE);
        when(creditCardPaymentInfoModel.getValidToMonth()).thenReturn(VALID_TO_MONTH_VALUE);
        when(creditCardPaymentInfoModel.getValidToYear()).thenReturn(VALID_TO_YEAR_VALUE);
        when(creditCardPaymentInfoModel.getValidFromMonth()).thenReturn(VALID_FROM_MONTH_VALUE);
        when(creditCardPaymentInfoModel.getValidFromYear()).thenReturn(VALID_FROM_YEAR_VALUE);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID_VALUE);

        unit.populate(creditCardPaymentInfoModel, paymentInfoData);

        assertNull(paymentInfoData.getPayerEmail());
        assertNull(paymentInfoData.getPayerId());
        assertNull(paymentInfoData.getPayPalOrderId());
        assertNull(paymentInfoData.getExpirationStatus());
        assertNull(paymentInfoData.getOriginalId());
        assertNull(paymentInfoData.getIsEnable());
    }

    @Test
    public void shouldMapSubscriptionIdWhenPresent() {
        when(paymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID_VALUE);
        unit.populate(paymentInfoModel, paymentInfoData);

        assertEquals(SUBSCRIPTION_ID_VALUE, paymentInfoData.getSubscriptionId());
    }

    @Test
    public void shouldNotMapRemainderMonthToExpireWhenStatusIsNotExpireSoon() {
        when(paymentInfoModel.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRED);
        unit.populate(paymentInfoModel, paymentInfoData);

        verify(expiredDateValidatorUtil, times(0))
                .getMonthsToExpiration(VALID_TO_MONTH_VALUE, VALID_TO_YEAR_VALUE);
    }
}
