package com.paypal.hybris.facade.populator;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.when;

import java.math.BigDecimal;
import java.util.Collections;

import com.paypal.hybris.facade.service.BreakdownAmountCalculationService;
import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.CartEntryModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.product.UnitModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.util.DiscountValue;
import de.hybris.platform.util.TaxValue;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class PayPaCartEntryL2L3DataPopulatorTest {

    private static final String AMOUNT = "100.0";
    private static final double AMOUNT_DOUBLE = 100D;
    private static final String USD = "USD";
    private static final String TAX_CODE = "taxCode";
    private static final String DISCOUNT_CODE = "discountCode";
    private static final String PCS = "pcs";

    @Mock
    private BreakdownCalculationService breakdownCalculationService;

    @Mock
    private CartEntryModel cartEntryModel;

    @Mock
    private CartModel cartModel;

    @Mock
    private UnitModel unitModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private BreakdownAmountCalculationService breakdownAmountCalculationService;

    @InjectMocks
    private PayPaCartEntryL2L3DataPopulator unit;

    private OrderEntryData target;
    private TaxValue taxValue;
    private DiscountValue discountValue;

    @Before
    public void setUp() {
        MockitoAnnotations.openMocks(this);

        target = new OrderEntryData();

        taxValue = new TaxValue(TAX_CODE, AMOUNT_DOUBLE, true, AMOUNT_DOUBLE, USD);
        discountValue = new DiscountValue(DISCOUNT_CODE, AMOUNT_DOUBLE, true, AMOUNT_DOUBLE, USD);

    }

    @Test
    public void shouldPopulateTargetWithCorrectValues() throws ConversionException {
        when(cartEntryModel.getOrder()).thenReturn(cartModel);
        when(cartModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);

        when(breakdownAmountCalculationService.calculateItemsAmount(cartModel)).thenReturn(new BigDecimal(AMOUNT));

        when(cartEntryModel.getTaxValues()).thenReturn(Collections.singletonList(taxValue));
        when(cartEntryModel.getTotalPrice()).thenReturn(AMOUNT_DOUBLE);
        when(cartEntryModel.getDiscountValues()).thenReturn(Collections.singletonList(discountValue));
        when(cartEntryModel.getUnit()).thenReturn(unitModel);
        when(unitModel.getUnitType()).thenReturn(PCS);

        unit.populate(cartEntryModel, target);

        assertEquals(AMOUNT, target.getTaxAmount());
        assertEquals(AMOUNT, target.getDiscountAmount());
        assertEquals(AMOUNT, target.getTotalPrice().getValue().toString());
        assertEquals(USD, target.getTotalPrice().getCurrencyIso());
        assertEquals(PCS, target.getUnitOfMeasure());
    }
}