package com.paypal.hybris.facade.populator;


import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.AuthenticationResult;
import com.paypal.hybris.data.AuthenticationResultData;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.NameData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayerData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.PurchaseUnitData;
import com.paypal.hybris.data.ShippingDetailData;
import com.paypal.hybris.data.ThreeDSecure;
import de.hybris.bootstrap.annotations.UnitTest;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.ArrayList;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

@UnitTest
public class OrderResponseToOrderDetailsPopulatorTest {

	private static final String ORDER_ID = "orderId";
	private static final String PAYER_ID = "payerId";
	private static final String AMOUNT = "123.45";
	private static final String FULL_NAME = "Full Name";
	private static final String ADDRESS_LINE_1 = "address line 1";
	private static final String ADDRESS_LINE_2 = "address line 2";
	private static final String CITY = "CITY";
	private static final String REGION = "REGION";
	private static final String COUNTRY_CODE = "COUNTRY_CODE";
	private static final String POSTAL_CODE = "postal code";
	private static final String EMAIL = "someEmail@paypal.com";
	private static final String FIRST_NAME_FULL_NAME = "Full";
	private static final String LAST_NAME_FULL_NAME = "Name";
	private static final String ENROLLMENT = "Enrollment";
	private static final String AUTH_STATUS = "AuthStatus";
	private static final String LIABILITY_SHIFT = "Liability Shift";

	@Spy
	private OrderResponseToOrderDetailsPopulator unit;
	private PayPalOrderResponseData responseData;

	@Before
	public void setUp() {
		MockitoAnnotations.openMocks(this);
		responseData = new PayPalOrderResponseData();

		PayerData payerData = new PayerData();
		AddressPortableData address = new AddressPortableData();
		NameData buyerName = new NameData();
		PurchaseUnitData purchaseUnitData = new PurchaseUnitData();
		AmountWithBreakdownData amount = new AmountWithBreakdownData();
		ShippingDetailData shipping = new ShippingDetailData();

		shipping.setName(buyerName);
		shipping.setAddress(address);

		amount.setValue(AMOUNT);

		purchaseUnitData.setAmount(amount);
		purchaseUnitData.setShipping(shipping);

		buyerName.setGivenName(FIRST_NAME_FULL_NAME);
		buyerName.setSurname(LAST_NAME_FULL_NAME);
		buyerName.setFullName(FULL_NAME);

		address.setAdminArea1(REGION);
		address.setAdminArea2(CITY);
		address.setCountryCode(COUNTRY_CODE);
		address.setPostalCode(POSTAL_CODE);
		address.setAddressLine1(ADDRESS_LINE_1);
		address.setAddressLine2(ADDRESS_LINE_2);

		payerData.setPayerId(PAYER_ID);
		payerData.setName(buyerName);
		payerData.setEmail(EMAIL);

		responseData.setId(ORDER_ID);
		responseData.setPayer(payerData);
		responseData.setPurchaseUnits(new ArrayList<>(List.of(purchaseUnitData)));
	}


	@Test
	public void shouldCreateShippingAddressWithAllFieldsWhenResponseDataContainsNameAndEmailAndAddress() {
		PayPalAddressDetailsData address = unit.convertPayPalShippingAddress(responseData);

		assertEquals(FIRST_NAME_FULL_NAME, address.getFirstName());
		assertEquals(LAST_NAME_FULL_NAME, address.getLastName());
		assertEquals(EMAIL, address.getEmail());

		assertEquals(CITY, address.getCity());
		assertEquals(REGION, address.getRegion());
		assertEquals(COUNTRY_CODE, address.getCountryCode());
		assertEquals(POSTAL_CODE, address.getPostalCode());
		assertEquals(ADDRESS_LINE_1, address.getLine1());
		assertEquals(ADDRESS_LINE_2, address.getLine2());
	}

	@Test
	public void shouldCreateShippingAddressWithOnlyFirstNameAndEmptyEmailWhenPayerMissingAndNameConsistsOfOneWord() {
		responseData.setPayer(null);
		NameData fullName = new NameData();
		fullName.setFullName(FIRST_NAME_FULL_NAME);
		responseData.getPurchaseUnits().get(0).getShipping().setName(fullName);

		PayPalAddressDetailsData address = unit.convertPayPalShippingAddress(responseData);

		assertEquals(FIRST_NAME_FULL_NAME, address.getFirstName());
		assertEquals(StringUtils.EMPTY, address.getLastName());
		assertEquals(StringUtils.EMPTY, address.getEmail());
	}

	@Test
	public void shouldReturnNullShippingAddressWhenPurchaseUnitDoesNotHaveShipping() {
		responseData.getPurchaseUnits().get(0).setShipping(null);

		assertNull(unit.convertPayPalShippingAddress(responseData));
	}

	@Test
	public void shouldReturnNullBillingAddressWhenPayerDataIsNull() {
		assertNull(unit.convertPayPalBillingAddress(null));
	}

	@Test
	public void shouldReturnNullAuthenticationDataWhenPaymentSourceIsNull() {
		responseData.setPaymentSource(null);
		assertNull(unit.convertAuthenticationData(responseData));
	}

	@Test
	public void shouldReturnNullAuthenticationDataWhenCardDataIsNull() {
		PaymentSourceData paymentSourceData = new PaymentSourceData();
		paymentSourceData.setCard(null);
		responseData.setPaymentSource(paymentSourceData);
		assertNull(unit.convertAuthenticationData(responseData));
	}
	@Test
	public void shouldReturnNullAuthenticationDataWhenAuthenticationResultIsNull() {
		PaymentSourceData paymentSourceData = new PaymentSourceData();
		CardData cardData = new CardData();
		cardData.setAuthenticationResult(null);
		paymentSourceData.setCard(cardData);
		responseData.setPaymentSource(paymentSourceData);
		assertNull(unit.convertAuthenticationData(responseData));
	}

	@Test
	public void shouldPopulateAuthenticationResultData() {
		createAuthResult();

		AuthenticationResultData authenticationResultData = unit.convertAuthenticationData(responseData);

		assertEquals(LIABILITY_SHIFT, authenticationResultData.getLiabilityShift());
		assertEquals(AUTH_STATUS, authenticationResultData.getAuthenticationStatus());
		assertEquals(ENROLLMENT, authenticationResultData.getEnrollmentStatus());
	}

	private void createAuthResult() {
		PaymentSourceData paymentSourceData = new PaymentSourceData();
		CardData cardData = new CardData();
		AuthenticationResult authenticationResult = new AuthenticationResult();
		cardData.setAuthenticationResult(authenticationResult);
		paymentSourceData.setCard(cardData);
		ThreeDSecure threeDSecure = new ThreeDSecure();
		threeDSecure.setEnrollmentStatus(ENROLLMENT);
		threeDSecure.setAuthenticationStatus(AUTH_STATUS);
		authenticationResult.setThreeDSecure(threeDSecure);
		authenticationResult.setLiabilityShift(LIABILITY_SHIFT);
		responseData.setPaymentSource(paymentSourceData);
	}
}