package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.L3ItemData;
import com.paypal.hybris.data.Level2CardData;
import com.paypal.hybris.data.Level3CardData;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.SupplementaryData;
import com.paypal.hybris.facade.service.BreakdownAmountCalculationService;
import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.externaltax.DeliveryFromAddressStrategy;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.math.BigDecimal;
import java.util.Collections;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.when;

@UnitTest
public class OrderRequestDataPopulatorTest {

	private static final String ORDER_ID = "ORDER_ID";
	private static final String EXPECTED_AMOUNT = "22.99";
	private static final String PRICE_VALUE_NON_DECIMAL = "22";
	private static final String PRICE_CURRENCY_NON_DECIMAL = "NON_DECIMAL";
	private static final String USD = "USD";
	private static final String EXPECTED_DISCOUNT = "2.00";
	private static final String EXPECTED_TAX = "1.00";
	private static final String EXPECTED_SHIPPING = "5.00";
	private static final String EXPECTED_ITEMS_AMOUNT = "19.99";
	private static final String POSTAL_CODE = "1221";

	@Mock
	private Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter;
	@Mock
	private Converter<AddressData, AddressPortableData> payPalAddressPortableDataConverter;
	@Mock
	private Converter<AbstractOrderEntryModel, L3ItemData> l3ItemDataDataConverter;
	@Mock
	private Converter<AddressModel, AddressData> addressConverter;
	@Mock
	private Converter<AbstractOrderEntryModel, OrderEntryData> orderEntryConverter;
	@Mock
	private BreakdownCalculationService breakdownCalculationService;
	@Mock
	private PayPalConfigurationService payPalConfigurationService;
	@Mock
	private DeliveryFromAddressStrategy deliveryFromAddressStrategy;
	@Mock
	private BreakdownAmountCalculationService calculationStrategy;
	@Mock
	private AddressModel addressModel;
	@Mock
	private OrderModel orderModel;
	@Mock
	private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
	@Mock
	private CartModel source;
	@InjectMocks
	private OrderRequestDataPopulator unit;

	private PayPalOrderRequestData orderRequestData;
	private AddressPortableData addressPortableData;

	@Before
	public void setUp() {
		MockitoAnnotations.openMocks(this);

		addressPortableData = new AddressPortableData();

		orderRequestData = new PayPalOrderRequestData();
		AddressData addressData = new AddressData();
		L3ItemData l3ItemData = new L3ItemData();

		CurrencyModel currencyModel = new CurrencyModel();
		currencyModel.setIsocode(USD);
		when(source.getCurrency()).thenReturn(currencyModel);
		when(orderModel.getCurrency()).thenReturn(currencyModel);

		AbstractOrderEntryModel entryModel = new AbstractOrderEntryModel();
		entryModel.setQuantity(1L);
		entryModel.setBasePrice(19.99);
		when(source.getEntries()).thenReturn(Collections.singletonList(entryModel));
		when(orderModel.getEntries()).thenReturn(Collections.singletonList(entryModel));

		when(source.getPayPalOrderId()).thenReturn(ORDER_ID);

		when(source.getDeliveryCost()).thenReturn(5.0);
		when(source.getTotalDiscounts()).thenReturn(2.0);

		when(orderModel.getDeliveryCost()).thenReturn(5.0);
		when(orderModel.getTotalDiscounts()).thenReturn(2.0);

		when(calculationStrategy.calculateTotalAmount(source)).thenReturn(BigDecimal.valueOf(22.99));
		when(calculationStrategy.calculateItemsAmount(source)).thenReturn(BigDecimal.valueOf(19.99));
		when(calculationStrategy.calculateTaxAmount(source)).thenReturn(BigDecimal.valueOf(1.0));
		when(calculationStrategy.calculateTotalAmount(orderModel)).thenReturn(BigDecimal.valueOf(22.99));
		when(calculationStrategy.calculateItemsAmount(orderModel)).thenReturn(BigDecimal.valueOf(19.99));
		when(calculationStrategy.calculateTaxAmount(orderModel)).thenReturn(BigDecimal.valueOf(1.0));

		when(orderItemDataConverter.convert(entryModel)).thenReturn(new OrderItemData());
		when(orderEntryConverter.convert(entryModel)).thenReturn(new OrderEntryData());
		when(payPalConfigurationService.isL2L3DataEnabled()).thenReturn(Boolean.TRUE);
		when(deliveryFromAddressStrategy.getDeliveryFromAddressForOrder(source)).thenReturn(addressModel);
		when(deliveryFromAddressStrategy.getDeliveryFromAddressForOrder(orderModel)).thenReturn(addressModel);
		when(addressModel.getPostalcode()).thenReturn(POSTAL_CODE);
		when(source.getDeliveryAddress()).thenReturn(addressModel);
		when(orderModel.getDeliveryAddress()).thenReturn(addressModel);
		when(addressConverter.convert(addressModel)).thenReturn(addressData);
		when(payPalAddressPortableDataConverter.convert(addressData)).thenReturn(addressPortableData);
		when(l3ItemDataDataConverter.convert(entryModel)).thenReturn(l3ItemData);
		when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(PRICE_CURRENCY_NON_DECIMAL);

		when(breakdownCalculationService.calculateTaxAmount(orderModel)).thenReturn(BigDecimal.valueOf(1.00));
		when(breakdownCalculationService.calculateItemsAmount(orderModel)).thenReturn(BigDecimal.valueOf(19.99));
		when(breakdownCalculationService.calculateTotalDiscount(orderModel)).thenReturn(BigDecimal.valueOf(2.00));
		when(breakdownCalculationService.calculateTotalAmount(orderModel)).thenReturn(BigDecimal.valueOf(22));

		when(breakdownCalculationService.calculateTaxAmount(source)).thenReturn(BigDecimal.valueOf(1.00));
		when(breakdownCalculationService.calculateItemsAmount(source)).thenReturn(BigDecimal.valueOf(19.99));
		when(breakdownCalculationService.calculateTotalDiscount(source)).thenReturn(BigDecimal.valueOf(2.00));
		when(breakdownCalculationService.calculateTotalAmount(source)).thenReturn(BigDecimal.valueOf(22));
	}

	@Test
	public void shouldPopulateCartModel() {
		when(breakdownCalculationService.calculateTotalAmount(orderModel)).thenReturn(BigDecimal.valueOf(22.99));
		unit.populate(orderModel, orderRequestData);

		assertEquals(EXPECTED_AMOUNT, orderRequestData.getAmount());
		assertEquals(USD, orderRequestData.getCurrency());
		assertEquals(EXPECTED_ITEMS_AMOUNT, orderRequestData.getBreakdownAmountData().getBreakdown().getItemTotal().getValue());
		assertEquals(EXPECTED_SHIPPING, orderRequestData.getBreakdownAmountData().getBreakdown().getShipping().getValue());
		assertEquals(EXPECTED_TAX, orderRequestData.getBreakdownAmountData().getBreakdown().getTax().getValue());
		assertEquals(EXPECTED_DISCOUNT, orderRequestData.getBreakdownAmountData().getBreakdown().getDiscount().getValue());
		assertEquals(1, orderRequestData.getOrderItems().size());
	}

	@Test
	public void shouldPopulateOrderIdWhenOriginIsCartModel() {
		unit.populate(source, orderRequestData);

		assertEquals(ORDER_ID, orderRequestData.getOrderId());
	}

	@Test
	public void shouldCorrectlyGetAddressDataFromOrderModel() {
		AddressPortableData result = unit.getAddressData(source);

		assertEquals(addressPortableData, result);
	}

	@Test
	public void shouldReturnNullWhenGetDeliveryAddressIsNull() {
		when(source.getDeliveryAddress()).thenReturn(null);

		AddressPortableData result = unit.getAddressData(source);

		assertNull(result);
	}

	@Test
	public void shouldConvertPriceToBigIntegerWhenPriceContainsNonDecimalCurrency() {
		CurrencyModel currencyModel = new CurrencyModel();
		currencyModel.setIsocode(PRICE_CURRENCY_NON_DECIMAL);
		when(orderModel.getCurrency()).thenReturn(currencyModel);
		PayPalOrderRequestData payPalOrderRequestData = new PayPalOrderRequestData();

		unit.populate(orderModel, payPalOrderRequestData);

		assertEquals(PRICE_VALUE_NON_DECIMAL, payPalOrderRequestData.getAmount());
	}

	@Test
	public void shouldReturnNullSupplementaryDataWhenL2L3IsDisabled() {
		when(payPalConfigurationService.isL2L3DataEnabled()).thenReturn(false);

		assertNull(unit.createSupplementaryData(orderModel));
	}

	@Test
	public void shouldReturnNullAddressPortableDataWhenDeliveryAddressIsNull() {
		when(orderModel.getDeliveryAddress()).thenReturn(null);

		assertNull(unit.getAddressData(orderModel));
	}

	@Test
	public void shouldNotPopulateOrderIdWhenPopulatingOrderModel() {
		unit.populate(orderModel, orderRequestData);

		assertNull(orderRequestData.getOrderId());
	}

	@Test
	public void shouldCreateSupplementaryData() {
		SupplementaryData supplementaryData = unit.createSupplementaryData(orderModel);

		assertNotNull(supplementaryData);
		assertNotNull(supplementaryData.getCard());
		assertNotNull(supplementaryData.getCard().getLevel2());
		assertNotNull(supplementaryData.getCard().getLevel3());

		Level2CardData level2 = supplementaryData.getCard().getLevel2();
		Level3CardData level3 = supplementaryData.getCard().getLevel3();

		assertEquals(EXPECTED_TAX, level2.getTaxTotal().getValue());
		assertEquals(USD, level2.getTaxTotal().getCurrencyCode());

		assertEquals(EXPECTED_DISCOUNT, level3.getDiscountAmount().getValue());
		assertEquals(EXPECTED_SHIPPING, level3.getShippingAmount().getValue());
		assertEquals(1, level3.getLineItems().size());
	}

}
