package com.paypal.hybris.facade.populator;

import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorservices.urlresolver.SiteBaseUrlResolutionService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commerceservices.url.UrlResolver;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.product.ProductModel;
import de.hybris.platform.site.BaseSiteService;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

@UnitTest
public class OrderItemPopulatorTest {
	private static final String CODE = "CODE";
	private static final String NAME = "NAME";
	private static final String URL = "URL";
	private static final Long QUANTITY = 10L;
	private static final String DESCRIPTION = "DESCRIPTION";
	private static final String PRICE_VALUE_STRING = "19.99";
	private static final double PRICE_VALUE_DOUBLE = 19.99;
	private static final String USD = "USD";
	private static final String IMAGE_URL = "https://image.com/media.jpg";
	private static final String INVALID_IMAGE_URL = "https://image.com/media";
	private static final String LOCAL_IMAGE_URL = "https://imagelocal:9002/media.jpg";
	private static final String LOCAL_IMAGE_VALID_URL = "https://imagelocal9002/media.jpg";
	private static final String PAYPAL_PRODUCT_ITEM_TYPE = "PHYSICAL_GOODS";

	@Mock(answer = Answers.RETURNS_DEEP_STUBS)
	private AbstractOrderEntryModel source;
	@Mock
	private BaseSiteService baseSiteService;
	@Mock
	private SiteBaseUrlResolutionService siteBaseUrlResolutionService;
	@Mock
	private BaseSiteModel baseSiteModel;
	@Mock
	private UrlResolver<ProductModel> productModelUrlResolver;
	@Mock
	private MediaModel mediaModel;
	@InjectMocks
	private OrderItemPopulator unit;

	@Before
	public void setUp() {
		MockitoAnnotations.openMocks(this);

		when(source.getOrder().getCurrency().getIsocode()).thenReturn(USD);
		when(source.getBasePrice()).thenReturn(PRICE_VALUE_DOUBLE);
		when(source.getProduct().getName()).thenReturn(NAME);
		when(source.getQuantity()).thenReturn(QUANTITY);
		when(source.getProduct().getDescription()).thenReturn(DESCRIPTION);
		when(source.getProduct().getCode()).thenReturn(CODE);

		when(baseSiteService.getCurrentBaseSite()).thenReturn(baseSiteModel);
		when(productModelUrlResolver.resolve(source.getProduct())).thenReturn(URL);
		when(source.getProduct().getThumbnail()).thenReturn(mediaModel);
		when(mediaModel.getPayPalPrettyImgUrl()).thenReturn(URL);
		when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel, StringUtils.EMPTY, true, URL)).thenReturn(IMAGE_URL);
	}

	@Test
	public void shouldCorrectlyPopulateOrderItemDataFromEntryData() {
		OrderItemData expected = createOrderItemData();
		OrderItemData orderItemData = new OrderItemData();

		unit.populate(source, orderItemData);

		assertThat(orderItemData).isEqualToComparingFieldByFieldRecursively(expected);
	}

	@Test
	public void shouldPopulateOrderItemDataFromEntryDataWithoutOrder() {
		OrderItemData expected = createOrderItemData();
		OrderItemData orderItemData = new OrderItemData();
		expected.setImageUrl(null);

		when(source.getProduct().getThumbnail()).thenReturn(null);

		unit.populate(source, orderItemData);

		assertThat(orderItemData).isEqualToComparingFieldByFieldRecursively(expected);
	}

	@Test
	public void shouldNotPopulateImageUrlIfNotValid(){
		OrderItemData expected = createOrderItemData();
		OrderItemData orderItemData = new OrderItemData();
		expected.setImageUrl(null);

		when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel, StringUtils.EMPTY, true, URL)).thenReturn(INVALID_IMAGE_URL);

		unit.populate(source, orderItemData);

		assertThat(orderItemData).isEqualToComparingFieldByFieldRecursively(expected);
	}

	@Test
	public void shouldMakeValidAndPopulateImageUrlForLocalUrlWithColonSymbols(){
		OrderItemData expected = createOrderItemData();
		OrderItemData orderItemData = new OrderItemData();
		expected.setImageUrl(LOCAL_IMAGE_VALID_URL);

		when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel,  StringUtils.EMPTY,true, URL)).thenReturn(LOCAL_IMAGE_URL);

		unit.populate(source, orderItemData);

		assertThat(orderItemData).isEqualToComparingFieldByFieldRecursively(expected);
	}
	private OrderItemData createOrderItemData() {
		OrderItemData orderItemData = new OrderItemData();

		SimpleAmount amount = new SimpleAmount();
		amount.setValue(PRICE_VALUE_STRING);
		amount.setCurrencyCode(USD);

		orderItemData.setName(NAME);
		orderItemData.setQuantity(QUANTITY.intValue());
		orderItemData.setDescription(DESCRIPTION);
		orderItemData.setSku(CODE);
		orderItemData.setUrl(URL);
		orderItemData.setImageUrl(IMAGE_URL);
		orderItemData.setCategory(PAYPAL_PRODUCT_ITEM_TYPE);
		orderItemData.setAmount(amount);

		return orderItemData;
	}
}