package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.L3ItemData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.util.DiscountValue;
import de.hybris.platform.util.TaxValue;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.Collections;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

@UnitTest
public class L3ItemDataPopulatorTest {
    private static final String CODE = "CODE";
    private static final String NAME = "NAME";
    private static final int QUANTITY = 2;
    private static final long QUANTITY_LONG = 2L;
    private static final String AMOUNT = "324.00";
    private static final double AMOUNT_DOUBLE = 324D;
    private static final String DESCRIPTION = "DESCRIPTION";
    private static final String USD = "USD";
    private static final String UNIT_TYPE = "UnitType";

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private AbstractOrderEntryModel source;
    @InjectMocks
    private L3ItemDataPopulator unit;

    private TaxValue taxValue;
    private DiscountValue discountValue;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        taxValue = new TaxValue(CODE, AMOUNT_DOUBLE, true, AMOUNT_DOUBLE, USD);
        discountValue = new DiscountValue(CODE, AMOUNT_DOUBLE, true, USD);

        populateSource();
    }

    @Test
    public void shouldPopulateL3Item() {
        L3ItemData target = new L3ItemData();
        L3ItemData expectedTarget = getExpectedTarget();

        unit.populate(source, target);

        assertThat(target).isEqualToComparingFieldByFieldRecursively(expectedTarget);
    }

    private void populateSource() {
        when(source.getOrder().getCurrency().getIsocode()).thenReturn(USD);
        when(source.getTaxValues()).thenReturn(Collections.singletonList(taxValue));
        when(source.getDiscountValues()).thenReturn(Collections.singletonList(discountValue));
        when(source.getTotalPrice()).thenReturn(AMOUNT_DOUBLE);
        when(source.getBasePrice()).thenReturn(AMOUNT_DOUBLE);
        when(source.getProduct().getName()).thenReturn(NAME);
        when(source.getQuantity()).thenReturn(QUANTITY_LONG);
        when(source.getProduct().getDescription()).thenReturn(DESCRIPTION);
        when(source.getProduct().getCode()).thenReturn(CODE);
        when(source.getUnit().getUnitType()).thenReturn(UNIT_TYPE);
    }

    private L3ItemData getExpectedTarget() {
        L3ItemData l3ItemData = new L3ItemData();

        l3ItemData.setSku(CODE);
        l3ItemData.setDescription(DESCRIPTION);
        l3ItemData.setName(NAME);
        l3ItemData.setUnitOfMeasure(UNIT_TYPE);
        l3ItemData.setQuantity(QUANTITY);
        l3ItemData.setTotalAmount(getAmount());
        l3ItemData.setTax(getAmount());
        l3ItemData.setDiscountAmount(getAmount());
        l3ItemData.setAmount(getAmount());

        return l3ItemData;
    }

    private SimpleAmount getAmount() {
        return GenericBuilder.of(SimpleAmount::new)
                .with(SimpleAmount::setValue, AMOUNT)
                .with(SimpleAmount::setCurrencyCode, USD)
                .build();
    }
}