package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.exception.ApplePayInvalidAddressException;
import com.paypal.hybris.data.ApplePayAddressDetailsData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.ArrayList;
import java.util.List;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.when;

@UnitTest
public class ApplePayAddressDataPopulatorTest {

    private static final String CITY = "city";
    private static final String LINE_1 = "line1";
    private static final String POSTAL_CODE = "postalCode";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "secondName";
    private static final String EMAIL_ADDRESS = "email";
    private static final String PHONE_NUMBER = "phone";
    private static final String US = "US";
    private static final String ADMINISTRATIVE_AREA = "TX";
    private static final String REGION_ISO_CODE = "US-TX";
    private static final String ID = "ID";

    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private CountryModel countryModel;
    @Mock
    private RegionModel regionModel;
    private ApplePayAddressDetailsData source;
    private AddressData target;

    @InjectMocks
    private ApplePayAddressDataPopulator unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        source = new ApplePayAddressDetailsData();
        target = new AddressData();

        source.setId(ID);
        source.setLocality(CITY);
        source.setCountryCode(US);
        source.setAddressLines(new ArrayList<>(List.of(LINE_1)));
        source.setPostalCode(POSTAL_CODE);
        source.setGivenName(FIRST_NAME);
        source.setFamilyName(LAST_NAME);
        source.setEmailAddress(EMAIL_ADDRESS);
        source.setPhoneNumber(PHONE_NUMBER);
    }

    @Test
    public void shouldPopulateRegionData() {
        RegionData expected = new RegionData();
        expected.setIsocode(REGION_ISO_CODE);
        expected.setIsocodeShort(ADMINISTRATIVE_AREA);
        expected.setCountryIso(US);
        source.setAdministrativeArea(ADMINISTRATIVE_AREA);

        when(commonI18NService.getCountry(US)).thenReturn(countryModel);
        when(commonI18NService.getRegion(countryModel, REGION_ISO_CODE)).thenReturn(regionModel);

        unit.populate(source, target);

        assertThat(target.getRegion()).isEqualToComparingFieldByFieldRecursively(expected);
    }

    @Test
    public void shouldNotPopulateRegionWhenAdministrativeAreaIsEmpty() {
        unit.populate(source, target);

        assertNull(target.getRegion());
    }

    @Test(expected = ApplePayInvalidAddressException.class)
    public void shouldThrowApplePayInvalidAddressExceptionWhenAdministrativeAreaNotMatchingRegion() {
        source.setAdministrativeArea(ADMINISTRATIVE_AREA);

        when(commonI18NService.getCountry(US)).thenReturn(countryModel);
        when(commonI18NService.getRegion(countryModel, REGION_ISO_CODE)).thenThrow(UnknownIdentifierException.class);

        unit.populate(source, target);
    }

    @Test
    public void shouldPopulateApplePayAddressDataWithAddressLines() {
        unit.populate(source, target);

        assertEquals(ID, target.getId());
        assertEquals(CITY, target.getTown());
        assertEquals(US, target.getCountry().getIsocode());
        assertEquals(LINE_1, target.getLine1());
        assertEquals(POSTAL_CODE, target.getPostalCode());
        assertEquals(FIRST_NAME, target.getFirstName());
        assertEquals(LAST_NAME, target.getLastName());
        assertEquals(EMAIL_ADDRESS, target.getEmail());
        assertEquals(PHONE_NUMBER, target.getPhone());
    }

    @Test
    public void shouldPopulateApplePayAddressDataWithoutAddressLines() {
        source.setAddressLines(null);
        unit.populate(source, target);

        assertEquals(CITY, target.getTown());
        assertEquals(US, target.getCountry().getIsocode());
        assertEquals(StringUtils.EMPTY, target.getLine1());
        assertEquals(POSTAL_CODE, target.getPostalCode());
        assertEquals(FIRST_NAME, target.getFirstName());
        assertEquals(LAST_NAME, target.getLastName());
        assertEquals(EMAIL_ADDRESS, target.getEmail());
        assertEquals(PHONE_NUMBER, target.getPhone());
    }
}
