package com.paypal.hybris.facade.hooks;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.exception.PayPalCreditCardOrderAmountWasChangedException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentService;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

import java.math.BigDecimal;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.StrictStubs.class)
public class UpdateOrderAmountHookTest {

    private static final String PAYPAL_ORDER_ID = "orderId";
    private final BigDecimal sameAmount = new BigDecimal("12.0");
    private static final String NOT_SAME_AMOUNT = "12.0";
    private final BigDecimal notSameAmount = new BigDecimal("10.0");

    @Mock
    private PayPalConfigurationService payPalConfigurationService;
    @Mock
    private DefaultPayPalPaymentService payPalPaymentService;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private PayPalOrderDetailsData payPalOrderDetailsData;
    @Mock
    private CartModel cartModel;
    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private CommerceCheckoutParameter parameter;
    @Mock
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    @Mock
    private CurrencyModel currencyModel;
    private PayPalOrderRequestData orderRequestData;

    @InjectMocks
    private UpdateOrderAmountHook unit;

    @Before
    public void initSetUp() {
        orderRequestData = new PayPalOrderRequestData();
        when(payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable()).thenReturn(true);
        when(parameter.getCart()).thenReturn(cartModel);

        when(orderRequestDataConverter.convert(cartModel)).thenReturn(orderRequestData);

        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
    }

    @Test(expected = PayPalCreditCardOrderAmountWasChangedException.class)
    public void shouldThrowExceptionWhen3dsEnabledAndHostedFieldsPMAndTheAmountIsDifferent() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        when(cartModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID);
        when(payPalPaymentService.getOrderDetails(PAYPAL_ORDER_ID)).thenReturn(payPalOrderDetailsData);
        when(payPalOrderDetailsData.getAmount()).thenReturn(NOT_SAME_AMOUNT);

        when(parameter.getAuthorizationAmount()).thenReturn(notSameAmount);

        unit.beforeAuthorizePaymentAmount(parameter);
    }

    @Test
    public void shouldDoNothingWhen3dsEnabledAndHostedFieldsPMAndTheAmountIsSame() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        when(cartModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID);
        when(payPalPaymentService.getOrderDetails(PAYPAL_ORDER_ID)).thenReturn(payPalOrderDetailsData);
        when(payPalOrderDetailsData.getAmount()).thenReturn(String.valueOf(sameAmount));
        when(parameter.getAuthorizationAmount()).thenReturn(sameAmount);

        unit.beforeAuthorizePaymentAmount(parameter);

        verify(payPalPaymentService, never()).updateOrderAmountDetails(any());
    }

    @Test
    public void doNothingWhenCardIsSaved() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.beforeAuthorizePaymentAmount(parameter);

        verify(payPalPaymentService, never()).updateOrderAmountDetails(any());
    }

    @Test
    public void shouldUpdateAmountWhen3dsDisabledAndHostedFieldsPMAndTheAmountNotSame() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        when(payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable()).thenReturn(false);

        when(cartModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID);
        when(payPalPaymentService.getOrderDetails(PAYPAL_ORDER_ID)).thenReturn(payPalOrderDetailsData);
        when(payPalOrderDetailsData.getAmount()).thenReturn(String.valueOf(sameAmount));
        when(parameter.getAuthorizationAmount()).thenReturn(notSameAmount);

        unit.beforeAuthorizePaymentAmount(parameter);

        verify(payPalPaymentService).updateOrderAmountDetails(orderRequestData);
    }

    @Test
    public void shouldUpdateAmountWhenPayPalPaymentProvider() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        when(cartModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID);
        when(payPalPaymentService.getOrderDetails(PAYPAL_ORDER_ID)).thenReturn(payPalOrderDetailsData);
        when(payPalOrderDetailsData.getAmount()).thenReturn(String.valueOf(sameAmount));
        when(parameter.getAuthorizationAmount()).thenReturn(notSameAmount);

        unit.beforeAuthorizePaymentAmount(parameter);

        verify(payPalPaymentService).updateOrderAmountDetails(orderRequestData);
    }

    @Test
    public void shouldUpdateOrderAmountDetailsWhenCartDataIsNotNull() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);
        when(cartModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID);
        when(payPalPaymentService.getOrderDetails(PAYPAL_ORDER_ID)).thenReturn(payPalOrderDetailsData);
        when(payPalOrderDetailsData.getAmount()).thenReturn(String.valueOf(sameAmount));
        when(parameter.getAuthorizationAmount()).thenReturn(notSameAmount);

        unit.beforeAuthorizePaymentAmount(parameter);

        verify(payPalPaymentService).updateOrderAmountDetails(orderRequestData);
    }

}
