package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.List;

import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;
import static org.junit.Assert.assertEquals;

@UnitTest
public class DefaultPaypalUserFacadeTest {

    @Mock
    private DefaultPayPalConfigurationService configurationService;
    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;
    @Mock
    private CustomerAccountService customerAccountService;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel defaultPaymentInfoModel;

    private CCPaymentInfoData paymentInfoData;

    @Mock
    private Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter;

    @InjectMocks
    private DefaultPaypalUserFacade unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        paymentInfoData = new CCPaymentInfoData();
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfos(customerModel, true))
                .thenReturn(List.of(paymentInfoModel));
        when(customerModel.getDefaultPaymentInfo()).thenReturn(paymentInfoModel);
        when(creditCardPaymentInfoConverter.convert(paymentInfoModel)).thenReturn(paymentInfoData);
    }

    @Test
    public void shouldGetCCPaymentInfosWhenPaymentProviderIsPayPal() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);
        when(configurationService.isPayPalEnable()).thenReturn(true);

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true);

        assertEquals(1, result.size());
        assertEquals(paymentInfoData, result.get(0));
        verify(checkoutCustomerStrategy).getCurrentUserForCheckout();
        verify(customerAccountService).getCreditCardPaymentInfos(customerModel, true);
        verify(customerModel).getDefaultPaymentInfo();
    }

    @Test
    public void shouldGetCCPaymentInfosWhenPaymentProviderIsHostedFields() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        when(configurationService.isPayPalHostedFieldsEnabled()).thenReturn(true);

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true);

        assertEquals(1, result.size());
        assertEquals(paymentInfoData, result.get(0));
        verify(checkoutCustomerStrategy).getCurrentUserForCheckout();
        verify(customerAccountService).getCreditCardPaymentInfos(customerModel, true);
        verify(customerModel).getDefaultPaymentInfo();
        verify(configurationService).isPayPalHostedFieldsEnabled();
    }

    @Test
    public void shouldGetCCPaymentInfosWhenPaymentProviderIsApplePay() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.APPLEPAY);
        when(configurationService.isApplePayEnable()).thenReturn(true);

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true);

        assertEquals(1, result.size());
        assertEquals(paymentInfoData, result.get(0));
        verify(checkoutCustomerStrategy).getCurrentUserForCheckout();
        verify(customerAccountService).getCreditCardPaymentInfos(customerModel, true);
        verify(customerModel).getDefaultPaymentInfo();
        verify(configurationService).isApplePayEnable();
    }

    @Test
    public void shouldGetCCPaymentInfosWhenPaymentProviderIsNull() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(null);
        when(customerModel.getDefaultPaymentInfo()).thenReturn(defaultPaymentInfoModel);

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true);

        assertEquals(1, result.size());
        assertEquals(paymentInfoData, result.get(0));
        verify(checkoutCustomerStrategy).getCurrentUserForCheckout();
        verify(customerAccountService).getCreditCardPaymentInfos(customerModel, true);
        verify(customerModel).getDefaultPaymentInfo();
    }

}
