package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.SessionPaymentsEnabledType;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.model.PayPalGeneralConfigurationModel;
import com.paypal.hybris.core.model.PayPalPluginConfigurationModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Date;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertFalse;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.times;

@UnitTest
public class DefaultPayPalSessionPaymentMethodFacadeTest {

    private static final String ORDER_ID = "orderId";
    private static final String AMOUNT = "19.99";
    private static final String PAYPAL = "paypal";
    private static final String VENMO = "venmo";

    @Mock
    private CartModel cart;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private PayPalConfigurationService payPalConfigurationService;
    @Mock
    private PayPalCartService payPalCartService;
    @Mock
    private PayPalPaymentService payPalPaymentService;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private PayPalPluginConfigurationModel payPalPluginConfigurationModel;
    @Mock
    private PayPalGeneralConfigurationModel generalConfigurationModel;
    @Mock
    private Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter;

    private CCPaymentInfoData paymentInfoData;

    @Spy
    @InjectMocks
    private DefaultPayPalSessionPaymentMethodFacade unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(payPalCartService.getSessionCart()).thenReturn(cart);
        when(cart.getPaymentInfo()).thenReturn(paymentInfoModel);

        paymentInfoData = new CCPaymentInfoData();
        when(creditCardPaymentInfoConverter.convert(paymentInfoModel)).thenReturn(paymentInfoData);
    }

    @Test
    public void shouldGetSessionPaymentMethod() {
        doReturn(true).when(unit).isSessionPaymentFlowEnable();
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(true).when(unit).isSessionPaymentValid(cart);

        CCPaymentInfoData result = unit.getSessionPaymentMethod();

        assertEquals(paymentInfoData, result);
        verify(payPalCartService).getSessionCart();
        verify(cart, times(2)).getPaymentInfo();
        verify(unit).isSessionPaymentFlowEnable();
        verify(unit).isPaymentMethodSaved(cart);
        verify(unit).isSessionPaymentValid(cart);
        verify(creditCardPaymentInfoConverter).convert(paymentInfoModel);
    }

    @Test
    public void isPayPalSessionPaymentForExpressCheckoutExist() {
        doReturn(true).when(unit).isSessionPaymentFlowEnable();
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(true).when(unit).isSessionPaymentValid(cart);

        paymentInfoData.setCardType(PAYPAL);
        assertTrue(unit.isPayPalSessionPaymentForExpressCheckoutExist());

        paymentInfoData.setCardType(VENMO);
        assertFalse(unit.isPayPalSessionPaymentForExpressCheckoutExist());
    }

    @Test
    public void isSessionPaymentForExpressCheckoutNotExist() {
        doReturn(false).when(unit).isSessionPaymentFlowEnable();

        boolean result = unit.isPayPalSessionPaymentForExpressCheckoutExist();

        assertFalse(result);
    }

    @Test
    public void isSessionPaymentFlowEnableGuestAndLoggedInUser() {
        final SessionPaymentsEnabledType enabledType = SessionPaymentsEnabledType.GUEST_AND_LOGGED_IN_USERS;

        doReturn(enabledType).when(unit).getSessionPaymentsEnabledType();

        boolean result = unit.isSessionPaymentFlowEnable();

        assertTrue(result);
        verify(unit).getSessionPaymentsEnabledType();
    }

    @Test
    public void isSessionPaymentFlowEnableLoggedInUsersOnly() {
        final SessionPaymentsEnabledType enabledType = SessionPaymentsEnabledType.LOGGED_IN_USERS_ONLY;

        doReturn(enabledType).when(unit).getSessionPaymentsEnabledType();
        doReturn(CustomerType.REGISTERED).when(unit).getCustomerTypeFromCurrentSessionCart();

        boolean result = unit.isSessionPaymentFlowEnable();

        assertTrue(result);
        verify(unit).getSessionPaymentsEnabledType();
        verify(unit).getCustomerTypeFromCurrentSessionCart();
    }

    @Test
    public void isSessionPaymentFlowEnableLoggedInUsersOnlyAndCustomerTypeIsGuest() {
        final SessionPaymentsEnabledType enabledType = SessionPaymentsEnabledType.LOGGED_IN_USERS_ONLY;

        doReturn(enabledType).when(unit).getSessionPaymentsEnabledType();
        doReturn(CustomerType.GUEST).when(unit).getCustomerTypeFromCurrentSessionCart();

        boolean result = unit.isSessionPaymentFlowEnable();

        assertFalse(result);
        verify(unit).getSessionPaymentsEnabledType();
        verify(unit).getCustomerTypeFromCurrentSessionCart();
    }

    @Test
    public void shouldReturnFalseWhenWhenSessionPaymentsDisabled() {
        final SessionPaymentsEnabledType enabledType = SessionPaymentsEnabledType.DISABLED;

        doReturn(enabledType).when(unit).getSessionPaymentsEnabledType();

        boolean result = unit.isSessionPaymentFlowEnable();

        assertFalse(result);
    }

    @Test
    public void isPaymentMethodValid() {
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(true).when(unit).isSessionPaymentValid(cart);

        boolean result = unit.isPaymentMethodValid();

        assertTrue(result);
        verify(unit).isPaymentMethodSaved(cart);
        verify(unit).isSessionPaymentValid(cart);
    }

    @Test
    public void isPaymentMethodValidWhenSessionPaymentValidIsFalse() {
        doReturn(true).when(unit).isPaymentMethodSaved(cart);
        doReturn(false).when(unit).isSessionPaymentValid(cart);

        boolean result = unit.isPaymentMethodValid();

        assertTrue(result);
        verify(unit).isPaymentMethodSaved(cart);
    }

    @Test
    public void shouldReturnFalseWhenPaymentMethodSavedAndSessionPaymentValidAreFalse() {
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(false).when(unit).isSessionPaymentValid(cart);

        boolean result = unit.isPaymentMethodValid();

        assertFalse(result);
        verify(unit).isPaymentMethodSaved(cart);
    }

    @Test
    public void isPaymentMethodSaved() {
        when(paymentInfoModel.isSaved()).thenReturn(false);
        when(paymentInfoModel.isShouldBeSaved()).thenReturn(true);

        boolean result = unit.isPaymentMethodSaved(cart);

        assertTrue(result);
        verify(cart, times(2)).getPaymentInfo();
        verify(paymentInfoModel).isSaved();
        verify(paymentInfoModel).isShouldBeSaved();
    }

    @Test
    public void isPaymentMethodSavedWhenIsSavedIsTrue() {
        when(paymentInfoModel.isSaved()).thenReturn(true);
        when(paymentInfoModel.isShouldBeSaved()).thenReturn(false);

        boolean result = unit.isPaymentMethodSaved(cart);

        assertTrue(result);
        verify(cart, atLeastOnce()).getPaymentInfo();
        verify(paymentInfoModel).isSaved();
        verify(paymentInfoModel, never()).isShouldBeSaved();
    }

    @Test
    public void isPaymentMethodSavedWhenIsSavedAndShouldBeSavedAreFalse() {
        when(paymentInfoModel.isSaved()).thenReturn(false);
        when(paymentInfoModel.isShouldBeSaved()).thenReturn(false);

        boolean result = unit.isPaymentMethodSaved(cart);

        assertFalse(result);
        verify(cart, times(2)).getPaymentInfo();
        verify(paymentInfoModel).isSaved();
        verify(paymentInfoModel).isShouldBeSaved();
    }

    @Test
    public void isSessionPaymentValid() {
        doReturn(true).when(unit).isTimeCorrectForSessionPaymentMethod(paymentInfoModel);
        doReturn(true).when(unit).isPaymentMethodTypeEligible(paymentInfoModel);
        doReturn(true).when(unit).isAmountCorrectForSessionPaymentMethod(cart);

        boolean result = unit.isSessionPaymentValid(cart);

        assertTrue(result);
        verify(cart).getPaymentInfo();
        verify(unit).isTimeCorrectForSessionPaymentMethod(paymentInfoModel);
        verify(unit).isAmountCorrectForSessionPaymentMethod(cart);
    }

    @Test
    public void isSessionPaymentValidWhenTimeCorrectForSessionPaymentMethodIsFalse() {
        doReturn(false).when(unit).isTimeCorrectForSessionPaymentMethod(paymentInfoModel);
        doReturn(true).when(unit).isAmountCorrectForSessionPaymentMethod(cart);

        boolean result = unit.isSessionPaymentValid(cart);

        assertFalse(result);
        verify(cart).getPaymentInfo();
        verify(unit).isTimeCorrectForSessionPaymentMethod(paymentInfoModel);
    }

    @Test
    public void isSessionPaymentNotValidWhenPaymentTypeIsNotEligible() {
        doReturn(false).when(unit).isPaymentMethodTypeEligible(paymentInfoModel);

        boolean result = unit.isSessionPaymentValid(cart);

        assertFalse(result);
        verify(cart).getPaymentInfo();
        verify(unit, never()).isTimeCorrectForSessionPaymentMethod(paymentInfoModel);
        verify(unit, never()).isAmountCorrectForSessionPaymentMethod(cart);
    }

    @Test
    public void isTimeCorrectForSessionPaymentMethod() {
        when(paymentInfoModel.getModifiedtime()).thenReturn(new Date());

        boolean result = unit.isTimeCorrectForSessionPaymentMethod(paymentInfoModel);

        assertTrue(result);
        verify(paymentInfoModel).getModifiedtime();
    }

    @Test
    public void isAmountCorrectForSessionPaymentMethod() {
        final PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();
        orderDetailsData.setAmount(AMOUNT);

        when(payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable()).thenReturn(true);
        doReturn(true).when(unit).isHostedFieldsPaymentProvider(cart);
        when(cart.getPayPalOrderId()).thenReturn(ORDER_ID);
        when(payPalPaymentService.getOrderDetails(ORDER_ID)).thenReturn(orderDetailsData);
        when(cart.getTotalPrice()).thenReturn(19.99);

        boolean result = unit.isAmountCorrectForSessionPaymentMethod(cart);

        assertTrue(result);
    }

    @Test
    public void isAmountCorrectForSessionPaymentMethodWhenThreeDsVerificationDisabled() {
        when(payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable()).thenReturn(false);

        boolean result = unit.isAmountCorrectForSessionPaymentMethod(cart);

        assertTrue(result);
    }

    @Test
    public void isHostedFieldsPaymentProvider() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        boolean result = unit.isHostedFieldsPaymentProvider(cart);

        assertTrue(result);
        verify(cart).getPaymentInfo();
        verify(paymentInfoModel).getPaymentProvider();
    }

    @Test
    public void shouldReturnFalseWhenPaymentProviderIsPayPal() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        boolean result = unit.isHostedFieldsPaymentProvider(cart);

        assertFalse(result);
        verify(cart).getPaymentInfo();
        verify(paymentInfoModel).getPaymentProvider();
    }

    @Test
    public void shouldReturnGuestAndLoggedInUserType() {
        final SessionPaymentsEnabledType enabledType = SessionPaymentsEnabledType.GUEST_AND_LOGGED_IN_USERS;

        when(payPalConfigurationService.getPayPalPluginConfiguration()).thenReturn(payPalPluginConfigurationModel);
        when(payPalPluginConfigurationModel.getPayPalGeneralConfiguration()).thenReturn(generalConfigurationModel);
        when(generalConfigurationModel.getSessionPaymentsEnabled()).thenReturn(enabledType);

        SessionPaymentsEnabledType result = unit.getSessionPaymentsEnabledType();

        assertEquals(SessionPaymentsEnabledType.GUEST_AND_LOGGED_IN_USERS, result);
        verify(payPalConfigurationService).getPayPalPluginConfiguration();
        verify(payPalPluginConfigurationModel).getPayPalGeneralConfiguration();
        verify(generalConfigurationModel).getSessionPaymentsEnabled();
    }

    @Test
    public void shouldReturnGuestCustomerType() {
        when(cart.getUser()).thenReturn(customerModel);
        when(customerModel.getType()).thenReturn(CustomerType.GUEST);

        CustomerType result = unit.getCustomerTypeFromCurrentSessionCart();

        assertEquals(CustomerType.GUEST, result);
        verify(payPalCartService).getSessionCart();
        verify(cart).getUser();
        verify(customerModel).getType();
    }

    @Test
    public void isVenmoSessionPaymentForExpressCheckoutExist() {
        paymentInfoData.setCardType(VENMO);

        doReturn(true).when(unit).isSessionPaymentFlowEnable();
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(true).when(unit).isSessionPaymentValid(cart);

        boolean result = unit.isVenmoSessionPaymentForExpressCheckoutExist();

        assertTrue(result);
    }

    @Test
    public void isVenmoSessionPaymentForExpressCheckoutNotExist() {
        doReturn(false).when(unit).isSessionPaymentFlowEnable();

        boolean result = unit.isVenmoSessionPaymentForExpressCheckoutExist();

        assertFalse(result);
    }

    @Test
    public void isVenmoSessionPaymentForExpressCheckoutWhenCardTypeIsPayPal() {
        paymentInfoData.setCardType(PAYPAL);
        doReturn(true).when(unit).isSessionPaymentFlowEnable();
        doReturn(false).when(unit).isPaymentMethodSaved(cart);
        doReturn(true).when(unit).isSessionPaymentValid(cart);

        boolean result = unit.isVenmoSessionPaymentForExpressCheckoutExist();

        assertFalse(result);
    }

    @Test
    public void isPaymentMethodTypeEligibleWhenCardTypeIsPayPal() {
        when(paymentInfoModel.getType()).thenReturn(CreditCardType.PAYPAL);

        assertTrue(unit.isPaymentMethodTypeEligible(paymentInfoModel));
    }

    @Test
    public void isPaymentMethodTypeEligibleWhenCardTypeIsVenmo() {
        when(paymentInfoModel.getType()).thenReturn(CreditCardType.VENMO);

        assertTrue(unit.isPaymentMethodTypeEligible(paymentInfoModel));
    }

    @Test
    public void isPaymentMethodTypeNotEligibleWhenCardTypeIsApplePay() {
        when(paymentInfoModel.getType()).thenReturn(CreditCardType.APPLEPAY);

        assertFalse(unit.isPaymentMethodTypeEligible(paymentInfoModel));
    }

    @Test
    public void shouldReturnFalseWhenHasNoSessionCart() {
        when(payPalCartService.hasSessionCart()).thenReturn(false);

        assertFalse(unit.isPaymentMethodTypeEligible());
    }

    @Test
    public void shouldRetrieveSessionCartAndCheckPaymentEligibility() {
        when(payPalCartService.hasSessionCart()).thenReturn(true);

        boolean result = unit.isPaymentMethodTypeEligible();

        verify(unit).isPaymentMethodTypeEligible(paymentInfoModel);
        assertTrue(result);
    }

}
