package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPageType;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.data.PayPalSDKData;
import com.paypal.hybris.facade.facades.PayPalClientTokenFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalSDKFacadeTest {

    private static final String PAYPAL_SDK_JS = "https://www.paypal.com/sdk/js?";
    private static final String CLIENT_ID = "CLIENT_ID";
    private static final String ISO_CODE = "isoCode";
    private static final String ENVIRONMENT_TYPE_NO_SANDBOX = "no_sandbox";
    private static final String LPM_LOCALE = "US_us";
    private static final String COMMIT_PARAMETER_TRUE = "commit=true";
    private static final String COMMIT_PARAMETER_FALSE = "commit=false";
    private static final String BUYER_COUNTRY = "US";
    private static final String VENMO_FUNDING = "venmo";
    private static final String SANDBOX = "sandbox";
    private static final String CLIENT_ID_PARAMETER = "client-id";
    private static final String VAULT_PARAMETER_FALSE = "&vault=false";
    private static final String VAULT_PARAMETER_TRUE = "&vault=true";
    private static final String CURRENCY_PARAMETER = "currency";
    private static final String BUYER_COUNTRY_PARAMETER = "buyer-country";
    private static final String LOCALE_PARAMETER = "locale";
    private static final String MESSAGES = "messages";
    private static final String FUNDING_ELIGIBILITY = "funding-eligibility";
    private static final String HOSTED_FIELDS = "hosted-fields";
    private static final String PAYMENT_FIELDS = "payment-fields";
    private static final String CAPTURE = "Capture";
    private static final String INTENT = "intent";
    private static final String PARTNER_ATTRIBUTION_ID = "PartnerAttributionId";
    private static final String CLIENT_TOKEN = "ClientToken";
    private static final String EXPECTED_SDK_URL_LOGIN = "https://www.paypal.com/sdk/js?components=buttons&commit=false" +
            "&currency=isoCode&client-id=CLIENT_ID&buyer-country=US&locale=US_us&vault=true";
    private static final String EXPECTED_SDK_URL_MINI_CART = "https://www.paypal.com/sdk/js?components=buttons," +
            "funding-eligibility,messages&commit=true&currency=isoCode&client-id=CLIENT_ID&buyer-country=US" +
            "&locale=US_us&intent=Capture&vault=false";
    private static final String COMPONENTS_PARAMETER = "components";
    private static final String BUTTONS = "buttons";
    private static final String COMPONENTS_PARAMETER_BUTTONS = COMPONENTS_PARAMETER + "=" + BUTTONS;
    private static final String ENABLE_FUNDING_PARAMETER = "enable-funding";


    @Mock
    private PayPalConfigurationService payPalConfigurationService;
    @Mock
    private UserFacade userFacade;
    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private PayPalClientTokenFacade payPalClientTokenFacade;
    @Mock
    private PayPalConnectService payPalConnectService;
    @Mock
    private CurrencyModel currencyModel;

    @InjectMocks
    private DefaultPayPalSDKFacade unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(payPalConfigurationService.getPayPalPartnerAttributionId()).thenReturn(PARTNER_ATTRIBUTION_ID);
        when(payPalClientTokenFacade.getClientToken()).thenReturn(CLIENT_TOKEN);

        when(payPalConfigurationService.getPayPalSdkJsUrl()).thenReturn(PAYPAL_SDK_JS);
        when(payPalConfigurationService.isPayPalHostedFieldsEnabled()).thenReturn(true);
        when(payPalConfigurationService.isLocalPaymentsEnable()).thenReturn(true);
        when(payPalConfigurationService.isVenmoEnabled()).thenReturn(true);
        when(payPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
        when(payPalConfigurationService.getBuyerCountry()).thenReturn(BUYER_COUNTRY);
        when(payPalConfigurationService.isCommitEnabled()).thenReturn(true);
        when(userFacade.isAnonymousUser()).thenReturn(false);
        when(commonI18NService.getCurrentCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(ISO_CODE);
        when(payPalConfigurationService.getLocale()).thenReturn(LPM_LOCALE);
        when(payPalConfigurationService.getPayPalIntent()).thenReturn(CAPTURE);
        when(payPalConfigurationService.isPayLaterEnabledForNonUsCountries()).thenReturn(true);
        when(payPalConfigurationService.isApplePayEnable()).thenReturn(true);
    }

    @Test
    public void shouldGetPayPalSDK() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(ENVIRONMENT_TYPE_NO_SANDBOX);

        final PayPalSDKData payPalSDKData = unit.getPayPalSDKData();

        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(PAYPAL_SDK_JS));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(COMMIT_PARAMETER_TRUE));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(MESSAGES));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(FUNDING_ELIGIBILITY));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(HOSTED_FIELDS));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(PAYMENT_FIELDS));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(HOSTED_FIELDS));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(CLIENT_ID_PARAMETER));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(CAPTURE));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(VAULT_PARAMETER_FALSE));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(CURRENCY_PARAMETER));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(VENMO_FUNDING));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(LOCALE_PARAMETER));
        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(LPM_LOCALE));

        assertFalse(payPalSDKData.getPayPalSDKUrl().contains(BUYER_COUNTRY_PARAMETER));

        assertEquals(CLIENT_TOKEN, payPalSDKData.getClientToken());
        assertEquals(PARTNER_ATTRIBUTION_ID, payPalSDKData.getPartnerAttributionId());
    }

    @Test
    public void shouldSetVaultFalseWhenItsEnableAndUserIsAnonymous() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(ENVIRONMENT_TYPE_NO_SANDBOX);
        when(payPalConfigurationService.isPayPalVaultEnabled()).thenReturn(true);
        when(userFacade.isAnonymousUser()).thenReturn(true);

        PayPalSDKData payPalSDKData = unit.getPayPalSDKData();

        assertTrue(payPalSDKData.getPayPalSDKUrl().contains(VAULT_PARAMETER_FALSE));
    }

    @Test
    public void shouldGetPayPalSDKTestByPageTypeWhenPageTypeIsLogin() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(SANDBOX);

        PayPalSDKData result = unit.getPayPalSDKData(PayPalPageType.LOGIN);

        assertEquals(EXPECTED_SDK_URL_LOGIN, result.getPayPalSDKUrl());
        assertEquals(CLIENT_TOKEN, result.getClientToken());
        assertEquals(PARTNER_ATTRIBUTION_ID, result.getPartnerAttributionId());
    }

    @Test
    public void shouldGetPayPalSDKTestByPageTypeWhenPageTypeIsMiniCart() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(SANDBOX);
        when(payPalConfigurationService.isPayPalHostedFieldsEnabled()).thenReturn(false);
        when(payPalConfigurationService.isLocalPaymentsEnable()).thenReturn(false);
        when(payPalConfigurationService.isApplePayEnable()).thenReturn(false);
        when(payPalConfigurationService.isVenmoEnabled()).thenReturn(false);
        when(payPalConfigurationService.isPayLaterEnabledForNonUsCountries()).thenReturn(false);

        PayPalSDKData result = unit.getPayPalSDKData(PayPalPageType.MINICART);

        assertEquals(EXPECTED_SDK_URL_MINI_CART, result.getPayPalSDKUrl());
        assertEquals(CLIENT_TOKEN, result.getClientToken());
        assertEquals(PARTNER_ATTRIBUTION_ID, result.getPartnerAttributionId());
        verify(payPalConfigurationService).getPayPalSdkJsUrl();
    }

    @Test
    public void shouldGetPayPalSDKURLAndNoIntentForAccountPage() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(SANDBOX);

        PayPalSDKData result = unit.getPayPalSDKData(PayPalPageType.ACCOUNT);

        assertFalse(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), INTENT));
    }

    @Test
    public void shouldGetPayPalSDKURLForLogin() {
        when(payPalConfigurationService.getEnvironmentType()).thenReturn(SANDBOX);

        PayPalSDKData result = unit.getPayPalSDKData(PayPalPageType.LOGIN);

        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), CLIENT_ID_PARAMETER));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), COMPONENTS_PARAMETER_BUTTONS));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), COMMIT_PARAMETER_FALSE));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), VAULT_PARAMETER_TRUE));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), CURRENCY_PARAMETER));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), BUYER_COUNTRY_PARAMETER));
        assertTrue(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), LOCALE_PARAMETER));
        assertFalse(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), INTENT));

        assertFalse(StringUtils.containsIgnoreCase(result.getPayPalSDKUrl(), ENABLE_FUNDING_PARAMETER));
    }

}
