package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.exception.PayPalUnverifiedAccountException;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.data.PayPalEmailData;
import com.paypal.hybris.facade.facades.PayPalSetUpPasswordFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.user.data.RegisterData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Collections;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.times;

@UnitTest
public class DefaultPayPalRegistrationUserFacadeTest {

    private static final String FIRST_EMAIL = "test1@gmail.com";
    private static final String SECOND_EMAIL = "test2@gmail.com";
    private static final String ACCESS_TOKEN = "accessToken";
    private static final String PAYER_ID = "payerId";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";

    @Mock
    private CustomerFacade customerFacade;
    @Mock
    private PayPalConnectService payPalConnectService;
    @Mock
    private ModelService modelService;
    @Mock
    private UserService userService;
    @Mock
    private PayPalSetUpPasswordFacade payPalSetUpPasswordFacade;
    @Mock
    private CustomerModel customerModel;

    @Spy
    @InjectMocks
    private DefaultPayPalRegistrationUserFacade unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldRegisterPayPalUser() throws DuplicateUidException {
        final PayPalConnectUserData userData = new PayPalConnectUserData();
        final String fullName = FIRST_NAME + " " + LAST_NAME;
        userData.setName(fullName);
        userData.setPayer_id(PAYER_ID);

        when(payPalConnectService.getUserDataByAccessToken(ACCESS_TOKEN)).thenReturn(userData);
        doReturn(FIRST_EMAIL).when(unit).getEmailFromPayPalUserData(userData);
        doNothing().when(customerFacade).register(any(RegisterData.class));
        doNothing().when(payPalSetUpPasswordFacade).setPasswordForSingleAuthenticationActiveToCustomer(FIRST_EMAIL, false);
        doNothing().when(unit).setPayerIdToUser(FIRST_EMAIL.toLowerCase(), PAYER_ID);

        String result = unit.registerPayPalUser(ACCESS_TOKEN);

        assertEquals(PAYER_ID, result);
        verify(payPalConnectService).getUserDataByAccessToken(ACCESS_TOKEN);
        verify(unit, times(2)).getEmailFromPayPalUserData(userData);
        verify(customerFacade).register(any(RegisterData.class));
        verify(payPalSetUpPasswordFacade).setPasswordForSingleAuthenticationActiveToCustomer(FIRST_EMAIL, false);

        verify(unit).setPayerIdToUser(FIRST_EMAIL.toLowerCase(), PAYER_ID);
    }

    @Test(expected = DuplicateUidException.class)
    public void shouldNotRegisterPayPalUserWhenUserAlreadyExistsAndThrowException() throws DuplicateUidException {
        final PayPalConnectUserData userData = new PayPalConnectUserData();
        final String fullName = FIRST_NAME + " " + LAST_NAME;
        userData.setName(fullName);
        userData.setPayer_id(PAYER_ID);

        when(payPalConnectService.getUserDataByAccessToken(ACCESS_TOKEN)).thenReturn(userData);
        doReturn(FIRST_EMAIL).when(unit).getEmailFromPayPalUserData(userData);
        doThrow(DuplicateUidException.class).when(customerFacade).register(any(RegisterData.class));

        unit.registerPayPalUser(ACCESS_TOKEN);
    }

    @Test
    public void shouldSetPayerIdToUser() {
        when(userService.getUserForUID(FIRST_EMAIL)).thenReturn(customerModel);
        doNothing().when(modelService).save(customerModel);

        unit.setPayerIdToUser(FIRST_EMAIL, PAYER_ID);

        verify(userService).getUserForUID(FIRST_EMAIL);
        verify(customerModel).setPayPalPayerId(PAYER_ID);
        verify(modelService).save(customerModel);
    }

    @Test(expected = PayPalUnverifiedAccountException.class)
    public void shouldThrowExceptionWhenUserWithoutEmails() {
        final PayPalConnectUserData userData = new PayPalConnectUserData();
        userData.setEmails(Collections.emptyList());
        userData.setVerified_account(Boolean.TRUE.toString());

        unit.getEmailFromPayPalUserData(userData);
    }

    @Test(expected = PayPalUnverifiedAccountException.class)
    public void shouldThrowExceptionWhenUserHasUnverifiedAccount() {
        final PayPalConnectUserData userData = new PayPalConnectUserData();
        userData.setEmails(Collections.emptyList());
        userData.setVerified_account(Boolean.FALSE.toString());

        unit.getEmailFromPayPalUserData(userData);
    }

    @Test
    public void shouldGetEmailFromPayPalUserData() {
        final PayPalConnectUserData userData = new PayPalConnectUserData();
        PayPalEmailData emailData1 = new PayPalEmailData();
        emailData1.setValue(FIRST_EMAIL);
        emailData1.setConfirmed(Boolean.FALSE.toString());
        emailData1.setPrimary(Boolean.TRUE.toString());
        PayPalEmailData emailData2 = new PayPalEmailData();
        emailData2.setValue(SECOND_EMAIL);
        emailData2.setConfirmed(Boolean.TRUE.toString());
        emailData2.setPrimary(Boolean.TRUE.toString());

        userData.setEmails(List.of(emailData1, emailData2));
        userData.setVerified_account(Boolean.TRUE.toString());

        assertEquals(SECOND_EMAIL, unit.getEmailFromPayPalUserData(userData));

    }
}
