/*

 */
package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.order.OrderFacade;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.PrincipalData;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.servicelayer.session.SessionService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalGuestCheckoutFacadeTest {

    private static final String ANONYMOUS_USER_UID = "anonymous";
    private static final String ANONYMOUS_CHECKOUT = "anonymous_checkout";
    private static final String NOT_ANONYMOUS_CHECKOUT = "not_anonymous_checkout";
    private static final String UID_PART = "|second_part";
    private static final String ANONYMOUS_CHECKOUT_GUID = "anonymous_checkout_guid";
    private static final String FIRST_NAME = "firstName";
    private static final String EMAIL = "email@gmail.com";
    private static final String GUID = "GUID";

    @Mock
    private CustomerFacade customerFacade;
    @Mock
    private PayPalAcceleratorCheckoutFacade checkoutFacade;
    @Mock
    private SessionService sessionService;
    @Mock
    private OrderFacade orderFacade;

    @InjectMocks
    private DefaultPayPalGuestCheckoutFacade unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldProcessCheckoutForAnonymous()
        throws DuplicateUidException {
        final PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
        payPalAddressDetailsData.setEmail(EMAIL);
        payPalAddressDetailsData.setFirstName(FIRST_NAME);

        when(checkoutFacade.getCurrentSessionUserUid()).thenReturn(ANONYMOUS_USER_UID);

        unit.processCheckoutForAnonymous(payPalAddressDetailsData);

        verify(sessionService).setAttribute(ANONYMOUS_CHECKOUT, true);
        verify(customerFacade).createGuestUserForAnonymousCheckout(EMAIL, FIRST_NAME);
    }

    @Test
    public void shouldNotProcessCheckoutForAnonymousAndThrowException()
            throws DuplicateUidException {
        final PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
        payPalAddressDetailsData.setEmail(EMAIL);
        payPalAddressDetailsData.setFirstName(FIRST_NAME);

        when(checkoutFacade.getCurrentSessionUserUid()).thenReturn(ANONYMOUS_USER_UID);
        doThrow(DuplicateUidException.class).when(customerFacade).createGuestUserForAnonymousCheckout(EMAIL, FIRST_NAME);

        unit.processCheckoutForAnonymous(payPalAddressDetailsData);

        verify(sessionService, times(0)).setAttribute(ANONYMOUS_CHECKOUT, true);
    }

    @Test
    public void shouldCreateGuestUserAndSetGuidForNotAnonymousUser()
        throws DuplicateUidException {
        final PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
        payPalAddressDetailsData.setEmail(EMAIL);
        payPalAddressDetailsData.setFirstName(FIRST_NAME);

        when(checkoutFacade.getCurrentSessionUserUid()).thenReturn(NOT_ANONYMOUS_CHECKOUT + UID_PART);

        unit.processCheckoutForAnonymous(payPalAddressDetailsData);

        verify(sessionService).setAttribute(ANONYMOUS_CHECKOUT, true);
        verify(customerFacade).createGuestUserForAnonymousCheckout(EMAIL, FIRST_NAME);
        verify(sessionService).setAttribute(ANONYMOUS_CHECKOUT_GUID, NOT_ANONYMOUS_CHECKOUT);
    }

    @Test
    public void getCostumerEmailByGuidTest() {
        final OrderData orderData = new OrderData();
        final PrincipalData principalData = new PrincipalData();
        principalData.setUid(EMAIL);
        orderData.setUser(principalData);

        when(orderFacade.getOrderDetailsForGUID(GUID)).thenReturn(orderData);

        String result = unit.getCostumerEmailByGuid(GUID);

        assertEquals(EMAIL, result);
        verify(orderFacade).getOrderDetailsForGUID(GUID);
    }
}
