package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.facade.facades.PayPalConnectWithPayPalNotificationFacade;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.times;

@UnitTest
public class DefaultPayPalCustomerFacadeTest {

    private static final String PAYER_ID = "payerID";
    private static final String UID = "UID";
    private static final String REGION = "region";
    private static final String ISO_CODE = "isoCode";
    private static final String COUNTRY = "country";
    private static final String OTHER_COUNTRY = "otherCountry";
    private static final String OTHER_POSTAL_CODE = "otherPostalCode";
    private static final String OTHER_LINE = "otherLine";
    private static final String OTHER_LOCALITY = "otherLocality";
    private static final String POSTAL_CODE = "postalCode";
    private static final String STREET_ADDRESS = "streetAddress";
    private static final String LOCALITY = "locality";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String TOWN = "town";
    private static final String LINE1 = "line1";
    private static final String LINE2 = "line2";
    private static final String PHONE = "phone";

    @Mock
    private CustomerModel customerModel;
    @Mock
    private ModelService modelService;
    @Mock
    private AddressModel addressModel;
    @Mock
    private CountryModel countryModel;
    @Mock
    private RegionModel regionModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private UserService userService;
    @Mock
    private UserFacade userFacade;
    @Mock
    private PayPalRegistrationUserFacade payPalRegistrationUserFacade;
    @Mock
    private CustomerAccountService customerAccountService;
    @Mock
    private PayPalConnectWithPayPalNotificationFacade payPalConnectWithPayPalNotificationFacade;

    @Spy
    @InjectMocks
    private DefaultPayPalCustomerFacade unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        when(userService.getUserForUID(UID)).thenReturn(customerModel);
        when(customerModel.getAddresses()).thenReturn(List.of(addressModel));
        when(addressModel.getCountry()).thenReturn(countryModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
    }

    @Test
    public void shouldReturnTrueWhenPayPalConnectAddressDataAndAddressModelAreNull() {
        final PayPalConnectAddressData payPalAddressData = new PayPalConnectAddressData();

        boolean result = unit.isPayPalAddressPresent(UID, payPalAddressData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnFalseWhenPayPalAddressDetailsNotEquals() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenPostalCodeDiffers() {
        final PayPalConnectAddressData connectAddressData = getConnectAddressData();
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressModel.getPostalcode()).thenReturn(OTHER_POSTAL_CODE);
        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenLine1Differs() {
        final PayPalConnectAddressData connectAddressData = getConnectAddressData();
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressModel.getLine1()).thenReturn(OTHER_LINE);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenLocalityDiffers() {
        final PayPalConnectAddressData connectAddressData = getConnectAddressData();
        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressModel.getTown()).thenReturn(OTHER_LOCALITY);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnTrueWhenPayPalAddressDetailsEquals() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setCountry(COUNTRY);
        connectAddressData.setPostal_code(POSTAL_CODE);
        connectAddressData.setStreet_address(STREET_ADDRESS);
        connectAddressData.setLocality(LOCALITY);

        when(countryModel.getIsocode()).thenReturn(COUNTRY);
        when(addressModel.getPostalcode()).thenReturn(POSTAL_CODE);
        when(addressModel.getLine1()).thenReturn(STREET_ADDRESS);
        when(addressModel.getTown()).thenReturn(LOCALITY);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnFalseWhenOneFieldIsNull() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setCountry(COUNTRY);

        when(countryModel.getIsocode()).thenReturn(null);
        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenPayPalConnectAddressDataCountryIsNull() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();

        when(countryModel.getIsocode()).thenReturn(COUNTRY);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenRegionsNotEquals() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setRegion(REGION);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnTrueWhenRegionsEquals() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setRegion(REGION);

        when(addressModel.getRegion()).thenReturn(regionModel);
        when(regionModel.getIsocodeShort()).thenReturn(REGION);

        boolean result = unit.isPayPalAddressPresent(UID, connectAddressData);

        assertTrue(result);
    }

    @Test
    public void isPayPalPaymentMethodPresent() {
        final PayPalConnectUserData userData = new PayPalConnectUserData();

        when(customerAccountService.getCreditCardPaymentInfos(customerModel, true)).thenReturn(List.of(paymentInfoModel));
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);
        when(paymentInfoModel.getPayerEmail()).thenReturn(UID);
        when(payPalRegistrationUserFacade.getEmailFromPayPalUserData(userData)).thenReturn(UID);

        boolean result = unit.isPayPalPaymentMethodPresent(UID, userData);

        assertTrue(result);
        verify(userService).getUserForUID(UID);
        verify(customerAccountService).getCreditCardPaymentInfos(customerModel, true);
        verify(payPalRegistrationUserFacade).getEmailFromPayPalUserData(userData);
    }

    @Test
    public void isPayerIdInCustomer() {
        when(customerModel.getPayPalPayerId()).thenReturn(PAYER_ID);

        boolean result = unit.isPayerIdInCustomer(UID);

        assertTrue(result);
        verify(userService).getUserForUID(UID);
        verify(customerModel).getPayPalPayerId();
    }

    @Test
    public void isUserDisabled() {
        final Date date = new GregorianCalendar(2021, Calendar.FEBRUARY, 11).getTime();

        boolean result = unit.isUserDisabled(date);

        assertTrue(result);
    }

    @Test
    public void shouldReturnFalseWhenDeactivationTimeIsNull() {
        boolean result = unit.isUserDisabled(null);

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenDeactivationTimeIsGreaterThanCurrentDate() {
        final Date date = new GregorianCalendar(2099, Calendar.FEBRUARY, 11).getTime();
        boolean result = unit.isUserDisabled(date);

        assertFalse(result);
    }

    @Test
    public void shouldUnlinkLoginWithPayPal() {
        when(customerModel.getPayPalPayerId()).thenReturn(PAYER_ID);

        unit.unlinkLoginWithPayPal();

        verify(customerModel).setPayPalPayerId(null);
        verify(modelService).save(customerModel);
    }

    @Test
    public void shouldReturnTrueAddShippingAddressToCurrentUser() {
        AddressData addressData = new AddressData();

        doReturn(false).when(unit).isDuplicateAddressForCustomer(addressData, customerModel);
        doNothing().when(userFacade).addAddress(addressData);

        boolean result = unit.addShippingAddressToCurrentUser(addressData);

        assertTrue(result);
        verify(unit).isDuplicateAddressForCustomer(addressData, customerModel);
        verify(userFacade).addAddress(addressData);
    }

    @Test
    public void shouldReturnFalseAddShippingAddressToCurrentUser() {
        AddressData addressData = new AddressData();
        doReturn(true).when(unit).isDuplicateAddressForCustomer(addressData, customerModel);
        doNothing().when(userFacade).addAddress(addressData);

        boolean result = unit.addShippingAddressToCurrentUser(addressData);

        assertFalse(result);
        verify(unit).isDuplicateAddressForCustomer(addressData, customerModel);
        verify(userFacade, times(0)).addAddress(addressData);
    }

    @Test
    public void isDuplicateAddressForCustomer() {
        CountryData countryData = new CountryData();
        AddressData addressData = new AddressData();
        addressData.setCountry(countryData);

        when(addressModel.getVisibleInAddressBook()).thenReturn(true);

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertTrue(result);
        verify(customerModel).getAddresses();
        verify(addressModel).getVisibleInAddressBook();
    }

    @Test
    public void shouldReturnFalseWhenNoAddresses() {
        CountryData countryData = new CountryData();
        AddressData addressData = new AddressData();
        addressData.setCountry(countryData);

        when(customerModel.getAddresses()).thenReturn(Collections.emptyList());

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertFalse(result);
        verify(customerModel).getAddresses();
    }

    @Test
    public void shouldReturnTrueWhenAddressModelAndAddressDataArePresent() {
        AddressData addressData = getAddressData();

        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressModel.getFirstname()).thenReturn(FIRST_NAME);
        when(addressModel.getLastname()).thenReturn(LAST_NAME);
        when(addressModel.getTown()).thenReturn(TOWN);
        when(addressModel.getPostalcode()).thenReturn(POSTAL_CODE);
        when(addressModel.getLine1()).thenReturn(LINE1);
        when(addressModel.getLine2()).thenReturn(LINE2);
        when(addressModel.getPhone1()).thenReturn(PHONE);
        when(addressModel.getVisibleInAddressBook()).thenReturn(true);
        when(addressModel.getRegion()).thenReturn(regionModel);
        when(regionModel.getIsocode()).thenReturn(ISO_CODE);

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertTrue(result);
        verify(customerModel).getAddresses();
    }

    @Test
    public void shouldReturnTrueWhenPhoneIsEmpty() {
        AddressData addressData = getAddressData();
        addressData.setPhone(null);

        when(countryModel.getIsocode()).thenReturn(ISO_CODE);
        when(addressModel.getFirstname()).thenReturn(FIRST_NAME);
        when(addressModel.getLastname()).thenReturn(LAST_NAME);
        when(addressModel.getTown()).thenReturn(TOWN);
        when(addressModel.getPostalcode()).thenReturn(POSTAL_CODE);
        when(addressModel.getLine1()).thenReturn(LINE1);
        when(addressModel.getLine2()).thenReturn(LINE2);
        when(addressModel.getPhone1()).thenReturn(StringUtils.EMPTY);
        when(addressModel.getVisibleInAddressBook()).thenReturn(true);
        when(addressModel.getRegion()).thenReturn(regionModel);
        when(regionModel.getIsocode()).thenReturn(ISO_CODE);

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertTrue(result);
        verify(customerModel).getAddresses();
    }

    @Test
    public void shouldReturnFalseWhenAddressDataHasOtherCountry() {
        AddressData addressData = new AddressData();
        CountryData countryData = new CountryData();
        countryData.setIsocode(OTHER_COUNTRY);
        addressData.setCountry(countryData);

        when(countryModel.getIsocode()).thenReturn(ISO_CODE);

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertFalse(result);
        verify(customerModel).getAddresses();
    }

    @Test
    public void shouldReturnFalseWhenAddressDataHasNoRegion() {
        AddressData addressData = new AddressData();

        when(addressModel.getRegion()).thenReturn(regionModel);

        boolean result = unit.isDuplicateAddressForCustomer(addressData, customerModel);

        assertFalse(result);
        verify(customerModel).getAddresses();
    }

    private AddressData getAddressData() {
        AddressData addressData = new AddressData();
        CountryData countryData = new CountryData();
        RegionData regionData = new RegionData();
        regionData.setIsocode(ISO_CODE);
        countryData.setIsocode(ISO_CODE);
        addressData.setCountry(countryData);
        addressData.setFirstName(FIRST_NAME);
        addressData.setLastName(LAST_NAME);
        addressData.setTown(TOWN);
        addressData.setPostalCode(POSTAL_CODE);
        addressData.setLine1(LINE1);
        addressData.setLine2(LINE2);
        addressData.setPhone(PHONE);
        addressData.setRegion(regionData);

        return addressData;
    }

    private PayPalConnectAddressData getConnectAddressData() {
        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setCountry(COUNTRY);
        connectAddressData.setPostal_code(POSTAL_CODE);
        connectAddressData.setStreet_address(STREET_ADDRESS);
        connectAddressData.setLocality(LOCALITY);
        return connectAddressData;
    }
}
