package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.exception.PayPalCreditCardRemovalException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.impl.DefaultPayPalCustomerAccountService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentInfoService;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalPaymentSource;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.facade.builders.PaypalGetPaymentTokenRequestBuilder;
import com.paypal.hybris.facade.builders.TokenRequestDirector;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.strategy.payment.PaymentStrategy;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.logging.log4j.util.Strings;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.web.client.HttpClientErrorException;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.doReturn;

@UnitTest
public class DefaultPayPalCreditCardFacadeTest {

    private static final String PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER = "SUBSCRIPTION_ID_PLACEHOLDER";
    private static final String TEST_PAYMENT_METHOD_ID = "testId";
    private static final String TEST_CODE = "testCode";
    private static final String TEST_SUBSCRIPTION_ID = "testSubscriptionId";
    private static final String CUSTOMER_UID= "customerUid";
    private static final String PAYMENT_INFO_PK = "PAYMENT_INFO_PK";
    private static final String BILLING_ADDRESS_PK = "billing address pk";
    private static final String CUSTOMER_ID = "customerId";
    private static final String SETUP_TOKEN_RESPONSE_STATUS = "setup token response status";
    private static final String SETUP_TOKEN_RESPONSE_ID = "setup token response id";
    private static final String PAYER_ID = "payerId";

    @Mock
    private UserService userService;
    @Mock
    private ModelService modelService;
    @Mock
    private CustomerModel customerModel;
    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfo;
    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private AddressModel addressModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel fCard;
    @Mock
    private PayPalCreditCardPaymentInfoModel sCard;
    @Mock
    private DefaultPayPalPaymentInfoService paymentInfoService;
    @Mock
    private UserFacade userFacade;
    @Mock
    private DefaultPayPalCustomerAccountService customerAccountService;
    @Mock
    private TokenRequestDirector tokenRequestDirector;
    @Mock
    private PaymentInfoModel defaultPaymentInfoModel;
    @Mock
    private PayPalAcceleratorCheckoutFacade payPalAcceleratorCheckoutFacade;
    @Mock
    private PaymentStrategy paymentStrategy;
    @Mock
    private SessionService sessionService;
    private List<PaymentStrategy> paymentPopulatorStrategies;

    @Spy
    @InjectMocks
    private DefaultPayPalCreditCardFacade unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        paymentPopulatorStrategies = new ArrayList<>(List.of(paymentStrategy));
        unit.setPaymentPopulatorStrategies(paymentPopulatorStrategies);
        when(paymentStrategy.test(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS)).thenReturn(true);
        doNothing().when(paymentStrategy)
                .populate(any(SetupTokenRequestData.class), any(PayPalSavePaymentForPurchaseLaterRequest.class));
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getDefaultPaymentInfo()).thenReturn(defaultPaymentInfoModel);
        when(payPalCreditCardPaymentInfo.getPk().getLongValueAsString()).thenReturn(PAYMENT_INFO_PK);
        when(payPalCreditCardPaymentInfo.getBillingAddress()).thenReturn(addressModel);
        when(addressModel.getPk().getLongValueAsString()).thenReturn(
                BILLING_ADDRESS_PK);
        when(sessionService.getAttribute(PAYER_ID)).thenReturn(PAYER_ID);
        when(customerAccountService.getCustomerModelByPayerId(PAYER_ID)).thenReturn(customerModel);
    }

    @Test
    public void shouldRequestSetupTokenTest() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        setupTokenRequestData.setPayerId(PAYER_ID);
        when(customerAccountService.getCustomerModel(PAYER_ID)).thenReturn(customerModel);
        PayPalSetupTokenResponse payPalSetupTokenResponse = new PayPalSetupTokenResponse();
        payPalSetupTokenResponse.setId(SETUP_TOKEN_RESPONSE_ID);
        when(customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel)).thenReturn(payPalCreditCardPaymentInfo);
        when(paymentInfoService.getPaypalSetupToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenReturn(payPalSetupTokenResponse);

        PayPalCustomerData customerData = new PayPalCustomerData();
        customerData.setId(CUSTOMER_ID);
        payPalSetupTokenResponse.setCustomer(customerData);
        payPalSetupTokenResponse.setStatus(SETUP_TOKEN_RESPONSE_STATUS);
        payPalSetupTokenResponse.setLinks(new ArrayList<>());

        CCSetupTokenData result = unit.requestSetupToken(setupTokenRequestData);

        assertEquals(CUSTOMER_ID, result.getCustomerId());
        assertEquals(SETUP_TOKEN_RESPONSE_STATUS, result.getStatus());
        assertEquals(SETUP_TOKEN_RESPONSE_ID, result.getId());
        assertEquals(PAYMENT_INFO_PK, result.getPaymentInfoPK());
        assertThat(result.getApproveLink()).isEmpty();

        verify(payPalCreditCardPaymentInfo).setSubscriptionId(SETUP_TOKEN_RESPONSE_ID);
        verify(modelService).save(payPalCreditCardPaymentInfo);
    }

    @Test(expected = RuntimeException.class)
    public void shouldThrowRuntimeExceptionWhenNoSuitablePaymentStrategy() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.APPLEPAY);
        when(customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel)).thenReturn(payPalCreditCardPaymentInfo);

        unit.requestSetupToken(setupTokenRequestData);
    }

    @Test
    public void shouldSetVaultIdToCustomerWhenItIsAbsentTest() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        setupTokenRequestData.setPayerId(PAYER_ID);
        when(customerAccountService.getCustomerModel(PAYER_ID)).thenReturn(customerModel);
        PayPalSetupTokenResponse payPalSetupTokenResponse = new PayPalSetupTokenResponse();
        when(customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel)).thenReturn(payPalCreditCardPaymentInfo);
        when(paymentInfoService.getPaypalSetupToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenReturn(payPalSetupTokenResponse);
        when(customerModel.getVaultCustomerId()).thenReturn(Strings.EMPTY);

        PayPalCustomerData customerData = new PayPalCustomerData();
        customerData.setId(CUSTOMER_ID);
        payPalSetupTokenResponse.setId(SETUP_TOKEN_RESPONSE_ID);
        payPalSetupTokenResponse.setCustomer(customerData);
        payPalSetupTokenResponse.setLinks(new ArrayList<>());

        unit.requestSetupToken(setupTokenRequestData);

        verify(customerModel).setVaultCustomerId(CUSTOMER_ID);
        verify(modelService).save(customerModel);
    }

    @Test
    public void shouldNotSetVaultIdToCustomerWhenItIsPresentTest() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPayerId(PAYER_ID);
        when(customerAccountService.getCustomerModel(PAYER_ID)).thenReturn(customerModel);
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        PayPalSetupTokenResponse payPalSetupTokenResponse = new PayPalSetupTokenResponse();
        when(customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel)).thenReturn(payPalCreditCardPaymentInfo);
        when(paymentInfoService.getPaypalSetupToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenReturn(payPalSetupTokenResponse);
        when(customerModel.getVaultCustomerId()).thenReturn(CUSTOMER_ID);
        PayPalCustomerData customerData = new PayPalCustomerData();
        customerData.setId(CUSTOMER_ID);
        payPalSetupTokenResponse.setId(SETUP_TOKEN_RESPONSE_ID);
        payPalSetupTokenResponse.setCustomer(customerData);
        payPalSetupTokenResponse.setLinks(new ArrayList<>());

        unit.requestSetupToken(setupTokenRequestData);

        verify(customerModel, times(0)).setVaultCustomerId(CUSTOMER_ID);
        verify(modelService, times(0)).save(customerModel);
    }

    @Test(expected = HttpClientErrorException.class)
    public void shouldRemoveExistingPaymentInfoIfTheRequestToSetupTokenIsFailedTest() {
        SetupTokenRequestData setupTokenRequestData = new SetupTokenRequestData();
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        setupTokenRequestData.setPayerId(PAYER_ID);
        when(customerAccountService.getCustomerModel(PAYER_ID)).thenReturn(customerModel);
        when(customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel)).thenReturn(payPalCreditCardPaymentInfo);
        when(paymentInfoService.getPaypalSetupToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenThrow(
                HttpClientErrorException.class);

        unit.requestSetupToken(setupTokenRequestData);

        verify(paymentInfoService).removePaymentInfoByPK(PAYMENT_INFO_PK);
    }

    @Test
    public void shouldRequestPaymentTokenTestForPayPalPM() throws Exception {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setPaymentInfoPK(PAYMENT_INFO_PK);
        PayPalSetupTokenResponse response = getPayPalSetupTokenResponse();
        doNothing().when(tokenRequestDirector).constructGetPaymentTokenRequest(any(PaypalGetPaymentTokenRequestBuilder.class), eq(paymentTokenData));
        when(paymentInfoService.getPaymentInfoByPK(PAYMENT_INFO_PK)).thenReturn(Optional.of(payPalCreditCardPaymentInfo));
        when(paymentInfoService.getPaypalPaymentToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenReturn(response);
        when(payPalAcceleratorCheckoutFacade.createPayPalPaymentSubscription(response, true, payPalCreditCardPaymentInfo, customerModel))
                .thenReturn(payPalCreditCardPaymentInfo);
        doReturn(customerModel).when(customerAccountService).getCustomerModel(PAYER_ID);
        unit.requestPaymentToken(paymentTokenData);

        verify(payPalAcceleratorCheckoutFacade).createPayPalPaymentSubscription(response, true, payPalCreditCardPaymentInfo, customerModel);
    }

    @Test
    public void shouldRequestPaymentTokenTestForHostedFieldsPM() throws Exception {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setPaymentInfoPK(PAYMENT_INFO_PK);
        PayPalSetupTokenResponse response = new PayPalSetupTokenResponse();
        response.setPaymentSource(new PayPalPaymentSource());
        doNothing().when(tokenRequestDirector).constructGetPaymentTokenRequest(any(PaypalGetPaymentTokenRequestBuilder.class), eq(paymentTokenData));
        when(paymentInfoService.getPaymentInfoByPK(PAYMENT_INFO_PK)).thenReturn(Optional.of(payPalCreditCardPaymentInfo));
        when(paymentInfoService.getPaypalPaymentToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenReturn(response);
        when(customerAccountService.updateStubCreditCardPaymentInfo(payPalCreditCardPaymentInfo, response))
                .thenReturn(payPalCreditCardPaymentInfo);

        unit.requestPaymentToken(paymentTokenData);

        verify(customerAccountService).updateStubCreditCardPaymentInfo(payPalCreditCardPaymentInfo, response);
    }

    @Test(expected = HttpClientErrorException.class)
    public void shouldRemoveExistingPaymentInfoIfTheRequestToPaymentTokenIsFailedTest() throws Exception {
        PaymentTokenData paymentTokenData = new PaymentTokenData();
        paymentTokenData.setPaymentInfoPK(PAYMENT_INFO_PK);
        doNothing().when(tokenRequestDirector).constructGetPaymentTokenRequest(any(PaypalGetPaymentTokenRequestBuilder.class), eq(paymentTokenData));
        when(paymentInfoService.getPaymentInfoByPK(PAYMENT_INFO_PK)).thenReturn(Optional.of(payPalCreditCardPaymentInfo));
        when(paymentInfoService.getPaypalPaymentToken(any(PayPalSavePaymentForPurchaseLaterRequest.class))).thenThrow(
                HttpClientErrorException.class);

        unit.requestPaymentToken(paymentTokenData);

        verify(paymentInfoService).removePaymentInfoByPK(PAYMENT_INFO_PK);
    }

    @Test
    public void shouldReturnExpiredStatusWhenFirstCardHasExpiredStatusAndSecondHasExpireSoonStatus() {
        final List<PaymentInfoModel> cards = List.of(fCard, sCard);

        when(fCard.isSaved()).thenReturn(Boolean.TRUE);
        when(fCard.getDuplicate()).thenReturn(Boolean.FALSE);
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRED);
        when(sCard.isSaved()).thenReturn(Boolean.TRUE);
        when(sCard.getDuplicate()).thenReturn(Boolean.FALSE);
        when(customerModel.getPaymentInfos()).thenReturn(cards);

        assertEquals(ExpirationStatus.EXPIRED.getCode(), unit.getCardsExpirationStatus());
    }

    @Test
    public void shouldReturnExpireSoonStatusWhenCardHasExpireSoonStatus() {
        final List<PaymentInfoModel> cards = List.of(fCard);

        when(fCard.isSaved()).thenReturn(Boolean.TRUE);
        when(fCard.getDuplicate()).thenReturn(Boolean.FALSE);
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRE_SOON);

        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.isHasNewExpireSoonCard()).thenReturn(Boolean.TRUE);
        when(customerModel.getPaymentInfos()).thenReturn(cards);

        assertEquals(ExpirationStatus.EXPIRE_SOON.getCode(), unit.getCardsExpirationStatus());
    }

    @Test
    public void shouldReturnExpiredStatusWhenCardHasExpiredStatus() {
        final List<PaymentInfoModel> cards = List.of(fCard);

        when(fCard.isSaved()).thenReturn(Boolean.TRUE);
        when(fCard.getDuplicate()).thenReturn(Boolean.FALSE);
        when(fCard.getExpirationStatus()).thenReturn(ExpirationStatus.EXPIRED);
        when(customerModel.getPaymentInfos()).thenReturn(cards);

        assertEquals(ExpirationStatus.EXPIRED.getCode(), unit.getCardsExpirationStatus());
    }

    @Test
    public void shouldGetCardsExpirationStatus() {
        final ExpirationStatus expirationStatus = ExpirationStatus.NOT_EXPIRED;

        when(userService.getUserForUID(CUSTOMER_UID, CustomerModel.class)).thenReturn(customerModel);
        doReturn(ExpirationStatus.NOT_EXPIRED.getCode()).when(unit).getCardsExpirationStatus(customerModel);

        String result = unit.getCardsExpirationStatus(CUSTOMER_UID);

        assertEquals(expirationStatus.getCode(), result);
        verify(userService).getUserForUID(CUSTOMER_UID, CustomerModel.class);
    }

    @Test
    public void shouldGetCardsExpirationStatusWhenCustomerIsNull() {
        String result = unit.getCardsExpirationStatus(CUSTOMER_UID);

        assertEquals(ExpirationStatus.NOT_EXPIRED.getCode(), result);
    }

    @Test
    public void shouldDeletedCreditCardFromPayPalSide() {
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, TEST_PAYMENT_METHOD_ID))
                .thenReturn(payPalCreditCardPaymentInfo);
        when(payPalCreditCardPaymentInfo.getSubscriptionId()).thenReturn(TEST_SUBSCRIPTION_ID);
        when(paymentInfoService.deletePaypalPaymentToken(TEST_SUBSCRIPTION_ID)).thenReturn(Boolean.TRUE);

        when(defaultPaymentInfoModel.getCode()).thenReturn(TEST_CODE);
        when(defaultPaymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(defaultPaymentInfoModel.getDuplicate()).thenReturn(Boolean.FALSE);
        when(payPalCreditCardPaymentInfo.getCode()).thenReturn(TEST_CODE);
        when(customerModel.getPaymentInfos()).thenReturn(List.of(defaultPaymentInfoModel));

        unit.deleteCreditCardFromPayPal(TEST_PAYMENT_METHOD_ID);

        verify(paymentInfoService, times(1)).deletePaypalPaymentToken(TEST_SUBSCRIPTION_ID);
        verify(userFacade, times(1)).removeCCPaymentInfo(TEST_PAYMENT_METHOD_ID);
        verify(modelService, times(1)).save(customerModel);
    }

    @Test(expected = PayPalCreditCardRemovalException.class)
    public void shouldThrowCreditCardRemovalException() {
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, TEST_PAYMENT_METHOD_ID))
                .thenReturn(payPalCreditCardPaymentInfo);
        when(payPalCreditCardPaymentInfo.getSubscriptionId()).thenReturn(TEST_SUBSCRIPTION_ID);
        when(paymentInfoService.deletePaypalPaymentToken(TEST_SUBSCRIPTION_ID)).thenReturn(Boolean.FALSE);

        unit.deleteCreditCardFromPayPal(TEST_PAYMENT_METHOD_ID);
    }

    @Test
    public void shouldDeletedPayPalPaymentMethod() {
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, TEST_PAYMENT_METHOD_ID))
                .thenReturn(payPalCreditCardPaymentInfo);
        when(payPalCreditCardPaymentInfo.getSubscriptionId()).thenReturn(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);

        when(defaultPaymentInfoModel.getCode()).thenReturn(TEST_CODE);
        when(defaultPaymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(defaultPaymentInfoModel.getDuplicate()).thenReturn(Boolean.FALSE);
        when(payPalCreditCardPaymentInfo.getCode()).thenReturn(TEST_CODE);
        when(customerModel.getPaymentInfos()).thenReturn(List.of(defaultPaymentInfoModel));
        when(paymentInfoService.deletePaypalPaymentToken(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER)).thenReturn(true);

        unit.deleteCreditCardFromPayPal(TEST_PAYMENT_METHOD_ID);

        verify(userFacade, times(1)).removeCCPaymentInfo(TEST_PAYMENT_METHOD_ID);
        verify(modelService, times(1)).save(customerModel);
        verify(paymentInfoService).deletePaypalPaymentToken(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);
    }

    @Test
    public void shouldDeletedCreditCardBySubscriptionId() {
        when(paymentInfoService.getCreditCardBySubscriptionId(TEST_SUBSCRIPTION_ID))
                .thenReturn(Optional.of(fCard));
        when(fCard.getUser()).thenReturn(customerModel);
        when(defaultPaymentInfoModel.getCode()).thenReturn(TEST_CODE);
        when(defaultPaymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(defaultPaymentInfoModel.getDuplicate()).thenReturn(Boolean.FALSE);
        when(fCard.getCode()).thenReturn(TEST_CODE);
        when(customerModel.getPaymentInfos()).thenReturn(List.of(defaultPaymentInfoModel));

        unit.deleteCreditCardFromWebhookEvent(TEST_SUBSCRIPTION_ID);

        verify(modelService, times(1)).save(customerModel);
        verify(customerAccountService, times(1)).deleteCCPaymentInfo(customerModel, fCard);
    }

    private PayPalSetupTokenResponse getPayPalSetupTokenResponse() {
        PayPalSetupTokenResponse response = new PayPalSetupTokenResponse();
        PayPalPaymentSource paymentSource = new PayPalPaymentSource();
        PayPalData payPalData = new PayPalData();
        paymentSource.setPaypal(payPalData);
        response.setPaymentSource(paymentSource);
        return response;
    }

}
