/*

 */
package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.exception.PayPalPaymentInfoNotFoundException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.core.strategy.storedcredential.StoredCredentialStrategy;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.ApplePayAddressDetailsData;
import com.paypal.hybris.data.ApplePayLineItem;
import com.paypal.hybris.data.ApplePaySessionDetailsData;
import com.paypal.hybris.data.AuthenticationResultData;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.Level2Level3CardData;
import com.paypal.hybris.data.NameData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalCheckoutData;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalCreateOrderData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalHostedFieldsCreateOrderData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalPaymentSource;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.PayPalVaultOrderRequestData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorfacades.order.AcceleratorCheckoutFacade;
import de.hybris.platform.acceleratorservices.uiexperience.UiExperienceService;
import de.hybris.platform.commercefacades.address.AddressVerificationFacade;
import de.hybris.platform.commercefacades.address.data.AddressVerificationResult;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.order.CartFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.DeliveryOrderEntryGroupData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.address.AddressVerificationDecision;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.delivery.DeliveryService;
import de.hybris.platform.commerceservices.enums.CountryType;
import de.hybris.platform.commerceservices.enums.UiExperienceLevel;
import de.hybris.platform.commerceservices.order.CommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.commerceservices.service.data.CommerceOrderResult;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.c2l.LanguageModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.delivery.DeliveryModeModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CalculationService;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_NONDECIMAL_CURRENCY;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.anyString;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import static org.assertj.core.api.Assertions.assertThat;

@UnitTest
public class DefaultPayPalAcceleratorCheckoutFacadeTest {

    private static final String PAYPAL_PAYMENT_METHOD_TYPE = "paypal";
    private static final String APPLEPAY_PAYMENT_METHOD_TYPE = "applepay";
    private static final String NAME = "name";
    private static final String ORDER_ID = "orderId";
    private static final String PAYER_ID = "payerId";
    private static final String BUYER_ID = "buyerId";
    private static final String PAYMENT_ID = "paymentId";
    private static final String PAYER_EMAIL = "payer@gmail.com";
    private static final String SECURITY_CODE = "securityCode";
    private static final String USD = "USD";
    private static final String PAYPAL_CREDIT_CARD_PAYMENT_INFO = "PayPalCreditCardPaymentInfo";
    private static final String CREDIT_CARD_PAYMENT_INFO = "CreditCardPaymentInfo";
    private static final String POSSIBLE = "POSSIBLE";
    private static final String NO = "NO";
    private static final String N = "N";
    private static final String U = "U";
    private static final String B = "B";
    private static final String D = "D";
    private static final String LOGIN_PAGE = "LOGIN";
    private static final String ID_VALUE = "id";
    private static final String PRICE_VALUE_STRING = "19.99";
    private static final BigDecimal PRICE_VALUE = new BigDecimal(PRICE_VALUE_STRING);
    private static final String US_ISO_CODE = "US";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String PAYPAL_HOSTED_FIELDS_CARD = "paypal_hosted_fields_card";
    private static final String VISA = "visa";
    private static final String LAST_DIGITS = "1111";
    private static final String UK = "UK";
    private static final String EMAIL_ADDRESS = "emailAddress";
    private static final String FULL_NAME = "fullName";
    private static final String LINE_1 = "line1";
    private static final String LINE_2 = "line2";
    private static final String POSTAL_CODE = "postalCode";
    private static final String COUNTRY_CODE = "countryCode";
    private static final String REGION = "region";
    private static final String CITY = "city";
    private static final String ISO_CODE = "IsoCode";
    private static final String TOWN = "Town";
    private static final String EXPRESS_CHECKOUT = "EXPRESS_CHECKOUT";
    private static final String MARK_CHECKOUT = "MARK_CHECKOUT";
    private static final String UNITED_KINGDOM = "United Kingdom";
    private static final String GB = "GB";
    private static final String BRAZIL = "Brazil";
    private static final String BRA = "BRA";
    private static final String VAULT_CUSTOMER_ID = "vault customer id";
    private static final String USAGE_TYPE = "MERCHANT";
    private static final String CUSTOMER_TYPE = "CONSUMER";
    private static final String INTENT = "INTENT";
    private static final Double PRICE = 10.0;
    private static final Double MATCHING_PRICE = 19.99;
    private static final String YES = "YES";
    private static final String UNKNOWN = "UNKNOWN";
    public static final String Y = "Y";

    @Mock
    private AddressVerificationFacade addressVerificationFacade;
    @Mock
    private UserFacade userFacade;
    @Mock
    private CartFacade cartFacade;
    @Mock
    private CartService cartService;
    @Mock
    private DeliveryService deliveryService;
    @Mock
    private CustomerFacade customerFacade;
    @Mock
    private CommerceCheckoutService checkoutService;
    @Mock
    private ModelService modelService;
    @Mock
    private UserService userService;
    @Mock
    private PayPalPaymentService payPalPaymentService;
    @Mock
    private PayPalCustomerAccountService payPalCustomerAccountService;
    @Mock
    private CheckoutCustomerStrategy customerStrategy;
    @Mock
    private Populator<PayPalAddressDetailsData, AddressData> payPalAddressDataPopulator;
    @Mock
    private Populator<AddressData, PayPalAddressDetailsData> payPalAddressDataReversePopulator;
    @Mock
    private Populator<AddressData, AddressModel> addressReversePopulator;
    @Mock
    private CustomerAccountService customerAccountService;
    @Mock
    private PayPalCartService payPalCartService;
    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private UiExperienceService uiExperienceService;
    @Mock
    private Converter<OrderModel, OrderData> orderConverter;
    @Mock
    private Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter;
    @Mock
    private Converter<ApplePayAddressDetailsData, AddressData> applePayAddressDataConverter;
    @Mock
    private Populator<AddressData, ApplePayAddressDetailsData> applePayAddressDataReversePopulator;
    @Mock
    private Populator<ApplePayAddressDetailsData, PayPalAddressDetailsData> applePayPayPalAddressDataPopulator;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private CartModel cartModel;
    @Mock
    private AbstractOrderEntryModel orderEntryModel;
    @Mock
    private List<StoredCredentialStrategy> storedCredentialStrategies;
    @Mock
    private StoredCredentialStrategy storedCredentialStrategy;
    @Mock
    private PayPalPaymentInfoService payPalPaymentInfoService;
    @Mock
    private PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel;
    @Mock
    private CreditCardPaymentInfoModel creditCardPaymentInfoModel;
    @Mock
    private CreditCardPaymentInfoModel notDefaultPaymentInfo;
    @Mock
    private PaymentInfoModel paymentInfoModel;
    @Mock
    private AddressModel addressModel;
    @Mock
    private PaymentTransactionEntryModel paymentTransactionEntryModel;
    @Mock
    private CountryModel countryModel;
    @Mock
    private RegionModel regionModel;
    @Mock
    private OrderModel orderModel;
    @Mock
    private UserModel userModel;
    @Mock
    private TitleModel titleModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel firstPayPalCreditCardPaymentInfoModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel secondPayPalCreditCardPaymentInfoModel;
    @Mock
    private BaseStoreService baseStoreService;
    @Mock
    private BaseStoreModel baseStoreModel;
    @Mock
    private LanguageModel languageModel;
    @Mock
    private CurrencyModel currencyModel;
    @Mock
    private CalculationService calculationService;
    @Mock
    private Converter<CartModel, CartData> cartConverter;
    @Mock
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    @Mock
    private Converter<AbstractOrderModel, PayPalOrderRequestData> vaultOrderRequestDataConverter;
    @Mock
    private Converter<CartData, Level2Level3CardData> payPalL2L3DataConverter;
    @Mock
    private DeliveryModeModel deliveryModeModel;
    @Captor
    private ArgumentCaptor<PayPalAddressDetailsData> addressDetailsDataCaptor;
    @Mock
    private BreakdownCalculationService breakdownCalculationService;

    private final CartData cartData = new CartData();

    @Spy
    @InjectMocks
    private DefaultPayPalAcceleratorCheckoutFacade unit;

    private CountryData validCountryData;
    private List<OrderEntryData> orderEntryData;
    private Level2Level3CardData level2Level3CardData;
    private PayPalOrderDetailsData orderDetailsData;
    private PayPalAddressDetailsData addressDetailsData;
    private ApplePayAddressDetailsData applePayAddressDetailsData;
    private AddressData addressData;
    private CCPaymentInfoData payPalPaymentSubscription;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        storedCredentialStrategies.add(storedCredentialStrategy);
        orderDetailsData = new PayPalOrderDetailsData();
        addressDetailsData = new PayPalAddressDetailsData();
        applePayAddressDetailsData = new ApplePayAddressDetailsData();
        addressData = new AddressData();
        payPalPaymentSubscription = new CCPaymentInfoData();
        level2Level3CardData = new Level2Level3CardData();
        validCountryData = new CountryData();
        validCountryData.setName(UNITED_KINGDOM);
        validCountryData.setIsocode(GB);

        PriceData priceData = new PriceData();
        OrderEntryData entryData = new OrderEntryData();
        entryData.setBasePrice(priceData);
        entryData.setQuantity(1L);
        priceData.setValue(PRICE_VALUE);
        priceData.setCurrencyIso(USD);
        cartData.setTotalTax(priceData);
        cartData.setTotalPriceWithTax(priceData);
        cartData.setTotalDiscounts(priceData);
        cartData.setDeliveryCost(priceData);
        cartData.setTotalPrice(priceData);
        cartData.setTotalPriceWithTax(priceData);
        cartData.setSubTotalWithoutQuoteDiscounts(priceData);
        orderEntryData = List.of(entryData);
        cartData.setEntries(orderEntryData);

        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(vaultOrderRequestDataConverter.convert(cartModel)).thenReturn(new PayPalVaultOrderRequestData());
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(PAYPAL_NONDECIMAL_CURRENCY);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(INTENT);

        when(cartConverter.convert(cartModel)).thenReturn(cartData);

        PayPalVaultOrderRequestData requestData = createOrderRequestData();
        when(vaultOrderRequestDataConverter.convert(cartModel)).thenReturn(requestData);
        when(breakdownCalculationService.calculateTotalAmount(cartModel)).thenReturn(BigDecimal.TEN);
        when(applePayAddressDataConverter.convert(applePayAddressDetailsData)).thenReturn(addressData);
    }

    @Test
    public void shouldReturnFalseWhenCountryNameNotSupported() {
        final CountryData invalidCountryData = new CountryData();
        invalidCountryData.setName(BRAZIL);

        doReturn(Collections.singletonList(validCountryData)).when(unit).getCountries(CountryType.SHIPPING);

        assertFalse(unit.isCountrySupported(invalidCountryData));
    }

    @Test
    public void shouldReturnFalseWhenCountryNameSupported() {
        final CountryData invalidCountryData = new CountryData();
        invalidCountryData.setName(UNITED_KINGDOM);

        doReturn(Collections.singletonList(validCountryData)).when(unit).getCountries(CountryType.SHIPPING);

        assertTrue(unit.isCountrySupported(invalidCountryData));
    }

    @Test
    public void shouldReturnFalseWhenCountryIsoCodeNotSupported() {
        final CountryData invalidCountryData = new CountryData();
        invalidCountryData.setIsocode(BRA);

        doReturn(Collections.singletonList(validCountryData)).when(unit).getCountries(CountryType.SHIPPING);

        assertFalse(unit.isCountrySupported(invalidCountryData));
    }

    @Test
    public void shouldReturnTrueWhenCountryIsoCodeSupported() {
        final CountryData invalidCountryData = new CountryData();
        invalidCountryData.setIsocode(GB);

        doReturn(Collections.singletonList(validCountryData)).when(unit).getCountries(CountryType.SHIPPING);

        assertTrue(unit.isCountrySupported(invalidCountryData));
    }

    @Test
    public void shouldSuccessfullyProcessExpressCheckout() {
        final String code = UUID.randomUUID().toString();
        final String id = UUID.randomUUID().toString();
        final PayPalOrderDetailsData orderDetails = new PayPalOrderDetailsData();
        final PayPalAddressDetailsData addressDetailsData = new PayPalAddressDetailsData();
        orderDetails.setShippingAddress(addressDetailsData);
        orderDetails.setAmount("10.0");
        final AddressData addressData = new AddressData();
        addressData.setVisibleInAddressBook(true);
        final CartData cartData = new CartData();
        cartData.setDeliveryAddress(addressData);
        final AddressVerificationResult verificationResult = new AddressVerificationResult();
        verificationResult.setDecision(AddressVerificationDecision.ACCEPT);
        final CCPaymentInfoData payPalPaymentSubscription = new CCPaymentInfoData();
        payPalPaymentSubscription.setId(id);

        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerModel.getTitle()).thenReturn(titleModel);
        when(titleModel.getName()).thenReturn(NAME);
        when(titleModel.getCode()).thenReturn(code);
        when(userFacade.getDefaultAddress()).thenReturn(addressData);
        when(addressVerificationFacade.verifyAddressData(any())).thenReturn(verificationResult);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        doReturn(Boolean.TRUE).when(unit).isCountrySupported(any());
        doReturn(payPalPaymentSubscription).when(unit).createPayPalPaymentSubscriptionForExpressCheckout(orderDetails, PAYPAL_PAYMENT_METHOD_TYPE);
        doReturn(Boolean.TRUE).when(unit).setPaymentDetails(id);
        when(cartModel.getUser()).thenReturn(userModel);

        assertTrue(unit.processExpressCheckout(orderDetails, PAYPAL_PAYMENT_METHOD_TYPE));
        verify(unit).setCheapestDeliveryModeForCheckout();
        verify(unit).createPayPalPaymentSubscriptionForExpressCheckout(orderDetails, PAYPAL_PAYMENT_METHOD_TYPE);
        verify(unit).setPaymentDetails(id);
    }

    @Test
    public void shouldNotSetDeliveryAddressWhenPickUpInStoreTrue() {

        payPalPaymentSubscription.setId(PAYMENT_ID);
        addressDetailsData.setEmail(EMAIL_ADDRESS);
        addressDetailsData.setFirstName(FIRST_NAME);
        orderDetailsData.setShippingAddress(addressDetailsData);

        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        doReturn(Boolean.TRUE).when(unit).isPickupInStore();
        doReturn(cartData).when(unit).getCheckoutCart();
        doNothing().when(unit).updatePayPalOrder(orderDetailsData);
        doReturn(payPalPaymentSubscription).when(unit).createPayPalPaymentSubscriptionForExpressCheckout(orderDetailsData, PAYPAL_PAYMENT_METHOD_TYPE);
        doReturn(Boolean.TRUE).when(unit).setPaymentDetails(PAYMENT_ID);

        boolean result = unit.processExpressCheckout(orderDetailsData, PAYPAL_PAYMENT_METHOD_TYPE);

        verify(unit, never()).setDeliveryAddress(any(AddressData.class));

        assertTrue(result);
    }

    @Test
    public void shouldNotProcessExpressCheckout() {
        final PayPalOrderDetailsData orderDetails = new PayPalOrderDetailsData();
        final PayPalAddressDetailsData addressDetailsData = new PayPalAddressDetailsData();
        orderDetails.setShippingAddress(addressDetailsData);
        final AddressVerificationResult verificationResult = new AddressVerificationResult();
        verificationResult.setDecision(AddressVerificationDecision.REJECT);

        doReturn(Boolean.TRUE).when(unit).isCountrySupported(any());
        doNothing().when(payPalAddressDataPopulator).populate(any(PayPalAddressDetailsData.class), any(AddressData.class));
        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(cartFacade.getSessionCart()).thenReturn(new CartData());
        when(addressVerificationFacade.verifyAddressData(any(AddressData.class))).thenReturn(verificationResult);

        boolean result = unit.processExpressCheckout(orderDetails, PAYPAL_PAYMENT_METHOD_TYPE);

        assertFalse(result);
    }

    @Test
    public void shouldUpdateCheckoutPaymentInfoOrderId() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(payPalPaymentInfoModel);

        unit.updateCheckoutPaymentInfoOrderId(ORDER_ID);

        verify(payPalPaymentInfoModel).setPayPalOrderId(ORDER_ID);
        verify(modelService).save(payPalPaymentInfoModel);
    }

    @Test
    public void shouldNotUpdateCheckoutPaymentInfoOrderIdIfTypeCreditCardPaymentInfoModel() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);

        unit.updateCheckoutPaymentInfoOrderId(ORDER_ID);

        verify(payPalPaymentInfoModel, times(0)).setPayPalOrderId(ORDER_ID);
        verify(modelService, times(0)).save(payPalPaymentInfoModel);
    }

    @Test
    public void shouldSetSelectedPaymentInfoToCart() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getUser()).thenReturn(customerModel);
        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID)).thenReturn(creditCardPaymentInfoModel);
        when(modelService.clone(any(CreditCardPaymentInfoModel.class))).thenReturn(creditCardPaymentInfoModel);
        doNothing().when(modelService).save(creditCardPaymentInfoModel);
        when(checkoutService.setPaymentInfo(any(CommerceCheckoutParameter.class))).thenReturn(true);

        boolean result = unit.setSelectedPaymentInfoToCart(PAYMENT_ID);

        assertTrue(result);
        verify(cartFacade, times(2)).hasSessionCart();
        verify(cartService, times(2)).getSessionCart();
        verify(cartModel).getUser();
        verify(customerStrategy, times(3)).getCurrentUserForCheckout();
        verify(customerAccountService).getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID);
        verify(modelService).clone(any(CreditCardPaymentInfoModel.class));
        verify(modelService).save(creditCardPaymentInfoModel);
        verify(checkoutService).setPaymentInfo(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldNotSetSelectedPaymentInfoToCartWhenNoPaymentInfoForCode() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getUser()).thenReturn(customerModel);
        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID)).thenReturn(null);

        boolean result = unit.setSelectedPaymentInfoToCart(PAYMENT_ID);

        assertFalse(result);
    }

    @Test
    public void isPickupInStoreWhenDeliveryOrderGroupsAreEmpty() {
        final CartData cartData = new CartData();
        cartData.setDeliveryOrderGroups(new ArrayList<>());

        doReturn(cartData).when(unit).getCheckoutCart();

        boolean result = unit.isPickupInStore();

        assertTrue(result);
        verify(unit).getCheckoutCart();
    }

    @Test
    public void isPickupInStoreDeliveryOrderGroupsArePresent() {
        final CartData cartData = new CartData();
        cartData.setDeliveryOrderGroups(List.of(new DeliveryOrderEntryGroupData()));

        doReturn(cartData).when(unit).getCheckoutCart();

        boolean result = unit.isPickupInStore();

        assertFalse(result);
        verify(unit).getCheckoutCart();
    }

    @Test
    public void isLocalPaymentFlow() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(CreditCardType.LOCAL_PAYMENT.getCode());
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(cartData).when(unit).getCheckoutCart();
        when(defaultPayPalConfigurationService.isCommitEnabled()).thenReturn(true);

        boolean result = unit.isLocalPaymentFlow();

        assertTrue(result);
        verify(unit, times(2)).getCheckoutCart();
        verify(defaultPayPalConfigurationService).isCommitEnabled();
    }

    @Test
    public void shouldReturnFalseWhenNoPaymentInfo() {
        final CartData cartData = new CartData();

        doReturn(cartData).when(unit).getCheckoutCart();

        boolean result = unit.isLocalPaymentFlow();

        assertFalse(result);
    }

    @Test
    public void shouldReturnFalseWhenCardTypeIsNotLocalPayment() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(CreditCardType.VISA.getCode());
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(cartData).when(unit).getCheckoutCart();

        boolean result = unit.isLocalPaymentFlow();

        assertFalse(result);
        verify(unit, times(2)).getCheckoutCart();
    }

    @Test
    public void shouldReturnFalseWhenCommitDisabled() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(CreditCardType.LOCAL_PAYMENT.getCode());
        cartData.setPaymentInfo(paymentInfoData);

        when(defaultPayPalConfigurationService.isCommitEnabled()).thenReturn(false);
        doReturn(cartData).when(unit).getCheckoutCart();

        boolean result = unit.isLocalPaymentFlow();

        assertFalse(result);
        verify(unit, times(2)).getCheckoutCart();
    }

    @Test
    public void shouldCreatePayPalOrderForExpressCheckout() {
        doReturn(false).when(unit).isPickupInStore();
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(ID_VALUE);
        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);

        String result = unit.createPayPalOrderForExpressCheckout();

        assertEquals(ORDER_ID, result);
        verify(unit).isPickupInStore();
        verify(defaultPayPalConfigurationService).isSaveOrderFlow();
        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldCreatePayPalOrderForExpressCheckoutWhenChangePaymentButtonActiveAndAddressExist() {
        final CustomerData customerData = new CustomerData();
        final AddressData addressData = new AddressData();

        customerData.setDefaultShippingAddress(addressData);
        addressData.setVisibleInAddressBook(Boolean.TRUE);

        doReturn(false).when(unit).isPickupInStore();
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(ID_VALUE);
        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);
        when(customerFacade.getCurrentCustomer()).thenReturn(customerData);
        when(payPalPaymentInfoService.isChangePaymentButtonActive()).thenReturn(Boolean.TRUE);
        String result = unit.createPayPalOrderForExpressCheckout();

        assertEquals(ORDER_ID, result);
        verify(unit).isPickupInStore();
        verify(defaultPayPalConfigurationService).isSaveOrderFlow();
//        verify(payPalAddressDataReversePopulator).populate(eq(addressData), any(PayPalAddressDetailsData.class));
        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldCreatePayPalOrderForExpressCheckoutWhenChangePaymentButtonActiveAndAddressNotExist() {

        final CustomerData customerData = new CustomerData();

        doReturn(false).when(unit).isPickupInStore();
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(ID_VALUE);
        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);
        when(customerFacade.getCurrentCustomer()).thenReturn(customerData);
        when(payPalPaymentInfoService.isChangePaymentButtonActive()).thenReturn(Boolean.TRUE);
        String result = unit.createPayPalOrderForExpressCheckout();

        assertEquals(ORDER_ID, result);
        verify(unit).isPickupInStore();
        verify(defaultPayPalConfigurationService).isSaveOrderFlow();
        verify(payPalAddressDataReversePopulator, never()).populate(any(AddressData.class), any(PayPalAddressDetailsData.class));
        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldCreatePayPalOrderForExpressCheckoutWhenPickupInStore() {
        doReturn(true).when(unit).isPickupInStore();
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(ID_VALUE);
        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);

        String result = unit.createPayPalOrderForExpressCheckout();

        assertEquals(ORDER_ID, result);
        verify(unit).isPickupInStore();
        verify(defaultPayPalConfigurationService).isSaveOrderFlow();
        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void createApplePayOrderForExpressCheckoutTest() {
        ApplePayAddressDetailsData applePayShippingAddress = GenericBuilder.of(ApplePayAddressDetailsData::new)
                .with(ApplePayAddressDetailsData::setAddressLines, new ArrayList<>(List.of("Test Road, Eastern Docks")))
                .with(ApplePayAddressDetailsData::setCountry, "United Kingdom")
                .with(ApplePayAddressDetailsData::setCountryCode, "GB")
                .with(ApplePayAddressDetailsData::setEmailAddress, "testepam@gmail.com")
                .with(ApplePayAddressDetailsData::setFamilyName, "Tester")
                .with(ApplePayAddressDetailsData::setGivenName, "Test")
                .with(ApplePayAddressDetailsData::setLocality, "Southampton")
                .with(ApplePayAddressDetailsData::setPhoneNumber, "222222222")
                .with(ApplePayAddressDetailsData::setPostalCode, "SO14 3GG")
                .build();

        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn("intent");
        doReturn(Boolean.TRUE).when(unit).isPickupInStore();
        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);

        String result = unit.createApplePayOrderForExpressCheckout(applePayShippingAddress);

        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);

        assertEquals(ORDER_ID, result);
    }

    @Test
    public void shouldCreatePayPalPaymentSubscriptionForExpressCheckout() {
        final PayPalOrderDetailsData orderDetails = new PayPalOrderDetailsData();
        orderDetails.setShippingAddress(new PayPalAddressDetailsData());
        orderDetails.setBillingAddress(new PayPalAddressDetailsData());
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();

        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(payPalCustomerAccountService.createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any())).thenReturn(payPalPaymentInfoModel);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(creditCardPaymentInfoConverter.convert(payPalPaymentInfoModel)).thenReturn(paymentInfoData);

        CCPaymentInfoData result = unit.createPayPalPaymentSubscriptionForExpressCheckout(orderDetails, PAYPAL_PAYMENT_METHOD_TYPE);

        assertEquals(paymentInfoData, result);
        verify(customerStrategy).getCurrentUserForCheckout();
        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(CustomerModel.class), any(CCPaymentInfoData.class), any(AddressModel.class), any());
        verify(modelService).create(AddressModel.class);
    }

    @Test
    public void shouldCreatePayPalPaymentSubscriptionForMarkCheckout() {
        final CartData cartData = new CartData();
        final AddressData addressData = new AddressData();
        final CountryData countryData = new CountryData();
        countryData.setIsocode(US_ISO_CODE);
        addressData.setCountry(countryData);
        cartData.setDeliveryAddress(addressData);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        doReturn(false).when(unit).isPickupInStore();

        unit.createPayPalPaymentSubscriptionForMarkCheckout(ORDER_ID, CreditCardType.LOCAL_PAYMENT, "my_credit_card", Boolean.TRUE);

        verify(cartFacade).getSessionCart();
        verify(modelService).create(AddressModel.class);
        verify(unit).isPickupInStore();
        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(), any(), any(), any());
    }

    @Test
    public void shouldCreatePayPalPaymentSubscriptionForMarkCheckoutWhenPickupInStore() {
        final CartData cartData = new CartData();
        final AddressData addressData = new AddressData();
        final CountryData countryData = new CountryData();
        countryData.setIsocode(US_ISO_CODE);
        addressData.setCountry(countryData);
        cartData.setDeliveryAddress(addressData);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        doReturn(false).when(unit).isPickupInStore();
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        doNothing().when(addressReversePopulator).populate(cartData.getDeliveryAddress(), addressModel);

        unit.createPayPalPaymentSubscriptionForMarkCheckout(ORDER_ID, CreditCardType.LOCAL_PAYMENT, "my_credit_card", Boolean.TRUE);

        verify(cartFacade).getSessionCart();
        verify(modelService).create(AddressModel.class);
        verify(unit).isPickupInStore();
        verify(addressReversePopulator).populate(cartData.getDeliveryAddress(), addressModel);
        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(), any(), any(), any());
    }

    @Test
    public void shouldUpdatePayPalPaymentSubscription() {
        final PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(PAYMENT_ID);
        final PayPalAddressDetailsData addressDetailsData = new PayPalAddressDetailsData();
        final PaymentSourceData paymentSourceData = new PaymentSourceData();
        final CardData cardData = new CardData();
        cardData.setLastDigits(LAST_DIGITS);
        cardData.setBrand(VISA);
        paymentSourceData.setCard(cardData);
        orderDetailsData.setPayment(paymentSourceData);
        orderDetailsData.setBillingAddress(addressDetailsData);
        orderDetailsData.setShippingAddress(addressDetailsData);

        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID))
                .thenReturn(payPalPaymentInfoModel);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(payPalCustomerAccountService.updatePayPalPaymentSubscription(customerModel,
                paymentInfoData, addressModel, payPalPaymentInfoModel)).thenReturn(payPalPaymentInfoModel);
        when(creditCardPaymentInfoConverter.convert(payPalPaymentInfoModel)).thenReturn(paymentInfoData);

        unit.updatePayPalPaymentSubscription(orderDetailsData, paymentInfoData);

        verify(customerAccountService).getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID);
        verify(payPalCustomerAccountService).updatePayPalPaymentSubscription(customerModel, paymentInfoData, addressModel, payPalPaymentInfoModel);
    }

    @Test
    public void shouldUpdatePayPalPaymentSubscriptionWhenPayerEmailPresentAndPaymentIsNull() {
        final PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setPayerEmail(PAYER_EMAIL);
        paymentInfoData.setId(PAYMENT_ID);
        final PayPalAddressDetailsData addressDetailsData = new PayPalAddressDetailsData();
        orderDetailsData.setBillingAddress(addressDetailsData);
        orderDetailsData.setShippingAddress(addressDetailsData);
        orderDetailsData.setBuyerId(BUYER_ID);

        when(customerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(customerAccountService.getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID))
                .thenReturn(payPalPaymentInfoModel);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(payPalCustomerAccountService.updatePayPalPaymentSubscription(customerModel,
                paymentInfoData, addressModel, payPalPaymentInfoModel)).thenReturn(payPalPaymentInfoModel);
        when(creditCardPaymentInfoConverter.convert(payPalPaymentInfoModel)).thenReturn(paymentInfoData);

        CCPaymentInfoData result = unit.updatePayPalPaymentSubscription(orderDetailsData, paymentInfoData);

        assertNull(result.getCardNumber());
        assertNull(result.getCardType());
        assertEquals(PAYER_EMAIL, result.getPayerEmail());
        assertEquals(BUYER_ID, result.getPayerId());
        verify(customerAccountService).getCreditCardPaymentInfoForCode(customerModel, PAYMENT_ID);
        verify(payPalCustomerAccountService).updatePayPalPaymentSubscription(customerModel, paymentInfoData, addressModel, payPalPaymentInfoModel);
    }

    @Test
    public void shouldGetCurrentSessionUserUid() {
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getUser()).thenReturn(customerModel);
        when(customerModel.getUid()).thenReturn(PAYER_EMAIL);

        String result = unit.getCurrentSessionUserUid();

        assertEquals(PAYER_EMAIL, result);
        verify(cartService).getSessionCart();
        verify(cartModel).getUser();
        verify(customerModel).getUid();
    }

    @Test
    public void shouldCreatePayPalOrderAndFundingTrue() {
        final PayPalCreateOrderData payPalCreateOrderData = new PayPalCreateOrderData();
        final PayPalHostedFieldsCreateOrderData hostedFieldsData = new PayPalHostedFieldsCreateOrderData();
        hostedFieldsData.setIsShouldBeSaved(true);
        payPalCreateOrderData.setHostedFieldsData(hostedFieldsData);
        payPalCreateOrderData.setFunding(PAYPAL_HOSTED_FIELDS_CARD);

        doReturn(cartData).when(unit).getCheckoutCart();
        when(cartModel.getDeliveryMode()).thenReturn(deliveryModeModel);
        doReturn(false).when(unit).isPickupInStore();
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(payPalPaymentService.createOrder(any(PayPalHostedFieldsOrderRequestData.class))).thenReturn(ORDER_ID);

        unit.createPayPalOrder(payPalCreateOrderData);

        verify(unit, times(1)).getCheckoutCart();
        verify(unit).isPickupInStore();
        verify(userService).getCurrentUser();

        verify(payPalPaymentService).createOrder(any(PayPalHostedFieldsOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }


    @Test
    public void shouldCreatePayPalOrderWhenHostedFieldsDataIsNotSaved() {
        final PayPalCreateOrderData payPalCreateOrderData = new PayPalCreateOrderData();
        final PayPalHostedFieldsCreateOrderData hostedFieldsData = new PayPalHostedFieldsCreateOrderData();
        hostedFieldsData.setIsShouldBeSaved(false);
        payPalCreateOrderData.setHostedFieldsData(hostedFieldsData);
        payPalCreateOrderData.setFunding(PAYPAL_HOSTED_FIELDS_CARD);

        doReturn(true).when(unit).isPickupInStore();
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);

        when(payPalPaymentService.createOrder(any(PayPalHostedFieldsOrderRequestData.class))).thenReturn(ORDER_ID);

        unit.createPayPalOrder(payPalCreateOrderData);

        verify(unit).isPickupInStore();
        verify(userService).getCurrentUser();

        verify(payPalPaymentService).createOrder(any(PayPalHostedFieldsOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldCreatePayPalOrderWhenFundingIsApplePay() {
        final PayPalCreateOrderData payPalCreateOrderData = new PayPalCreateOrderData();
        final PayPalHostedFieldsCreateOrderData hostedFieldsData = new PayPalHostedFieldsCreateOrderData();
        final ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
        hostedFieldsData.setIsShouldBeSaved(true);
        payPalCreateOrderData.setHostedFieldsData(hostedFieldsData);
        payPalCreateOrderData.setApplePayShippingAddress(applePayAddressDetailsData);
        payPalCreateOrderData.setFunding(APPLEPAY_PAYMENT_METHOD_TYPE);

        final CartData cartData = new CartData();
        final PriceData priceData = new PriceData();
        priceData.setValue(PRICE_VALUE);
        priceData.setCurrencyIso(USD);
        cartData.setTotalTax(priceData);
        cartData.setTotalDiscounts(priceData);
        cartData.setDeliveryCost(priceData);
        cartData.setTotalPrice(priceData);
        cartData.setTotalPriceWithTax(priceData);

        doReturn(false).when(unit).isPickupInStore();
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(cartFacade.hasSessionCart()).thenReturn(false);

        when(payPalPaymentService.createOrder(any(PayPalOrderRequestData.class))).thenReturn(ORDER_ID);

        unit.createPayPalOrder(payPalCreateOrderData);

        verify(unit).isPickupInStore();
        verify(userService).getCurrentUser();

        verify(payPalPaymentService).createOrder(any(PayPalOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldCreatePayPalOrderAndFundingFalse() {
        final PayPalCreateOrderData payPalCreateOrderData = new PayPalCreateOrderData();
        final PayPalHostedFieldsCreateOrderData hostedFieldsData = new PayPalHostedFieldsCreateOrderData();
        payPalCreateOrderData.setFunding(PAYPAL_PAYMENT_METHOD_TYPE);
        payPalCreateOrderData.setHostedFieldsData(hostedFieldsData);

        doReturn(cartData).when(unit).getCheckoutCart();
        when(cartModel.getDeliveryMode()).thenReturn(deliveryModeModel);
        doReturn(false).when(unit).isPickupInStore();
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(USD);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(ID_VALUE);
        when(payPalPaymentService.createOrder(any(PayPalVaultOrderRequestData.class))).thenReturn(ORDER_ID);

        unit.createPayPalOrder(payPalCreateOrderData);

        verify(unit, times(1)).getCheckoutCart();
        verify(unit).isPickupInStore();
        verify(defaultPayPalConfigurationService).isSaveOrderFlow();

        verify(payPalPaymentService).createOrder(any(PayPalVaultOrderRequestData.class));
        verify(payPalCartService).setPayPalOrderIdForCurrentCart(ORDER_ID);
    }

    @Test
    public void shouldGetPayPalOrderDetailsWithParameter() {
        final PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();

        when(payPalPaymentService.getOrderDetails(ORDER_ID)).thenReturn(orderDetailsData);

        assertEquals(orderDetailsData, unit.getPayPalOrderDetails(ORDER_ID));
    }

    @Test
    public void shouldGetPayPalOrderDetails() {
        final CartData cartData = new CartData();
        cartData.setPayPalOrderId(ORDER_ID);
        final PayPalOrderDetailsData payPalOrderDetailsData = new PayPalOrderDetailsData();

        doReturn(cartData).when(unit).getCheckoutCart();
        when(payPalPaymentService.getOrderDetails(ORDER_ID)).thenReturn(payPalOrderDetailsData);

        assertEquals(payPalOrderDetailsData, unit.getPayPalOrderDetails());
    }

    @Test
    public void shouldReturnTrueWhenLiabilityShiftPossible() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(POSSIBLE);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnTrueWhenLiabilityShiftYes() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(YES);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnFalseWhenLiabilityShiftUnknown() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(UNKNOWN);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertFalse(result);
    }

    @Test
    public void shouldReturnTrueWhenLiabilityShiftNoAndEnrolmentStatusN() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(NO);
        resultData.setEnrollmentStatus(N);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnTrueWhenLiabilityShiftNoAndEnrolmentStatusU() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(NO);
        resultData.setEnrollmentStatus(U);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnTrueWhenLiabilityShiftNoAndEnrolmentStatusB() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(NO);
        resultData.setEnrollmentStatus(B);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertTrue(result);
    }

    @Test
    public void shouldReturnFalseWhenLiabilityShiftNoAndEnrolmentStatusY() {
        final AuthenticationResultData resultData = new AuthenticationResultData();
        resultData.setLiabilityShift(NO);
        resultData.setEnrollmentStatus(Y);

        boolean result = unit.isThreeDsVerificationSuccess(resultData);

        assertFalse(result);
    }

    @Test
    public void shouldAddAddressForNewUser() {
        final String ALABAMA = "Alabama";
        final String AL = "AL";
        final String fullName = FIRST_NAME + " " + LAST_NAME;

        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setCountry(US_ISO_CODE);
        connectAddressData.setRegion(ALABAMA);


        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(commonI18NService.getCountry(anyString())).thenReturn(countryModel);
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);
        when(countryModel.getIsocode()).thenReturn(AL);
        when(commonI18NService.getRegion(any(CountryModel.class), anyString())).thenReturn(regionModel);
        doNothing().when(modelService).save(regionModel);
        when(customerModel.getName()).thenReturn(fullName);

        unit.addAddressForNewUser(connectAddressData, PAYER_ID);

        verify(modelService).create(AddressModel.class);
        verify(commonI18NService).getCountry(US_ISO_CODE);
        verify(payPalCustomerAccountDao).findCustomerByPayPalPayerId(PAYER_ID);
        verify(modelService).save(regionModel);
        verify(addressModel).setFirstname(FIRST_NAME);
        verify(addressModel).setLastname(LAST_NAME);
        verify(customerAccountService).saveAddressEntry(customerModel, addressModel);
    }

    @Test
    public void shouldAddAddressForNewUserWhenNoRegion() {
        final String fullName = FIRST_NAME + " " + LAST_NAME;

        final PayPalConnectAddressData connectAddressData = new PayPalConnectAddressData();
        connectAddressData.setCountry(US_ISO_CODE);

        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(commonI18NService.getCountry(anyString())).thenReturn(countryModel);
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);
        when(customerModel.getName()).thenReturn(fullName);

        unit.addAddressForNewUser(connectAddressData, PAYER_ID);

        verify(modelService).create(AddressModel.class);
        verify(commonI18NService).getCountry(US_ISO_CODE);
        verify(payPalCustomerAccountDao).findCustomerByPayPalPayerId(PAYER_ID);
        verify(modelService, times(0)).save(regionModel);
        verify(addressModel).setFirstname(FIRST_NAME);
        verify(addressModel).setLastname(LAST_NAME);
        verify(customerAccountService).saveAddressEntry(customerModel, addressModel);
    }

    @Test
    public void shouldAuthorizePayment() {
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(PAYMENT_ID);
        when(checkoutService.authorizePayment(any(CommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        boolean result = unit.authorizePayment(cartModel);

        assertTrue(result);
        verify(cartModel).getPaymentInfo();
        verify(creditCardPaymentInfoModel).getSubscriptionId();
        verify(checkoutService).authorizePayment(any(CommerceCheckoutParameter.class));
        verify(paymentTransactionEntryModel).getTransactionStatus();
    }

    @Test
    public void shouldNotAuthorizePaymentWhenTransactionStatusIsRejected() {
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(PAYMENT_ID);
        when(checkoutService.authorizePayment(any(CommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        boolean result = unit.authorizePayment(cartModel);

        assertFalse(result);
    }

    @Test
    public void shouldAuthorizePaymentWhenTransactionStatusIsReview() {
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(PAYMENT_ID);
        when(checkoutService.authorizePayment(any(CommerceCheckoutParameter.class))).thenReturn(paymentTransactionEntryModel);
        when(paymentTransactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REVIEW.name());

        boolean result = unit.authorizePayment(cartModel);

        assertTrue(result);
    }

    @Test
    public void shouldNotAuthorizePaymentWhenResultOfAuthorizationIsNull() {
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(PAYMENT_ID);
        when(checkoutService.authorizePayment(any(CommerceCheckoutParameter.class))).thenReturn(null);

        boolean result = unit.authorizePayment(cartModel);

        assertFalse(result);
    }

    @Test
    public void shouldNotAuthorizePaymentWhenSubscriptionIdIsNull() {
        when(cartModel.getPaymentInfo()).thenReturn(creditCardPaymentInfoModel);
        when(creditCardPaymentInfoModel.getSubscriptionId()).thenReturn(null);

        boolean result = unit.authorizePayment(cartModel);

        assertFalse(result);
    }

    @Test
    public void shouldNotAuthorizePaymentWhenCartModelIsNull() {
        boolean result = unit.authorizePayment((CartModel) null);

        assertFalse(result);
    }

    @Test
    public void shouldPlaceOrderByCart() throws InvalidCartException {
        final CommerceOrderResult orderResult = new CommerceOrderResult();
        final OrderData orderData = new OrderData();
        orderResult.setOrder(orderModel);

        when(uiExperienceService.getUiExperienceLevel()).thenReturn(UiExperienceLevel.DESKTOP);
        when(checkoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(orderConverter.convert(orderModel)).thenReturn(orderData);

        unit.placeOrderByCart(cartModel);

        verify(uiExperienceService).getUiExperienceLevel();
        verify(checkoutService).placeOrder(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldNotPlaceOrderByCartWhenCartModelIsNull() throws InvalidCartException {
        OrderData result = unit.placeOrderByCart(null);

        assertNull(result);
        verify(uiExperienceService, times(0)).getUiExperienceLevel();
        verify(checkoutService, times(0)).placeOrder(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldRemoveCartAfterPlaceOrder() {
        doNothing().when(modelService).remove(cartModel);
        doNothing().when(modelService).refresh(orderModel);

        unit.afterPlaceOrder(cartModel, orderModel);

        verify(modelService).remove(cartModel);
        verify(modelService).refresh(orderModel);
    }

    @Test
    public void shouldNotRemoveCartAfterPlaceOrderWhenOrderModelIsNull() {
        unit.afterPlaceOrder(cartModel, null);

        verify(modelService, times(0)).remove(cartModel);
        verify(modelService, times(0)).refresh(orderModel);
    }

    @Test
    public void shouldPrepareCartForCheckout() {
        unit.prepareCartForCheckout(cartModel);

        verify(checkoutService, times(1)).calculateCart(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldNotPrepareCartForCheckoutWhenCartModelIsNull() {
        unit.prepareCartForCheckout(null);

        verify(checkoutService, times(0)).calculateCart(any(CommerceCheckoutParameter.class));
    }

    @Test
    public void shouldSetCreditCardType() {
        assertEquals(CreditCardType.PAYPAL, unit.setCreditCardType("credit"));
        assertEquals(CreditCardType.VENMO, unit.setCreditCardType("venmo"));
        assertEquals(CreditCardType.CARD, unit.setCreditCardType("card"));
        assertEquals(CreditCardType.PAYPAL_HOSTED_FIELDS_CARD, unit.setCreditCardType(PAYPAL_HOSTED_FIELDS_CARD));
        assertEquals(CreditCardType.APPLEPAY, unit.setCreditCardType(APPLEPAY_PAYMENT_METHOD_TYPE));
        assertEquals(CreditCardType.LOCAL_PAYMENT, unit.setCreditCardType("local_payment"));
        assertEquals(CreditCardType.PAYPAL, unit.setCreditCardType("paylater"));
    }

    @Test
    public void shouldReturnAllAvailablePaymentsWhenHostedFieldsAndPayPalVaultEnabled() {
        final CCPaymentInfoData hostedFieldsPaymentMethod = new CCPaymentInfoData();
        final CCPaymentInfoData payPalPaymentMethod = new CCPaymentInfoData();
        hostedFieldsPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        payPalPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL);

        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(hostedFieldsPaymentMethod, payPalPaymentMethod));

        when(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled()).thenReturn(Boolean.TRUE);
        when(defaultPayPalConfigurationService.isPayPalVaultEnabled()).thenReturn(Boolean.TRUE);

        List<CCPaymentInfoData> result = unit.getAvailableCCPaymentInfosForCurrentUser();

        assertThat(result).hasSize(2);
        assertThat(result).containsExactlyInAnyOrder(payPalPaymentMethod, hostedFieldsPaymentMethod);
    }

    @Test
    public void shouldNotReturnPaymentsWhenHostedFieldsAndPayPalVaultDisabled() {
        final CCPaymentInfoData hostedFieldsPaymentMethod = new CCPaymentInfoData();
        final CCPaymentInfoData payPalPaymentMethod = new CCPaymentInfoData();
        hostedFieldsPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        payPalPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL);

        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(hostedFieldsPaymentMethod, payPalPaymentMethod));

        when(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled()).thenReturn(Boolean.FALSE);
        when(defaultPayPalConfigurationService.isPayPalVaultEnabled()).thenReturn(Boolean.FALSE);

        List<CCPaymentInfoData> result = unit.getAvailableCCPaymentInfosForCurrentUser();

        assertThat(result).hasSize(0);
    }

    @Test
    public void shouldReturnAllAvailablePaymentsWhenHostedFieldsVaultDisabled() {
        final CCPaymentInfoData hostedFieldsPaymentMethod = new CCPaymentInfoData();
        final CCPaymentInfoData payPalPaymentMethod = new CCPaymentInfoData();
        hostedFieldsPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        payPalPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL);

        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(hostedFieldsPaymentMethod, payPalPaymentMethod));

        when(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled()).thenReturn(Boolean.FALSE);
        when(defaultPayPalConfigurationService.isPayPalVaultEnabled()).thenReturn(Boolean.TRUE);

        List<CCPaymentInfoData> result = unit.getAvailableCCPaymentInfosForCurrentUser();

        assertThat(result).hasSize(1);
        assertEquals(payPalPaymentMethod, result.get(0));
    }

    @Test
    public void shouldReturnAllAvailablePaymentsWhenPayPalVaultDisabled() {
        final CCPaymentInfoData hostedFieldsPaymentMethod = new CCPaymentInfoData();
        final CCPaymentInfoData payPalPaymentMethod = new CCPaymentInfoData();
        hostedFieldsPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        payPalPaymentMethod.setPaymentProvider(PayPalPaymentProvider.PAYPAL);

        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(hostedFieldsPaymentMethod, payPalPaymentMethod));

        when(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled()).thenReturn(Boolean.TRUE);
        when(defaultPayPalConfigurationService.isPayPalVaultEnabled()).thenReturn(Boolean.FALSE);

        List<CCPaymentInfoData> result = unit.getAvailableCCPaymentInfosForCurrentUser();

        assertThat(result).hasSize(1);
        assertEquals(hostedFieldsPaymentMethod, result.get(0));
    }

    @Test
    public void shouldPerformExpressCheckout() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(PAYPAL_PAYMENT_METHOD_TYPE);
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(false).when(unit).hasShippingItems();
        doReturn(true).when(unit).setDeliveryModeIfAvailable();
        doReturn(true).when(unit).setDefaultPaymentInfoForCheckout();

        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(userFacade.getDefaultAddress()).thenReturn(new AddressData());
        when(cartModel.getPaymentInfo()).thenReturn(payPalPaymentInfoModel);
        when(payPalPaymentInfoModel.getItemtype()).thenReturn(PAYPAL_CREDIT_CARD_PAYMENT_INFO);

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.SUCCESS, result);
    }

    @Test
    public void shouldPerformExpressCheckoutButNotSavePaymentInfoWhenWrongPaymentType() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(APPLEPAY_PAYMENT_METHOD_TYPE);
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(false).when(unit).hasShippingItems();
        doReturn(true).when(unit).setDeliveryModeIfAvailable();
        doReturn(true).when(unit).setDefaultPaymentInfoForCheckout();

        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(userFacade.getDefaultAddress()).thenReturn(new AddressData());

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.SUCCESS, result);
    }

    @Test
    public void shouldPerformExpressCheckoutButNotSavePaymentInfoWhenNoPaymentInfo() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(PAYPAL_PAYMENT_METHOD_TYPE);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(false).when(unit).hasShippingItems();
        doReturn(true).when(unit).setDeliveryModeIfAvailable();
        doReturn(true).when(unit).setDefaultPaymentInfoForCheckout();

        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(userFacade.getDefaultAddress()).thenReturn(new AddressData());

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.SUCCESS, result);
    }

    @Test
    public void shouldPerformExpressCheckoutButNotSavePaymentInfoWhenItemTypeNotMatch() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(PAYPAL_PAYMENT_METHOD_TYPE);
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(false).when(unit).hasShippingItems();
        doReturn(true).when(unit).setDeliveryModeIfAvailable();
        doReturn(true).when(unit).setDefaultPaymentInfoForCheckout();

        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(userFacade.getDefaultAddress()).thenReturn(new AddressData());
        when(cartModel.getPaymentInfo()).thenReturn(payPalPaymentInfoModel);
        when(payPalPaymentInfoModel.getItemtype()).thenReturn(CREDIT_CARD_PAYMENT_INFO);

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.SUCCESS, result);
        verify(modelService, times(0)).save(cartModel.getPaymentInfo());
        verify(modelService, times(0)).refresh(cartModel);
    }

    @Test
    public void shouldNotPerformExpressCheckoutWhenResultIsErrorPaymentInfo() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(PAYPAL_PAYMENT_METHOD_TYPE);
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(false).when(unit).hasShippingItems();
        doReturn(true).when(unit).setDeliveryModeIfAvailable();
        doReturn(false).when(unit).setDefaultPaymentInfoForCheckout();

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.ERROR_PAYMENT_INFO, result);
    }

    @Test
    public void shouldNotPerformExpressCheckoutWhenResultIsErrorDeliveryAddress() {
        final CartData cartData = new CartData();
        final CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setCardType(PAYPAL_PAYMENT_METHOD_TYPE);
        cartData.setPaymentInfo(paymentInfoData);

        doReturn(true).when(unit).isExpressCheckoutEnabledForStore();
        doReturn(true).when(unit).hasShippingItems();
        doReturn(false).when(unit).setDefaultDeliveryAddressForCheckout();

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.ERROR_DELIVERY_ADDRESS, result);
    }

    @Test
    public void shouldNotPerformExpressCheckoutWhenCurrentBaseStoreIsNull() {

        doReturn(false).when(unit).isExpressCheckoutEnabledForStore();

        AcceleratorCheckoutFacade.ExpressCheckoutResult result = unit.performExpressCheckout();

        assertEquals(AcceleratorCheckoutFacade.ExpressCheckoutResult.ERROR_NOT_AVAILABLE, result);
    }

    @Test
    public void shouldAuthorizePaymentWithSecurityCode() {
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(payPalPaymentInfoModel);
        when(payPalPaymentInfoModel.isSaveOrderFlowActive()).thenReturn(true);
        when(payPalPaymentService.savePayPalOrder(payPalPaymentInfoModel, cartModel)).thenReturn(true);

        boolean result = unit.authorizePayment(SECURITY_CODE);

        assertTrue(result);
        verify(cartFacade).hasSessionCart();
        verify(cartService).getSessionCart();
        verify(cartModel).getPaymentInfo();
        verify(payPalPaymentInfoModel).isSaveOrderFlowActive();
        verify(payPalPaymentService).savePayPalOrder(payPalPaymentInfoModel, cartModel);
    }

    @Test
    public void shouldNotAuthorizePaymentWithSecurityCodeWhenNoSessionCart() {
        when(cartFacade.hasSessionCart()).thenReturn(false);

        boolean result = unit.authorizePayment(SECURITY_CODE);

        assertFalse(result);
    }

    @Test
    public void shouldGetPayPalCheckoutData() {
        when(defaultPayPalConfigurationService.isExpressCheckoutEnabled()).thenReturn(true);
        when(defaultPayPalConfigurationService.isPayPalHostedFieldsEnabled()).thenReturn(true);
        when(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled()).thenReturn(true);
        when(payPalPaymentInfoService.isSavedPayPalAccountsAmountLimited()).thenReturn(true);
        when(payPalPaymentInfoService.isChangePaymentButtonActive()).thenReturn(true);
        when(payPalPaymentInfoService.isSavedCreditCardsAmountLimited()).thenReturn(true);

        PayPalCheckoutData result = unit.getPayPalCheckoutData(LOGIN_PAGE);

        assertTrue(result.getPayPalPaymentMethod().getPayPalExpressEnabled());
        assertTrue(result.getPayPalPaymentMethod().getIsPayPalPaymentAmountLimited());
        assertTrue(result.getCreditCardPaymentMethod().isHostedFieldsEnable());
        assertTrue(result.getCreditCardPaymentMethod().isVaultFlowEnable());
    }

    @Test
    public void shouldGetCCPaymentInfos() {
        when(customerAccountService.getCreditCardPaymentInfos(customerModel, true))
                .thenReturn(List.of(creditCardPaymentInfoModel, notDefaultPaymentInfo));
        when(customerModel.getDefaultPaymentInfo()).thenReturn(paymentInfoModel);
        when(creditCardPaymentInfoConverter.convert(creditCardPaymentInfoModel)).thenReturn(new CCPaymentInfoData());

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true, customerModel);

        assertEquals(2, result.size());
        verify(creditCardPaymentInfoConverter).convert(creditCardPaymentInfoModel);
    }

    @Test
    public void shouldGetCCPaymentInfosWhenPaymentInfoEqualsDefaultPaymentInfo() {
        when(customerAccountService.getCreditCardPaymentInfos(customerModel, true))
                .thenReturn(List.of(creditCardPaymentInfoModel, notDefaultPaymentInfo));
        when(customerModel.getDefaultPaymentInfo()).thenReturn(paymentInfoModel);
        when(creditCardPaymentInfoConverter.convert(creditCardPaymentInfoModel)).thenReturn(new CCPaymentInfoData());
        when(creditCardPaymentInfoConverter.convert(notDefaultPaymentInfo)).thenReturn(new CCPaymentInfoData());

        List<CCPaymentInfoData> result = unit.getCCPaymentInfos(true, customerModel);

        assertEquals(2, result.size());
        verify(creditCardPaymentInfoConverter).convert(creditCardPaymentInfoModel);
    }

    @Test
    public void shouldGetCCPaymentInfoByOrderId() {
        when(payPalPaymentInfoService.getPaymentInfoByOrderId(ID_VALUE)).thenReturn(Optional.of(payPalPaymentInfoModel));
        when(creditCardPaymentInfoConverter.convert(payPalPaymentInfoModel)).thenReturn(new CCPaymentInfoData());

        unit.getCCPaymentInfoByOrderId(ID_VALUE);

        verify(creditCardPaymentInfoConverter).convert(payPalPaymentInfoModel);
        verify(payPalPaymentInfoService).getPaymentInfoByOrderId(ID_VALUE);
    }

    @Test(expected = PayPalPaymentInfoNotFoundException.class)
    public void shouldThrowPayPalPaymentInfoNotFoundException() {
        when(payPalPaymentInfoService.getPaymentInfoByOrderId(ID_VALUE)).thenReturn(Optional.empty());

        unit.getCCPaymentInfoByOrderId(ID_VALUE);
    }

    @Test
    public void shouldCreatePayPalPaymentSubscription() {
        PayPalSetupTokenResponse payPalSetupTokenResponse = createPayPalSetupTokenResponseData();
        when(cartFacade.getSessionCart()).thenReturn(null);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getEntries()).thenReturn(List.of(orderEntryModel)).thenReturn(List.of(orderEntryModel));
        when(customerModel.getPaymentInfos()).thenReturn(List.of(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(true);
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(EMAIL_ADDRESS);
        doNothing().when(customerAccountService).deleteCCPaymentInfo(customerModel, payPalPaymentInfoModel);
        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(new CCPaymentInfoData()));
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        doNothing().when(payPalAddressDataPopulator).populate(any(PayPalAddressDetailsData.class), any(AddressData.class));
        doNothing().when(addressReversePopulator).populate(any(AddressData.class), any(AddressModel.class));
        when(payPalCustomerAccountService.createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any(PayPalCreditCardPaymentInfoModel.class)))
                .thenReturn(secondPayPalCreditCardPaymentInfoModel);

        unit.createPayPalPaymentSubscription(payPalSetupTokenResponse, true, firstPayPalCreditCardPaymentInfoModel, customerModel);

        verify(payPalCustomerAccountService).removeDuplicatePaymentMethod(EMAIL_ADDRESS);
        verify(userFacade).getCCPaymentInfos(true);
        verify(payPalAddressDataPopulator).populate(any(PayPalAddressDetailsData.class), any(AddressData.class));
        verify(addressReversePopulator).populate(any(AddressData.class), any(AddressModel.class));
        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any(PayPalCreditCardPaymentInfoModel.class));
    }

    @Test
    public void shouldCreatePayPalPaymentSubscriptionWhenNotSavePaymentInfo() {
        PayPalSetupTokenResponse payPalSetupTokenResponse = createPayPalSetupTokenResponseData();
        when(cartFacade.getSessionCart()).thenReturn(null);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getEntries()).thenReturn(List.of(orderEntryModel)).thenReturn(List.of(orderEntryModel));
        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(new CCPaymentInfoData()));
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        doNothing().when(payPalAddressDataPopulator).populate(any(PayPalAddressDetailsData.class), any(AddressData.class));
        doNothing().when(addressReversePopulator).populate(any(AddressData.class), any(AddressModel.class));
        when(payPalCustomerAccountService.createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any(PayPalCreditCardPaymentInfoModel.class)))
                .thenReturn(secondPayPalCreditCardPaymentInfoModel);

        unit.createPayPalPaymentSubscription(payPalSetupTokenResponse, false, firstPayPalCreditCardPaymentInfoModel, customerModel);

        verify(customerAccountService, times(0)).deleteCCPaymentInfo(customerModel, payPalPaymentInfoModel);
        verify(userFacade).getCCPaymentInfos(true);
        verify(payPalAddressDataPopulator).populate(any(PayPalAddressDetailsData.class), any(AddressData.class));
        verify(addressReversePopulator).populate(any(AddressData.class), any(AddressModel.class));
        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any(PayPalCreditCardPaymentInfoModel.class));
    }

    @Test
    public void shouldReturnApplePayLineItemsWhenAllPricesPresent() {
        CartData cartData = new CartData();
        PriceData priceData = new PriceData();
        priceData.setValue(PRICE_VALUE);

        cartData.setTotalPriceWithTax(priceData);
        cartData.setSubTotal(priceData);
        cartData.setDeliveryCost(priceData);
        cartData.setTotalTax(priceData);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(baseStoreModel.getName()).thenReturn(NAME);

        List<ApplePayLineItem> applePayLineItemsFromCart = unit.getApplePayLineItemsFromCart();
        applePayLineItemsFromCart.forEach(applePayLineItem -> {
            assertEquals(PRICE_VALUE_STRING, applePayLineItem.getAmount());
            assertNotNull(applePayLineItem.getLabel());
            assertNotNull(applePayLineItem.getType());
        });
    }

    @Test
    public void shouldReturnApplePayLineItemsWithZeroAmountsWhenAllPricesAreNull() {
        CartData cartData = new CartData();
        cartData.setTotalPriceWithTax(null);
        cartData.setSubTotal(null);
        cartData.setDeliveryCost(null);
        cartData.setTotalTax(null);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(baseStoreModel.getName()).thenReturn(NAME);

        List<ApplePayLineItem> applePayLineItemsFromCart = unit.getApplePayLineItemsFromCart();
        applePayLineItemsFromCart.forEach(applePayLineItem -> {
            assertEquals(BigDecimal.ZERO.toString(), applePayLineItem.getAmount());
            assertNotNull(applePayLineItem.getLabel());
            assertNotNull(applePayLineItem.getType());
        });
    }

    @Test
    public void shouldCreatePaymentSubscriptionWhenFullNameAndAddressIsNull() {
        PayPalSetupTokenResponse payPalSetupTokenResponse = createEmptyAddressPayPalSetupTokenResponseData();
        when(cartFacade.getSessionCart()).thenReturn(null);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getEntries()).thenReturn(List.of(orderEntryModel)).thenReturn(List.of(orderEntryModel));
        when(userFacade.getCCPaymentInfos(true)).thenReturn(List.of(new CCPaymentInfoData()));
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);

        unit.createPayPalPaymentSubscription(payPalSetupTokenResponse, false, firstPayPalCreditCardPaymentInfoModel, customerModel);

        verify(payPalCustomerAccountService).createPayPalPaymentSubscription(any(CustomerModel.class),
                any(CCPaymentInfoData.class), any(AddressModel.class), any(PayPalCreditCardPaymentInfoModel.class));
    }

    private PayPalSetupTokenResponse createPayPalSetupTokenResponseData() {
        PayPalCustomerData customer = new PayPalCustomerData();
        customer.setId(VAULT_CUSTOMER_ID);
        PayPalSetupTokenResponse payPalSetupTokenResponse = new PayPalSetupTokenResponse();
        payPalSetupTokenResponse.setId(ID_VALUE);
        payPalSetupTokenResponse.setCustomer(customer);
        PayPalPaymentSource paymentSource = new PayPalPaymentSource();
        PayPalData payPalData = new PayPalData();
        payPalData.setEmailAddress(EMAIL_ADDRESS);
        payPalData.setPayerId(PAYER_ID);
        NameData nameData = new NameData();
        nameData.setFullName(FULL_NAME);
        payPalData.setName(nameData);
        payPalData.setAddress(createAddressPortableData());
        paymentSource.setPaypal(payPalData);
        payPalSetupTokenResponse.setPaymentSource(paymentSource);
        return payPalSetupTokenResponse;
    }

    private PayPalSetupTokenResponse createEmptyAddressPayPalSetupTokenResponseData() {
        PayPalCustomerData customer = new PayPalCustomerData();
        customer.setId(VAULT_CUSTOMER_ID);
        PayPalSetupTokenResponse payPalSetupTokenResponse = new PayPalSetupTokenResponse();
        payPalSetupTokenResponse.setId(ID_VALUE);
        payPalSetupTokenResponse.setCustomer(customer);
        PayPalPaymentSource paymentSource = new PayPalPaymentSource();
        PayPalData payPalData = new PayPalData();
        payPalData.setEmailAddress(EMAIL_ADDRESS);
        payPalData.setPayerId(PAYER_ID);
        paymentSource.setPaypal(payPalData);
        payPalSetupTokenResponse.setPaymentSource(paymentSource);
        return payPalSetupTokenResponse;
    }

    private AddressPortableData createAddressPortableData() {
        AddressPortableData addressPortableData = new AddressPortableData();
        addressPortableData.setAddressLine1(LINE_1);
        addressPortableData.setAddressLine2(LINE_2);
        addressPortableData.setPostalCode(POSTAL_CODE);
        addressPortableData.setCountryCode(COUNTRY_CODE);
        addressPortableData.setAdminArea1(REGION);
        addressPortableData.setAdminArea2(CITY);
        return addressPortableData;
    }

    private AddressData createAddressData() {
        CountryData countryData = new CountryData();
        countryData.setIsocode(ISO_CODE);
        return GenericBuilder.of(AddressData::new)
                .with(AddressData::setTown, TOWN)
                .with(AddressData::setCountry, countryData)
                .with(AddressData::setLine1, LINE_1)
                .with(AddressData::setFirstName, FIRST_NAME)
                .with(AddressData::setLastName, LAST_NAME)
                .with(AddressData::setEmail, EMAIL_ADDRESS)
                .build();
    }

    @Test
    public void shouldCreateApplePaySessionDetailsForExpressCheckout() {
        CartData cartData = new CartData();
        PriceData priceData = new PriceData();
        CustomerData customerData = new CustomerData();
        AddressData addressData = createAddressData();

        priceData.setValue(PRICE_VALUE);
        priceData.setCurrencyIso(USD);
        cartData.setTotalPrice(priceData);
        customerData.setDefaultShippingAddress(addressData);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(baseStoreModel.getName()).thenReturn(NAME);
        when(commonI18NService.getCurrentLanguage()).thenReturn(languageModel);
        when(languageModel.getIsocode()).thenReturn(US_ISO_CODE);
        when(customerFacade.getCurrentCustomer()).thenReturn(customerData);

        ApplePaySessionDetailsData sessionDetails = unit.createApplePaySessionDetails(EXPRESS_CHECKOUT);

        assertEquals(USD, sessionDetails.getCurrency());
        assertEquals(US_ISO_CODE, sessionDetails.getLocale());
        assertNotNull(sessionDetails.getLineItems());

        verify(applePayAddressDataReversePopulator).populate(eq(addressData), any());
    }

    @Test
    public void shouldCreateApplePaySessionDetailsForMarkCheckout() {
        CartData cartData = new CartData();
        PriceData priceData = new PriceData();
        AddressData addressData = createAddressData();

        priceData.setValue(PRICE_VALUE);
        priceData.setCurrencyIso(USD);
        cartData.setTotalPrice(priceData);
        cartData.setDeliveryAddress(addressData);

        when(cartFacade.getSessionCart()).thenReturn(cartData);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(baseStoreModel.getName()).thenReturn(NAME);
        when(commonI18NService.getCurrentLanguage()).thenReturn(languageModel);
        when(languageModel.getIsocode()).thenReturn(US_ISO_CODE);

        ApplePaySessionDetailsData sessionDetails = unit.createApplePaySessionDetails(MARK_CHECKOUT);

        assertEquals(USD, sessionDetails.getCurrency());
        assertEquals(US_ISO_CODE, sessionDetails.getLocale());
        assertNotNull(sessionDetails.getLineItems());

        verify(applePayAddressDataReversePopulator).populate(eq(addressData), any());
    }

    @Test
    public void shouldCreatePayPalPaymentSubscriptionWithSomeEmptyFieldsForBillingAddress() {
        doReturn(cartData).when(unit).getCheckoutCart();
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);

        unit.createPayPalPaymentSubscription(createPayPalSetupTokenResponseData(),
                false, payPalPaymentInfoModel, customerModel);

        verify(payPalAddressDataPopulator).populate(addressDetailsDataCaptor.capture(), any(AddressData.class));

        PayPalAddressDetailsData addressDetailsData = addressDetailsDataCaptor.getValue();
        assertEquals(LINE_1, addressDetailsData.getLine1());
        assertEquals(POSTAL_CODE, addressDetailsData.getPostalCode());
        assertEquals(COUNTRY_CODE, addressDetailsData.getCountryCode());
        assertNull(addressDetailsData.getLine2());
        assertNull(addressDetailsData.getCity());
        assertNull(addressDetailsData.getRegion());

    }

    @Test
    public void shouldCreateBasicPayPalVaultOrderRequestDataFromCartData() {
        PayPalVaultOrderRequestData expected = createOrderRequestData();
        when(vaultOrderRequestDataConverter.convert(cartModel)).thenReturn(expected);

        PayPalVaultOrderRequestData result = unit.createBasicPayPalVaultOrderRequestData();

        assertThat(result).isEqualToComparingFieldByFieldRecursively(expected);
    }

    @Test
    public void shouldUpdateOrderAmountDetailsWhenOrderAmountDoesNotMatchCartAmount() {
        PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();
        orderDetailsData.setAmount(PRICE.toString());
        PayPalOrderRequestData orderRequestData = new PayPalOrderRequestData();
        when(orderRequestDataConverter.convert(cartModel)).thenReturn(orderRequestData);

        unit.updatePayPalOrder(orderDetailsData);

        verify(payPalPaymentService).updateOrderAmountDetails(orderRequestData);
    }

    @Test
    public void shouldNotUpdateOrderAmountDetailsWhenOrderAmountMatchCartAmount() {
        PayPalOrderDetailsData orderDetailsData = new PayPalOrderDetailsData();
        orderDetailsData.setAmount(PRICE_VALUE_STRING);
        when(cartModel.getTotalPrice()).thenReturn(MATCHING_PRICE);
        when(breakdownCalculationService.calculateTotalAmount(cartModel)).thenReturn(BigDecimal.valueOf(MATCHING_PRICE));

        unit.updatePayPalOrder(orderDetailsData);

        verify(payPalPaymentService, never()).updateOrderAmountDetails(any());
    }

    @Test
    public void shouldUpdateApplePayShippingAddressWithoutId() {
        doReturn(null).when(unit).getAddressDataForId(null, false);

        unit.updateApplePayShippingAddress(applePayAddressDetailsData);

        verify(applePayAddressDataConverter).convert(applePayAddressDetailsData);
        verify(unit).setDeliveryAddress(addressData);
    }

    @Test
    public void shouldUpdateApplePayShippingAddressWithoutIdForAnonymousUser() {
        doReturn(null).when(unit).getAddressDataForId(null, false);
        when(userFacade.isAnonymousUser()).thenReturn(true);

        unit.updateApplePayShippingAddress(applePayAddressDetailsData);

        verify(applePayAddressDataConverter).convert(applePayAddressDetailsData);
        verify(userFacade, never()).addAddress(any());
        verify(unit).setDeliveryAddress(any(AddressData.class));
    }

    @Test
    public void shouldUpdateApplePayShippingAddressWithId() {
        ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
        AddressData addressData = new AddressData();
        applePayAddressDetailsData.setId(ID_VALUE);

        doReturn(addressData).when(unit).getAddressDataForId(ID_VALUE, false);

        unit.updateApplePayShippingAddress(applePayAddressDetailsData);

        verify(unit).setDeliveryAddress(addressData);
    }

    private PayPalVaultOrderRequestData createOrderRequestData(){
        return GenericBuilder.of(PayPalVaultOrderRequestData::new)
                .with(PayPalOrderRequestData::setAmount, PRICE_VALUE_STRING)
                .with(PayPalOrderRequestData::setCurrency, USD)
                .with(PayPalOrderRequestData::setSaveOrderFlowActive, false)
                .with(PayPalOrderRequestData::setIntent, INTENT)
                .with(PayPalOrderRequestData::setShippingAmount, PRICE_VALUE_STRING)
                .with(PayPalVaultOrderRequestData::setUsageType, USAGE_TYPE)
                .with(PayPalVaultOrderRequestData::setCustomerType, CUSTOMER_TYPE)
                .build();
    }
}
