package com.paypal.hybris.facade.strategy.payment.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.core.enums.ThreeDsVerificationOnMyAccountType;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.ExperienceContextData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalPaymentSource;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.facade.strategy.payment.PaymentStrategy;
import de.hybris.platform.acceleratorservices.urlresolver.SiteBaseUrlResolutionService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.util.Strings;
import org.springframework.util.Assert;

import java.util.Locale;
import java.util.Optional;

public class DefaultCreditCardPopulatorStrategy implements PaymentStrategy {

    private static final String MY_ACCOUNT_PAYMENT_DETAILS_APPROVE_LINK = "/my-account/payment-details/approved?guid=%s";
    private static final String MY_ACCOUNT_PAYMENT_DETAILS_CANCEL_LINK = "/my-account/payment-details/cancelled?guid=%s";

    private UserService userService;
    private PayPalConfigurationService payPalConfigurationService;
    private BaseSiteService baseSiteService;
    private SiteBaseUrlResolutionService siteBaseUrlResolutionService;

    @Override
    public void populate(SetupTokenRequestData source, PayPalSavePaymentForPurchaseLaterRequest target) throws ConversionException {
        CardData cardData = new CardData();
        CreditCardAddressData creditCardData = source.getCreditCardData();

        populateCardData(creditCardData, cardData);

        AddressData billingAddress = creditCardData.getBillingAddress();
        AddressPortableData address = new AddressPortableData();

        populateBillingAddress(billingAddress, address);

        cardData.setBillingAddress(address);

        String payPalVerifyCCOnMyAccount = payPalConfigurationService.getPayPalVerifyCCOnMyAccount();
        if (!payPalVerifyCCOnMyAccount.equalsIgnoreCase(ThreeDsVerificationOnMyAccountType.NO_VERIFICATION.getCode())) {
            cardData.setVerificationMethod(payPalVerifyCCOnMyAccount.toUpperCase(Locale.US));

            ExperienceContextData experienceContext = new ExperienceContextData();

            BaseSiteModel currentBaseSite = baseSiteService.getCurrentBaseSite();
            experienceContext.setReturnUrl(siteBaseUrlResolutionService.getWebsiteUrlForSite(currentBaseSite, true,
                    String.format(MY_ACCOUNT_PAYMENT_DETAILS_APPROVE_LINK, source.getPaymentMethodCode())));
            experienceContext.setCancelUrl(siteBaseUrlResolutionService.getWebsiteUrlForSite(currentBaseSite, true,
                    String.format(MY_ACCOUNT_PAYMENT_DETAILS_CANCEL_LINK, source.getPaymentMethodCode())));

            cardData.setExperienceContext(experienceContext);
        }

        target.setCustomer(getCustomerData());
        target.setPaymentSource(getPaymentSource(cardData));

    }

    private PayPalPaymentSource getPaymentSource(CardData cardData) {
        PayPalPaymentSource paymentSource = new PayPalPaymentSource();
        paymentSource.setCard(cardData);
        return paymentSource;
    }

    private PayPalCustomerData getCustomerData() {
        PayPalCustomerData customerData = new PayPalCustomerData();
        setCustomerId(customerData);
        return customerData;
    }

    private void setCustomerId(PayPalCustomerData customerData) {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        if (StringUtils.isNotEmpty(customerModel.getVaultCustomerId())) {
            customerData.setId(customerModel.getVaultCustomerId());
        }
    }

    private void populateCardData(CreditCardAddressData creditCardAddressData, CardData cardData) {
        Assert.notNull(creditCardAddressData, "CreditCardData cannot be null.");

        cardData.setName(creditCardAddressData.getNameOnCard());
        cardData.setExpiry(creditCardAddressData.getExpiryYear() + "-" + creditCardAddressData.getExpiryMonth());
        cardData.setNumber(creditCardAddressData.getCardNumber());
        cardData.setCvv(creditCardAddressData.getCvv());
    }

    private void populateBillingAddress(AddressData billingAddress, AddressPortableData address) {
        Assert.notNull(billingAddress, "BillingAddress cannot be null.");
        Assert.notNull(billingAddress.getLine1(), "Line1 of billingAddress cannot be null.");

        address.setAddressLine1(billingAddress.getLine1());
        address.setAddressLine2(billingAddress.getLine2());
        address.setAdminArea1(Optional.of(billingAddress)
                .map(AddressData::getRegion)
                .map(RegionData::getIsocode)
                .orElse(Strings.EMPTY));
        address.setAdminArea2(billingAddress.getTown());
        address.setPostalCode(billingAddress.getPostalCode());
        address.setCountryCode(Optional.of(billingAddress)
                .map(AddressData::getCountry)
                .map(CountryData::getIsocode)
                .orElse(Strings.EMPTY));
    }

    @Override
    public boolean test(PayPalPaymentProvider paymentProvider) {
        return PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS.equals(paymentProvider);
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setPayPalConfigurationService(PayPalConfigurationService payPalConfigurationService) {
        this.payPalConfigurationService = payPalConfigurationService;
    }

    public void setBaseSiteService(BaseSiteService baseSiteService) {
        this.baseSiteService = baseSiteService;
    }

    public void setSiteBaseUrlResolutionService(SiteBaseUrlResolutionService siteBaseUrlResolutionService) {
        this.siteBaseUrlResolutionService = siteBaseUrlResolutionService;
    }

}
