package com.paypal.hybris.facade.service.impl;

import com.google.common.math.DoubleMath;
import com.paypal.hybris.core.service.PayPalCommerceCheckoutService;
import com.paypal.hybris.facade.service.BreakdownAmountCalculationService;
import de.hybris.platform.commerceservices.constants.CommerceServicesConstants;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.util.DiscountValue;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;


public class BreakdownCalculationService implements BreakdownAmountCalculationService {

	private static final double EPSILON = 0.01d;
	private PayPalCommerceCheckoutService commerceCheckoutService;

	@Override
	public BigDecimal calculateTotalDiscount(AbstractOrderModel orderModel) {
		double productsDiscountsAmount = getProductsDiscountsAmount(orderModel);
		double orderDiscountsAmount = getOrderDiscountsAmount(orderModel);
		return BigDecimal.valueOf(productsDiscountsAmount + orderDiscountsAmount);
	}

	@Override
	public BigDecimal calculateTotalAmount(AbstractOrderModel abstractOrderModel) {
		return commerceCheckoutService.calculateCartTotalWithTax(abstractOrderModel);
	}

	@Override
	public BigDecimal calculateItemsAmount(AbstractOrderModel abstractOrderModel) {
		return BigDecimal.valueOf(abstractOrderModel.getEntries().stream()
				.map(entry -> entry.getBasePrice() * entry.getQuantity())
				.reduce(0d, Double::sum)).setScale(2, RoundingMode.HALF_UP);
	}

	protected double getProductsDiscountsAmount(AbstractOrderModel source) {
		double discounts = 0.0d;
		List<AbstractOrderEntryModel> entries = source.getEntries();
		if (entries != null) {
			for (AbstractOrderEntryModel entry : entries) {
				List<DiscountValue> discountValues = entry.getDiscountValues();
				if (discountValues != null) {
					for (DiscountValue dValue : discountValues) {
						discounts += dValue.getAppliedValue();
					}
				}
			}
		}
		return discounts;
	}


	protected double getOrderDiscountsAmount(AbstractOrderModel source) {
		double discounts = 0.0d;
		List<DiscountValue> discountList = source.getGlobalDiscountValues();
		if (discountList != null) {
			for (DiscountValue discount : discountList) {
				double value = discount.getAppliedValue();
				if (DoubleMath.fuzzyCompare(value, 0, EPSILON) > 0
						&& !CommerceServicesConstants.QUOTE_DISCOUNT_CODE.equals(discount.getCode())) {
					discounts += value;
				}
			}
		}
		return discounts;
	}

	public void setCommerceCheckoutService(PayPalCommerceCheckoutService commerceCheckoutService) {
		this.commerceCheckoutService = commerceCheckoutService;
	}
}
