package com.paypal.hybris.facade.populator;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.util.ExpiredDateValidatorUtil;
import de.hybris.platform.commercefacades.order.converters.populator.CreditCardPaymentInfoPopulator;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.util.Strings;

import java.util.Objects;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER;


public class PayPalCreditCardPaymentInfoPopulator extends CreditCardPaymentInfoPopulator {

    private ExpiredDateValidatorUtil expiredDateValidatorUtil;

    private PayPalConfigurationService defaultPayPalConfigurationService;

    @Override
    public void populate(final CreditCardPaymentInfoModel source,
        final CCPaymentInfoData target) throws ConversionException {
        super.populate(source, target);
        if (source instanceof PayPalCreditCardPaymentInfoModel) {
            PayPalCreditCardPaymentInfoModel paymentInfoModelSource = (PayPalCreditCardPaymentInfoModel) source;
            target.setPayerEmail(paymentInfoModelSource.getPayerEmail());
            target.setPayerId(paymentInfoModelSource.getPayerId());
            target.setPayPalOrderId(paymentInfoModelSource.getPayPalOrderId());
            target.setPaymentProvider(Optional.ofNullable(paymentInfoModelSource.getPaymentProvider())
                    .map(PaymentProvider::getCode)
                    .map(PayPalPaymentProvider::valueOf).orElse(null));
            if (StringUtils.isEmpty(paymentInfoModelSource.getSubscriptionId())) {
                target.setSubscriptionId(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);
            }
            target.setExpirationStatus(paymentInfoModelSource.getExpirationStatus().getCode());
            if (paymentInfoModelSource.getExpirationStatus().equals(ExpirationStatus.EXPIRE_SOON)) {
                target.setRemainderMonthToExpire(expiredDateValidatorUtil.getMonthsToExpiration(
                        paymentInfoModelSource.getValidToMonth(), paymentInfoModelSource.getValidToYear()) + 1);
            }
            target.setOriginalId(Optional.ofNullable(source.getOriginal()).map(originalPayment -> originalPayment.getPk().toString()).orElse(target.getId()));
            target.setIsEnable(isPaymentEnable(paymentInfoModelSource));
            target.setShouldBeSaved(paymentInfoModelSource.isShouldBeSaved());

            // the check should be removed after we start to receive again fields below from the PayPal API
            PaymentProvider paymentProvider = paymentInfoModelSource.getPaymentProvider();
            if (! Objects.isNull(paymentProvider) && paymentProvider.equals(PaymentProvider.PAYPAL)) {
                target.getBillingAddress().setFirstName(Strings.EMPTY);
                target.getBillingAddress().setLastName(Strings.EMPTY);
                target.getBillingAddress().setTown(Strings.EMPTY);
                target.getBillingAddress().setLine2(Strings.EMPTY);
                target.getBillingAddress().setRegion(null);
            }
        }
    }


    private boolean isPaymentEnable(PayPalCreditCardPaymentInfoModel paymentInfoModel) {
        PaymentProvider paymentProvider = paymentInfoModel.getPaymentProvider();
        if (Objects.isNull(paymentProvider)) {
            return false;
        }
        if (paymentProvider.equals(PaymentProvider.PAYPAL)) {
            return defaultPayPalConfigurationService.isPayPalVaultEnabled() && (StringUtils.isNotBlank(paymentInfoModel.getSubscriptionId()) &&
                    !PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER.equals(paymentInfoModel.getSubscriptionId()));
        } else if (paymentProvider.equals(PaymentProvider.PAYPAL_HOSTED_FIELDS)) {
            return defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled();
        } else {
            return true;
        }
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setExpiredDateValidatorUtil(ExpiredDateValidatorUtil expiredDateValidatorUtil) {
        this.expiredDateValidatorUtil = expiredDateValidatorUtil;
    }

}
