package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.AuthenticationResult;
import com.paypal.hybris.data.AuthenticationResultData;
import com.paypal.hybris.data.BreakdownData;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.NameData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayerData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.PurchaseUnitData;
import com.paypal.hybris.data.ThreeDSecure;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import org.apache.commons.lang.StringUtils;

import java.util.Optional;

import static com.paypal.hybris.core.util.PayPalCommandsUtil.getPriceValue;

public class OrderResponseToOrderDetailsPopulator implements Populator<PayPalOrderResponseData, PayPalOrderDetailsData> {

	private static final String NAME_PLACEHOLDER = "%s %s";

	@Override
	public void populate(PayPalOrderResponseData responseData, PayPalOrderDetailsData payPalOrderDetailsData) throws ConversionException {
		BreakdownData breakdownData = responseData.getPurchaseUnits().get(0).getAmount().getBreakdown();


		payPalOrderDetailsData.setOrderId(responseData.getId());
		payPalOrderDetailsData.setBuyerId(responseData.getPayer() != null ? responseData.getPayer().getPayerId() : StringUtils.EMPTY);
		payPalOrderDetailsData.setBuyer(getBuyerName(responseData));
		payPalOrderDetailsData.setAmount(responseData.getPurchaseUnits().get(0).getAmount().getValue());
		payPalOrderDetailsData.setShippingAddress(convertPayPalShippingAddress(responseData));
		payPalOrderDetailsData.setBillingAddress(convertPayPalBillingAddress(responseData.getPayer()));
		payPalOrderDetailsData.setPayment(responseData.getPaymentSource());
		payPalOrderDetailsData.setAuthenticationResult(convertAuthenticationData(responseData));
		payPalOrderDetailsData.setItems(responseData.getPurchaseUnits().get(0).getItems());
		payPalOrderDetailsData.setItemsTotal(breakdownData.getItemTotal().getValue());
		payPalOrderDetailsData.setDiscount(getPriceValue(breakdownData.getDiscount()));
		payPalOrderDetailsData.setTax(getPriceValue(breakdownData.getTax()));
		payPalOrderDetailsData.setShipping(getPriceValue(breakdownData.getShipping()));
		payPalOrderDetailsData.setCurrency(responseData.getPurchaseUnits().get(0).getAmount().getCurrencyCode());
	}

	protected PayPalAddressDetailsData convertPayPalShippingAddress(PayPalOrderResponseData payPalOrder) {
		PurchaseUnitData purchaseUnit = payPalOrder.getPurchaseUnits().get(0);
		if (purchaseUnit.getShipping() == null) {
			return null;
		}

		String email = payPalOrder.getPayer() != null ? payPalOrder.getPayer().getEmail() : StringUtils.EMPTY;
		String fullName = purchaseUnit.getShipping().getName().getFullName();
		String[] fullNameSplit = fullName.split("\\s+", 2);
		String lastName = (fullNameSplit.length > 1) ? fullNameSplit[1] : StringUtils.EMPTY;

		PayPalAddressDetailsData addressDetailsData = populateBasicAddressData(purchaseUnit.getShipping().getAddress());
		populatePayerData(addressDetailsData, fullNameSplit[0], lastName, email);

		return addressDetailsData;
	}

	protected PayPalAddressDetailsData populateBasicAddressData(AddressPortableData address) {
		return GenericBuilder.of(PayPalAddressDetailsData::new)
				.with(PayPalAddressDetailsData::setCity, address.getAdminArea2())
				.with(PayPalAddressDetailsData::setRegion, address.getAdminArea1())
				.with(PayPalAddressDetailsData::setCountryCode, address.getCountryCode())
				.with(PayPalAddressDetailsData::setPostalCode, address.getPostalCode())
				.with(PayPalAddressDetailsData::setLine1, address.getAddressLine1())
				.with(PayPalAddressDetailsData::setLine2, address.getAddressLine2())
				.build();
	}

	private void populatePayerData(PayPalAddressDetailsData addressDetailsData, String name, String surname, String email) {
		addressDetailsData.setEmail(email);
		addressDetailsData.setFirstName(name);
		addressDetailsData.setLastName(surname);
	}

	private String getBuyerName(PayPalOrderResponseData payPalOrder) {
		if (payPalOrder.getPayer() == null) {
			return StringUtils.EMPTY;
		}
		NameData buyerName = payPalOrder.getPayer().getName();
		return NAME_PLACEHOLDER.formatted(buyerName.getGivenName(), buyerName.getSurname());
	}

	protected PayPalAddressDetailsData convertPayPalBillingAddress(PayerData payerData) {
		if (payerData == null) {
			return null;
		}
		PayPalAddressDetailsData addressDetailsData = populateBasicAddressData(payerData.getAddress());
		populatePayerData(addressDetailsData, payerData.getName().getSurname(), payerData.getName().getGivenName(),
				payerData.getEmail());

		return addressDetailsData;
	}

	protected AuthenticationResultData convertAuthenticationData(PayPalOrderResponseData palOrderResponseData) {
		AuthenticationResult authenticationResult = Optional.of(palOrderResponseData)
				.map(PayPalOrderResponseData::getPaymentSource)
				.map(PaymentSourceData::getCard)
				.map(CardData::getAuthenticationResult).orElse(null);

		if (authenticationResult == null) {
			return null;
		}

		final String liabilityShift = authenticationResult.getLiabilityShift();
		final String enrollmentStatus = authenticationResult.getThreeDSecure().getEnrollmentStatus();
		final String authenticationStatus = authenticationResult.getThreeDSecure().getAuthenticationStatus();

		return GenericBuilder.of(AuthenticationResultData::new)
				.with(AuthenticationResultData::setLiabilityShift, liabilityShift)
				.with(AuthenticationResultData::setEnrollmentStatus, enrollmentStatus)
				.with(AuthenticationResultData::setAuthenticationStatus, authenticationStatus)
				.build();
	}

}
