package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.service.impl.DefaultPayPalConfigurationService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.BreakdownData;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.L3ItemData;
import com.paypal.hybris.data.Level2CardData;
import com.paypal.hybris.data.Level2Level3CardData;
import com.paypal.hybris.data.Level3CardData;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.SimpleAmount;
import com.paypal.hybris.data.SupplementaryData;
import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.externaltax.DeliveryFromAddressStrategy;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.servicelayer.dto.converter.Converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;

import static com.paypal.hybris.core.util.PayPalCommandsUtil.createAmount;

public class OrderRequestDataPopulator implements Populator<AbstractOrderModel, PayPalOrderRequestData> {

	private DefaultPayPalConfigurationService defaultPayPalConfigurationService;
	private Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter;
	private Converter<AddressData, AddressPortableData> payPalAddressPortableDataConverter;
	private Converter<AbstractOrderEntryModel, L3ItemData> l3ItemDataDataConverter;
	private Converter<AddressModel, AddressData> addressConverter;
	private BreakdownCalculationService breakdownCalculationService;
	private PayPalConfigurationService payPalConfigurationService;
	private DeliveryFromAddressStrategy deliveryFromAddressStrategy;

	@Override
	public void populate(AbstractOrderModel source, PayPalOrderRequestData target) throws ConversionException {

		target.setAmount(getIntegerPriceValue(source));
		target.setCurrency(source.getCurrency().getIsocode());
		if (source instanceof CartModel cartModel) {
			target.setOrderId(cartModel.getPayPalOrderId());
		}
		target.setOrderItems(convertItems(source.getEntries()));
		target.setSupplementaryData(createSupplementaryData(source));
		target.setBreakdownAmountData(createAmountBreakdown(source));
	}

	private String getIntegerPriceValue(AbstractOrderModel source) {
		BigDecimal totalAmount = breakdownCalculationService.calculateTotalAmount(source);
		return isNonDecimalCurrency(source) ? String.valueOf(totalAmount.toBigInteger()) : String.valueOf(totalAmount);
	}

	private Boolean isNonDecimalCurrency(AbstractOrderModel cartModel) {
		return defaultPayPalConfigurationService.getNonDecimalCurrency().contains(cartModel.getCurrency().getIsocode());
	}

	private List<OrderItemData> convertItems(List<AbstractOrderEntryModel> entries) {
		return entries.stream()
				.map(entry -> orderItemDataConverter.convert(entry))
				.toList();
	}

	protected SupplementaryData createSupplementaryData(AbstractOrderModel source) {
		if (payPalConfigurationService.isL2L3DataEnabled()) {
			Level2Level3CardData l2L3data = new Level2Level3CardData();
			l2L3data.setLevel2(createL2Data(source));
			l2L3data.setLevel3(createL3Data(source));

			return GenericBuilder.of(SupplementaryData::new)
					.with(SupplementaryData::setCard, l2L3data)
					.build();
		}
		return null;
	}

	private Level3CardData createL3Data(AbstractOrderModel source) {
		String currency = source.getCurrency().getIsocode();

		return GenericBuilder.of(Level3CardData::new)
				.with(Level3CardData::setShippingAmount, createAmount(source.getDeliveryCost(), currency))
				.with(Level3CardData::setDiscountAmount, createAmount(breakdownCalculationService.calculateTotalDiscount(source), currency))
				.with(Level3CardData::setShipsFromPostalCode, deliveryFromAddressStrategy.getDeliveryFromAddressForOrder(source).getPostalcode())
				.with(Level3CardData::setLineItems, createItemsForL3Data(source))
				.with(Level3CardData::setShippingAddress, getAddressData(source))
				.build();
	}

	private AmountWithBreakdownData createAmountBreakdown(AbstractOrderModel orderModel) {
		return GenericBuilder.of(AmountWithBreakdownData::new)
				.with(AmountWithBreakdownData::setValue, String.valueOf(breakdownCalculationService.calculateTotalAmount(orderModel)))
				.with(AmountWithBreakdownData::setCurrencyCode, orderModel.getCurrency().getIsocode())
				.with(AmountWithBreakdownData::setBreakdown, createBreakdownData(orderModel))
				.build();
	}

	private BreakdownData createBreakdownData(AbstractOrderModel orderModel) {
		String currencyCode = orderModel.getCurrency().getIsocode();
		SimpleAmount itemsTotal = createAmount(breakdownCalculationService.calculateItemsAmount(orderModel), currencyCode);
		SimpleAmount shipping = createAmount(orderModel.getDeliveryCost(), currencyCode);
		SimpleAmount tax = createAmount(breakdownCalculationService.calculateTaxAmount(orderModel), currencyCode);
		SimpleAmount discount = createAmount(breakdownCalculationService.calculateTotalDiscount(orderModel), currencyCode);
		return GenericBuilder.of(BreakdownData::new)
				.with(BreakdownData::setItemTotal, itemsTotal)
				.with(BreakdownData::setShipping, shipping)
				.with(BreakdownData::setTax, tax)
				.with(BreakdownData::setDiscount, discount)
				.build();
	}

	protected AddressPortableData getAddressData(AbstractOrderModel source) {
		return Optional.ofNullable(source.getDeliveryAddress())
				.map(addressModel ->  addressConverter.convert(addressModel))
				.map(addressData -> payPalAddressPortableDataConverter.convert(addressData))
				.orElse(null);
	}

	private Level2CardData createL2Data(AbstractOrderModel source) {
		String currency = source.getCurrency().getIsocode();

		return GenericBuilder.of(Level2CardData::new)
//				.with(Level2CardData::setInvoiceId, source.getCode()) //unique purchase identifier
				.with(Level2CardData::setTaxTotal, createAmount(breakdownCalculationService.calculateTaxAmount(source), currency))
				.build();
	}

	private List<L3ItemData> createItemsForL3Data(AbstractOrderModel source) {
		return source.getEntries().stream()
				.map(entry -> l3ItemDataDataConverter.convert(entry))
				.toList();
	}

	public void setOrderItemDataConverter(Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter) {
		this.orderItemDataConverter = orderItemDataConverter;
	}

	public void setPayPalAddressPortableDataConverter(Converter<AddressData, AddressPortableData> payPalAddressPortableDataConverter) {
		this.payPalAddressPortableDataConverter = payPalAddressPortableDataConverter;
	}

	public void setL3ItemDataDataConverter(Converter<AbstractOrderEntryModel, L3ItemData> l3ItemDataDataConverter) {
		this.l3ItemDataDataConverter = l3ItemDataDataConverter;
	}

	public void setAddressConverter(Converter<AddressModel, AddressData> addressConverter) {
		this.addressConverter = addressConverter;
	}

	public void setPayPalConfigurationService(PayPalConfigurationService payPalConfigurationService) {
		this.payPalConfigurationService = payPalConfigurationService;
	}

	public void setDeliveryFromAddressStrategy(DeliveryFromAddressStrategy deliveryFromAddressStrategy) {
		this.deliveryFromAddressStrategy = deliveryFromAddressStrategy;
	}

	public void setDefaultPayPalConfigurationService(DefaultPayPalConfigurationService defaultPayPalConfigurationService) {
		this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
	}

	public void setBreakdownCalculationService(BreakdownCalculationService breakdownCalculationService) {
		this.breakdownCalculationService = breakdownCalculationService;
	}
}
