package com.paypal.hybris.facade.populator;

import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.platform.acceleratorservices.urlresolver.SiteBaseUrlResolutionService;
import de.hybris.platform.commerceservices.url.UrlResolver;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.product.ProductModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.site.BaseSiteService;
import org.apache.commons.lang3.StringUtils;

import java.util.Optional;
import java.util.regex.Pattern;

import static com.paypal.hybris.core.util.PayPalCommandsUtil.createAmount;

public class OrderItemPopulator implements Populator<AbstractOrderEntryModel, OrderItemData> {
	private static final int MAX_DESCRIPTION_LENGTH = 127;
	private static final String COLON_SYMBOL = ":";
	private static final String PAYPAL_PRODUCT_ITEM_TYPE = "PHYSICAL_GOODS";
	private static final String PAYPAL_URL_PATTERN = "^(https:)([/|.|\\w|\\s|-])*\\.(?:jpg|gif|png|jpeg|JPG|GIF|PNG|JPEG)";
	private static final String SUPPRESS_ENCODING_ATTRIBUTES = StringUtils.EMPTY;

	private BaseSiteService baseSiteService;
	private SiteBaseUrlResolutionService siteBaseUrlResolutionService;
	private UrlResolver<ProductModel> productModelUrlResolver;

	@Override
	public void populate(AbstractOrderEntryModel source, OrderItemData target) throws ConversionException {
		String currency = source.getOrder().getCurrency().getIsocode();
		SimpleAmount amount = createAmount(source.getBasePrice(), currency);

		target.setName(source.getProduct().getName());
		target.setQuantity(source.getQuantity().intValue());
		target.setDescription(StringUtils.abbreviate(source.getProduct().getDescription(),
				MAX_DESCRIPTION_LENGTH));
		target.setSku(source.getProduct().getCode());
		target.setUrl(productModelUrlResolver.resolve(source.getProduct()));
		target.setCategory(PAYPAL_PRODUCT_ITEM_TYPE);
		target.setAmount(amount);

        Optional.ofNullable(source.getProduct())
                .map(ProductModel::getThumbnail)
                .map(MediaModel::getPayPalPrettyImgUrl)
                .map(url -> siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteService.getCurrentBaseSite(), SUPPRESS_ENCODING_ATTRIBUTES, true, url))
                .map(this::getValidUrl)
                .filter(this::validateUrl)
                .ifPresent(target::setImageUrl);
    }

    protected boolean validateUrl(String url) {
        return Pattern.compile(PAYPAL_URL_PATTERN).matcher(url).matches();
    }

	/**
	 * According to PayPal pattern for image_url ':' symbol is not allowed,
	 * which might cause request to fail on PayPal side during placing order on local environment.
	 */
	protected String getValidUrl(String url) {
		int index = url.indexOf(COLON_SYMBOL) + 1;
		return url.substring(0, index) + url.substring(index).replace(COLON_SYMBOL, StringUtils.EMPTY);
	}

	public void setBaseSiteService(BaseSiteService baseSiteService) {
		this.baseSiteService = baseSiteService;
	}

	public void setSiteBaseUrlResolutionService(SiteBaseUrlResolutionService siteBaseUrlResolutionService) {
		this.siteBaseUrlResolutionService = siteBaseUrlResolutionService;
	}

	public void setProductModelUrlResolver(UrlResolver<ProductModel> productModelUrlResolver) {
		this.productModelUrlResolver = productModelUrlResolver;
	}
}
