package com.paypal.hybris.facade.populator;

import com.paypal.hybris.core.exception.ApplePayInvalidAddressException;
import com.paypal.hybris.data.ApplePayAddressDetailsData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import org.apache.commons.lang.StringUtils;

import java.util.List;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.APPLE_PAY_REGION_FIELD_CODE;


public class ApplePayAddressDataPopulator implements Populator<ApplePayAddressDetailsData, AddressData> {

    private static final String REGION_ISO_CODE = "%s-%s";
    private static final String APPLE_PAY_INVALID_ADDRESS = "apple.pay.invalid.address.error";
    private CommonI18NService commonI18NService;

    @Override
    public void populate(ApplePayAddressDetailsData source, AddressData target) throws ConversionException {
        CountryData countryData = new CountryData();
        Optional<List<String>> addressLines = Optional.ofNullable(source.getAddressLines());
        countryData.setIsocode(source.getCountryCode());
        populateAdministrativeArea(source, target);
        target.setId(source.getId());
        target.setTown(source.getLocality());
        target.setCountry(countryData);
        target.setLine1(addressLines.orElse(List.of(StringUtils.EMPTY)).get(0));
        target.setPostalCode(source.getPostalCode());
        target.setFirstName(source.getGivenName());
        target.setLastName(source.getFamilyName());
        target.setEmail(source.getEmailAddress());
        target.setPhone(source.getPhoneNumber());
    }

    protected void populateAdministrativeArea(ApplePayAddressDetailsData source, AddressData target) {
        String administrativeArea = source.getAdministrativeArea();
        if (StringUtils.isNotEmpty(administrativeArea)) {
            String regionIso = REGION_ISO_CODE.formatted(source.getCountryCode(), administrativeArea);
            validateRegion(source, regionIso);

            RegionData regionData = new RegionData();
            regionData.setCountryIso(source.getCountryCode());
            regionData.setIsocodeShort(administrativeArea);
            regionData.setIsocode(regionIso);
            target.setRegion(regionData);
        }
    }

    private void validateRegion(ApplePayAddressDetailsData source, String regionIso) {
        try {
            CountryModel countryModel = commonI18NService.getCountry(source.getCountryCode());
            commonI18NService.getRegion(countryModel, regionIso);
        } catch (Exception e) {
            throw new ApplePayInvalidAddressException(APPLE_PAY_INVALID_ADDRESS, APPLE_PAY_REGION_FIELD_CODE);
        }
    }

    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

}
