package com.paypal.hybris.facade.hooks;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.exception.PayPalCreditCardOrderAmountWasChangedException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentService;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.commerceservices.order.hook.AuthorizePaymentMethodHook;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;

import java.math.BigDecimal;

public class UpdateOrderAmountHook implements AuthorizePaymentMethodHook {

    private static final String CREDIT_CARD_ORDER_AMOUT_WAS_CHANGED = "The order amount was changed. Please add Credit Card again.";

    private PayPalConfigurationService payPalConfigurationService;
    private DefaultPayPalPaymentService payPalPaymentService;
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;

    /**
     * Info should only be updated when order is not saved because payPalOrderId is generated after order placement.
     * Info should not be updated when 3ds for Hosted fields is enabled.
     * */
    @Override
    public void beforeAuthorizePaymentAmount(CommerceCheckoutParameter parameter) {
        CartModel cartModel = parameter.getCart();
        if (isCardSaved(parameter)) {
            return;
        }

        PayPalOrderDetailsData orderDetailsData = payPalPaymentService.getOrderDetails(cartModel.getPayPalOrderId());
        if (isCartAmountTheSameAsOrderAmount(parameter, orderDetailsData.getAmount())) {
            return;
        }

        if (payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable() && isHostedFieldProvider(parameter)) {
            throw new PayPalCreditCardOrderAmountWasChangedException(CREDIT_CARD_ORDER_AMOUT_WAS_CHANGED);
        }
        PayPalOrderRequestData orderRequestData = orderRequestDataConverter.convert(cartModel);
        payPalPaymentService.updateOrderAmountDetails(orderRequestData);
    }

    private boolean isCardSaved(CommerceCheckoutParameter parameter) {
        return (isHostedFieldProvider(parameter) || isPayPalProvider(parameter)) && parameter.getCart().getPaymentInfo().isSaved();
    }

    private static boolean isHostedFieldProvider(CommerceCheckoutParameter parameter) {
        PaymentInfoModel paymentInfoModel = parameter.getCart().getPaymentInfo();
        return paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel
                && PaymentProvider.PAYPAL_HOSTED_FIELDS.equals(payPalCreditCardPaymentInfoModel.getPaymentProvider());
    }

    private static boolean isPayPalProvider(CommerceCheckoutParameter parameter) {
        PaymentInfoModel paymentInfoModel = parameter.getCart().getPaymentInfo();
        return paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel
                && PaymentProvider.PAYPAL.equals(payPalCreditCardPaymentInfoModel.getPaymentProvider());
    }

    private boolean isCartAmountTheSameAsOrderAmount(CommerceCheckoutParameter parameter, String orderTotalPrice) {
        return parameter.getAuthorizationAmount().compareTo(new BigDecimal(orderTotalPrice)) == 0;
    }

    @Override
    public void afterAuthorizePaymentAmount(CommerceCheckoutParameter parameter,
                                            PaymentTransactionEntryModel paymentTransactionEntryModel) {
        // The implementation is not required
    }

    public void setPayPalPaymentService(DefaultPayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public void setPayPalConfigurationService(
            PayPalConfigurationService payPalConfigurationService) {
        this.payPalConfigurationService = payPalConfigurationService;
    }

    public void setOrderRequestDataConverter(Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter) {
        this.orderRequestDataConverter = orderRequestDataConverter;
    }
}
