package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.SessionPaymentsEnabledType;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.facade.facades.PayPalSessionPaymentMethodFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.apache.commons.lang.ObjectUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Optional;

public class DefaultPayPalSessionPaymentMethodFacade implements PayPalSessionPaymentMethodFacade {

    private static final String PAYPAL = "paypal";

    private static final String VENMO = "venmo";

    @Resource(name = "payPalConfigurationService")
    private PayPalConfigurationService payPalConfigurationService;

    @Resource(name = "payPalCartService")
    private PayPalCartService payPalCartService;

    @Resource(name = "paymentService")
    private PayPalPaymentService payPalPaymentService;

    private Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter;
    
    @Override
    public CCPaymentInfoData getSessionPaymentMethod() {
        final CartModel cart = payPalCartService.getSessionCart();
        if (!isSessionPaymentFlowEnable() || Optional.ofNullable(cart.getPaymentInfo()).isEmpty()
                || isPaymentMethodSaved(cart) || !isSessionPaymentValid(cart)) {
            return null;
        }
        return creditCardPaymentInfoConverter.convert((CreditCardPaymentInfoModel)cart.getPaymentInfo());
    }

    @Override
    public boolean isPayPalSessionPaymentForExpressCheckoutExist() {
        final Optional<CCPaymentInfoData> sessionPayment = Optional.ofNullable(getSessionPaymentMethod());
        return sessionPayment.isPresent() && sessionPayment.get().getCardType().equals(PAYPAL);
    }

    @Override
    public boolean isVenmoSessionPaymentForExpressCheckoutExist() {
        final Optional<CCPaymentInfoData> sessionPayment = Optional.ofNullable(getSessionPaymentMethod());
        return sessionPayment.isPresent() && sessionPayment.get().getCardType().equals(VENMO);
    }

    @Override
    public boolean isSessionPaymentFlowEnable() {
        final SessionPaymentsEnabledType sessionPaymentsEnabled = getSessionPaymentsEnabledType();
        if (sessionPaymentsEnabled == SessionPaymentsEnabledType.GUEST_AND_LOGGED_IN_USERS) {
            return true;
        } else if (sessionPaymentsEnabled == SessionPaymentsEnabledType.LOGGED_IN_USERS_ONLY) {
            final CustomerType customerType = getCustomerTypeFromCurrentSessionCart();
            return customerType == CustomerType.REGISTERED;
        }
        return false;
    }

    @Override
    public boolean isPaymentMethodValid() {
        final CartModel cart = payPalCartService.getSessionCart();
        return isPaymentMethodSaved(cart) || isSessionPaymentValid(cart);
    }

    public boolean isPaymentMethodSaved(final CartModel cart) {
        return cart.getPaymentInfo().isSaved()
                || ((PayPalCreditCardPaymentInfoModel) cart.getPaymentInfo()).isShouldBeSaved();
    }

    @Override
    public boolean isPaymentMethodTypeEligible(PaymentInfoModel paymentInfoModel) {
        boolean isEligible = false;
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel) {
            return ObjectUtils.notEqual(CreditCardType.APPLEPAY, payPalPaymentInfoModel.getType());
        }
        return isEligible;
    }

    @Override
    public boolean isPaymentMethodTypeEligible() {
        if (payPalCartService.hasSessionCart()) {
            PaymentInfoModel paymentInfo = payPalCartService.getSessionCart().getPaymentInfo();
            return isPaymentMethodTypeEligible(paymentInfo);
        }
        return false;
    }

    public boolean isSessionPaymentValid(final CartModel cart) {
        final PaymentInfoModel sessionPaymentInfo = cart.getPaymentInfo();
        return isPaymentMethodTypeEligible(sessionPaymentInfo)
                && isTimeCorrectForSessionPaymentMethod(sessionPaymentInfo)
                && isAmountCorrectForSessionPaymentMethod(cart);
    }

    public boolean isTimeCorrectForSessionPaymentMethod(final PaymentInfoModel paymentInfo) {
        final Date date = new Date();
        return paymentInfo.getModifiedtime().getTime() > (date.getTime() - 900000); //900000 millisecs = 15 min
    }

    public boolean isAmountCorrectForSessionPaymentMethod(final CartModel cart) {
        if (payPalConfigurationService.isThreeDsVerificationOnCheckoutEnable() && isHostedFieldsPaymentProvider(cart)) {
            final String orderTotalPrice = payPalPaymentService.getOrderDetails(cart.getPayPalOrderId()).getAmount();
            return cart.getTotalPrice().compareTo(Double.parseDouble(orderTotalPrice)) == 0;
        }
        return true;
    }

    public boolean isHostedFieldsPaymentProvider(final CartModel cart) {
        return ((PayPalCreditCardPaymentInfoModel)
                cart.getPaymentInfo()).getPaymentProvider() == PaymentProvider.PAYPAL_HOSTED_FIELDS;
    }

    public SessionPaymentsEnabledType getSessionPaymentsEnabledType() {
        return payPalConfigurationService.getPayPalPluginConfiguration().getPayPalGeneralConfiguration()
                .getSessionPaymentsEnabled();
    }

    public CustomerType getCustomerTypeFromCurrentSessionCart() {
        final CartModel cart = payPalCartService.getSessionCart();
        final CustomerModel customer = (CustomerModel) cart.getUser();
        return customer.getType();
    }

    public void setPayPalConfigurationService(PayPalConfigurationService payPalConfigurationService) {
        this.payPalConfigurationService = payPalConfigurationService;
    }

    public void setPayPalCartService(PayPalCartService payPalCartService) {
        this.payPalCartService = payPalCartService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public void setCreditCardPaymentInfoConverter(
            final Converter<CreditCardPaymentInfoModel, CCPaymentInfoData> creditCardPaymentInfoConverter)
    {
        this.creditCardPaymentInfoConverter = creditCardPaymentInfoConverter;
    }
}
