package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPageType;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalConnectService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalSDKData;
import com.paypal.hybris.facade.facades.PayPalClientTokenFacade;
import com.paypal.hybris.facade.facades.PayPalSDKFacade;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import org.apache.commons.lang.StringUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;


public class DefaultPayPalSDKFacade implements PayPalSDKFacade {

    private static final String DELIMITER_COMA = ",";
    private static final String VENMO_FUNDING = "venmo";
    private static final String PAYLATER_FUNDING = "paylater";
    private static final String APPLEPAY = "applepay";
    private static final String SANDBOX = "sandbox";
    private static final String ENABLE_FUNDING_PARAMETER = "enable-funding";
    private static final String CLIENT_ID_PARAMETER = "client-id";
    private static final String COMMIT_PARAMETER = "commit";
    private static final String INTENT_PARAMETER = "intent";
    private static final String VAULT_PARAMETER = "vault";
    private static final String CURRENCY_PARAMETER = "currency";
    private static final String BUYER_COUNTRY_PARAMETER = "buyer-country";
    private static final String LOCALE_PARAMETER = "locale";
    private static final String COMPONENTS_PARAMETER = "components";
    private static final String BUTTONS = "buttons";
    private static final String MESSAGES = "messages";
    private static final String FUNDING_ELIGIBILITY = "funding-eligibility";
    private static final String HOSTED_FIELDS = "hosted-fields";
    private static final String PAYMENT_FIELDS = "payment-fields";
    private static final String AMPERSAND = "&";
    private static final String EQUAL = "=";
    private static final Set<PayPalPageType> INTENT_REQUIRED_PAGE_TYPES = Set.of(PayPalPageType.MINICART,
            PayPalPageType.CART, PayPalPageType.MARK);

    private PayPalConfigurationService payPalConfigurationService;
    private CommonI18NService commonI18NService;
    private PayPalClientTokenFacade payPalClientTokenFacade;
    private PayPalConnectService payPalConnectService;

    @Override
    public PayPalSDKData getPayPalSDKData(PayPalPageType pageType) {
        String url;

        if (pageType.equals(PayPalPageType.LOGIN)) {
            url = getPayPalSDKUrlForConnect();
        } else if (pageType.equals(PayPalPageType.ACCOUNT)) {
            url = getPayPalSDKUrlForAccount();
        } else {
            url = getPayPalSDKUrl();
        }

        return GenericBuilder.of(PayPalSDKData::new)
                .with(PayPalSDKData::setPayPalSDKUrl, url)
                .with(PayPalSDKData::setClientToken, payPalClientTokenFacade.getClientToken())
                .with(PayPalSDKData::setDataUserIdToken, payPalConnectService.getIdToken(pageType))
                .with(PayPalSDKData::setPartnerAttributionId, payPalConfigurationService.getPayPalPartnerAttributionId())
                .build();
    }

    @Override
    public PayPalSDKData getPayPalSDKData() {
        return GenericBuilder.of(PayPalSDKData::new)
                .with(PayPalSDKData::setPayPalSDKUrl, getPayPalSDKUrl())
                .with(PayPalSDKData::setClientToken, payPalClientTokenFacade.getClientToken())
                .with(PayPalSDKData::setDataUserIdToken, payPalConnectService.getIdToken(PayPalPageType.MARK))
                .with(PayPalSDKData::setPartnerAttributionId, payPalConfigurationService.getPayPalPartnerAttributionId())
                .build();
    }

    private String getPayPalSDKUrl() {
        return payPalConfigurationService.getPayPalSdkJsUrl().concat(getPayPalSdkParameters());
    }

    private String getPayPalSDKUrlForConnect() {
        return payPalConfigurationService.getPayPalSdkJsUrl().concat(getPayPalSdkParametersForPageType(PayPalPageType.LOGIN));
    }

    private String getPayPalSDKUrlForAccount() {
        return payPalConfigurationService.getPayPalSdkJsUrl().concat(getPayPalSdkParametersForPageType(PayPalPageType.ACCOUNT));
    }

    private String getPayPalSdkParametersForPageType(PayPalPageType payPalPageType) {
        Map<String, String> parameters = new HashMap<>();

        parameters.put(COMPONENTS_PARAMETER, getEnableComponents());
        parameters.put(COMMIT_PARAMETER, String.valueOf(payPalConfigurationService.isCommitEnabled()));
        parameters.put(VAULT_PARAMETER, getVaultParameter());

        if (payPalPageType.equals(PayPalPageType.LOGIN)) {
            setSpecificPayPalSDKParametersForConnect(parameters);
        } else {
            parameters.put(ENABLE_FUNDING_PARAMETER, getEnableFundingParameter());
        }

        parameters.put(CLIENT_ID_PARAMETER, payPalConfigurationService.getClientID());
        parameters.put(CURRENCY_PARAMETER, commonI18NService.getCurrentCurrency().getIsocode());
        parameters.put(BUYER_COUNTRY_PARAMETER, getBuyerCountryParameter());
        parameters.put(LOCALE_PARAMETER, payPalConfigurationService.getLocale());

        if (INTENT_REQUIRED_PAGE_TYPES.contains(payPalPageType)) {
            parameters.put(INTENT_PARAMETER, payPalConfigurationService.getPayPalIntent());
        }

        return convertMapToUrl(parameters);
    }

    private String getPayPalSdkParameters() {
        Map<String, String> parameters = new HashMap<>();

        parameters.put(COMPONENTS_PARAMETER, getEnableComponents());
        parameters.put(ENABLE_FUNDING_PARAMETER, getEnableFundingParameter());
        parameters.put(CLIENT_ID_PARAMETER, payPalConfigurationService.getClientID());
        parameters.put(COMMIT_PARAMETER, String.valueOf(payPalConfigurationService.isCommitEnabled()));
        parameters.put(INTENT_PARAMETER, payPalConfigurationService.getPayPalIntent());
        parameters.put(VAULT_PARAMETER, getVaultParameter());
        parameters.put(CURRENCY_PARAMETER, commonI18NService.getCurrentCurrency().getIsocode());
        parameters.put(BUYER_COUNTRY_PARAMETER, getBuyerCountryParameter());
        parameters.put(LOCALE_PARAMETER, payPalConfigurationService.getLocale());

        return convertMapToUrl(parameters);

    }

    private void setSpecificPayPalSDKParametersForConnect(Map<String, String> parameters) {
        parameters.put(COMPONENTS_PARAMETER, BUTTONS);
        parameters.put(COMMIT_PARAMETER, Boolean.FALSE.toString());
        parameters.put(VAULT_PARAMETER, Boolean.TRUE.toString());
    }

    private String getVaultParameter() {
        // This parameter will always be set to 'false' because we handle vault flow on our side
        return Boolean.FALSE.toString();
    }

    private String getBuyerCountryParameter() {
        if (SANDBOX.equals(payPalConfigurationService.getEnvironmentType())) {
            return payPalConfigurationService.getBuyerCountry();
        }
        return StringUtils.EMPTY;
    }

    private static String convertMapToUrl(Map<String, String> map) {
        return map.entrySet().stream()
                .filter(e -> !e.getValue().isEmpty())
                .map(e -> e.getKey().concat(EQUAL).concat(e.getValue()))
                .collect(Collectors.joining(AMPERSAND));
    }

    private String getEnableFundingParameter() {
        Set<String> enableFundingParameters = new HashSet<>();
        if (payPalConfigurationService.isVenmoEnabled()) {
            enableFundingParameters.add(VENMO_FUNDING);
        }
        if (payPalConfigurationService.isPayLaterEnabledForNonUsCountries()) {
            enableFundingParameters.add(PAYLATER_FUNDING);
        }
        return String.join(DELIMITER_COMA, enableFundingParameters);
    }

    private String getEnableComponents() {
        Set<String> enableComponents = new HashSet<>(List.of(BUTTONS, MESSAGES, FUNDING_ELIGIBILITY));
        if (payPalConfigurationService.isPayPalHostedFieldsEnabled()) {
            enableComponents.add(HOSTED_FIELDS);
        }
        if (payPalConfigurationService.isLocalPaymentsEnable()) {
            enableComponents.add(PAYMENT_FIELDS);
        }
        if (payPalConfigurationService.isApplePayEnable()) {
            enableComponents.add(APPLEPAY);
        }
        return String.join(DELIMITER_COMA, enableComponents);
    }

    public void setPayPalConfigurationService(PayPalConfigurationService payPalConfigurationService) {
        this.payPalConfigurationService = payPalConfigurationService;
    }

    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

    public void setPayPalClientTokenFacade(PayPalClientTokenFacade payPalClientTokenFacade) {
        this.payPalClientTokenFacade = payPalClientTokenFacade;
    }

    public void setPayPalConnectService(PayPalConnectService payPalConnectService) {
        this.payPalConnectService = payPalConnectService;
    }

}
