package com.paypal.hybris.facade.facades.impl;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalConnectUserData;
import com.paypal.hybris.facade.facades.PayPalConnectWithPayPalNotificationFacade;
import com.paypal.hybris.facade.facades.PayPalCustomerFacade;
import com.paypal.hybris.facade.facades.PayPalRegistrationUserFacade;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.servicelayer.util.ServicesUtil;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static org.apache.commons.lang.StringUtils.defaultString;

public class DefaultPayPalCustomerFacade implements PayPalCustomerFacade {

    private static final String PAYPAL_PAYMENT_PROVIDER = "PAYPAL";

    private PayPalRegistrationUserFacade payPalRegistrationUserFacade;
    private UserService userService;
    private CustomerAccountService customerAccountService;
    private UserFacade userFacade;
    private ModelService modelService;
    private PayPalConnectWithPayPalNotificationFacade payPalConnectWithPayPalNotificationFacade;

    @Override
    public boolean isPayPalAddressPresent(String email, PayPalConnectAddressData payPalAddressData) {
        Collection<AddressModel> addresses = userService.getUserForUID(email).getAddresses();

        for (AddressModel addressModel : addresses) {
            if (compareAddressDetails(addressModel, payPalAddressData) && compareRegions(addressModel, payPalAddressData)) {
                return true;
            }
        }
        return false;
    }

    private static boolean compareRegions(AddressModel addressModel, PayPalConnectAddressData payPalAddressData) {
        if (Objects.isNull(addressModel.getRegion()) && Objects.isNull(payPalAddressData.getRegion())) {
            return true;
        } else if (Objects.nonNull(addressModel.getRegion())) {
            return Objects.equals(addressModel.getRegion().getIsocodeShort(), payPalAddressData.getRegion());
        }
        return false;
    }

    private static boolean compareAddressDetails(AddressModel addressModel, PayPalConnectAddressData payPalAddressData) {
        return Objects.equals(addressModel.getCountry().getIsocode(), payPalAddressData.getCountry())
                && Objects.equals(addressModel.getPostalcode(), payPalAddressData.getPostal_code())
                && Objects.equals(addressModel.getLine1(), payPalAddressData.getStreet_address())
                && Objects.equals(addressModel.getTown(), payPalAddressData.getLocality());
    }

    @Override
    public boolean isPayPalPaymentMethodPresent(String email, PayPalConnectUserData userData) {
        CustomerModel userForUID = (CustomerModel) userService.getUserForUID(email);
        List<CreditCardPaymentInfoModel> creditCardPaymentInfos = customerAccountService.getCreditCardPaymentInfos(userForUID, true);

        return creditCardPaymentInfos.stream()
                .filter(PayPalCreditCardPaymentInfoModel.class::isInstance)
                .map(PayPalCreditCardPaymentInfoModel.class::cast)
                .filter(payPalCreditCardPaymentInfoModel -> payPalCreditCardPaymentInfoModel.getPaymentProvider().getCode().equals(PAYPAL_PAYMENT_PROVIDER))
                .anyMatch(payPalCreditCardPaymentInfoModel -> payPalCreditCardPaymentInfoModel.getPayerEmail().equalsIgnoreCase(payPalRegistrationUserFacade.getEmailFromPayPalUserData(userData)));
    }

    @Override
    public boolean isPayerIdInCustomer(String uid) {
        return StringUtils.isNotBlank(((CustomerModel) userService.getUserForUID(uid)).getPayPalPayerId());
    }

    @Override
    public boolean isUserDisabled(Date deactivationDate) {
        return deactivationDate != null && deactivationDate.compareTo(new Date()) < 0;
    }

    @Override
    public void unlinkLoginWithPayPal() {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        ServicesUtil.validateParameterNotNull(customerModel.getPayPalPayerId(), "payerId must not be null");
        customerModel.setPayPalPayerId(null);
        modelService.save(customerModel);
        payPalConnectWithPayPalNotificationFacade.sendUnlinkCustomerEmailNotification(customerModel.getUid());
    }

    @Override
    public boolean addShippingAddressToCurrentUser(AddressData newShippingAddress) {
        if (!isDuplicateAddressForCustomer(newShippingAddress, (CustomerModel) userService.getCurrentUser())) {
            newShippingAddress.setVisibleInAddressBook(true);
            newShippingAddress.setShippingAddress(true);
            userFacade.addAddress(newShippingAddress);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isDuplicateAddressForCustomer(AddressData addressData, CustomerModel customerModel) {
        for (AddressModel address : customerModel.getAddresses()) {
            if (address.getVisibleInAddressBook()
                    && compareRegions(address, addressData)
                    && compareAddressData(address, addressData)) {
                return true;
            }
        }
        return false;
    }

    private static boolean compareAddressData(AddressModel address, AddressData addressData) {
        return Objects.equals(defaultString(address.getCountry().getIsocode()), defaultString(addressData.getCountry().getIsocode()))
                && Objects.equals(defaultString(address.getFirstname()), defaultString(addressData.getFirstName()))
                && Objects.equals(defaultString(address.getLastname()), defaultString(addressData.getLastName()))
                && Objects.equals(defaultString(address.getTown()), defaultString(addressData.getTown()))
                && Objects.equals(defaultString(address.getPostalcode()), defaultString(addressData.getPostalCode()))
                && Objects.equals(defaultString(address.getLine1()), defaultString(addressData.getLine1()))
                && Objects.equals(defaultString(address.getLine2()), defaultString(addressData.getLine2()))
                && Objects.equals(defaultString(address.getPhone1()), defaultString(addressData.getPhone()));
    }

    private static boolean compareRegions(AddressModel address, AddressData addressData) {
        if (address.getRegion() == null || addressData.getRegion() == null) {
            return true;
        } else {
            return Objects.equals(address.getRegion().getIsocode(), addressData.getRegion().getIsocode());
        }
    }

    public void setPayPalRegistrationUserFacade(PayPalRegistrationUserFacade payPalRegistrationUserFacade) {
        this.payPalRegistrationUserFacade = payPalRegistrationUserFacade;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setCustomerAccountService(CustomerAccountService customerAccountService) {
        this.customerAccountService = customerAccountService;
    }

    public void setUserFacade(UserFacade userFacade) {
        this.userFacade = userFacade;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setPayPalConnectWithPayPalNotificationFacade(PayPalConnectWithPayPalNotificationFacade payPalConnectWithPayPalNotificationFacade) {
        this.payPalConnectWithPayPalNotificationFacade = payPalConnectWithPayPalNotificationFacade;
    }
}
