package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CCSetupTokenData;
import com.paypal.hybris.addon.forms.PaymentTokenData;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.exception.PayPalCreditCardRemovalException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.impl.DefaultPayPalCustomerAccountService;
import com.paypal.hybris.core.service.impl.DefaultPayPalPaymentInfoService;
import com.paypal.hybris.data.PayPalPaymentLinks;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.SetupTokenRequestData;
import com.paypal.hybris.facade.builders.PaypalGetPaymentTokenRequestBuilder;
import com.paypal.hybris.facade.builders.TokenRequestDirector;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import com.paypal.hybris.facade.strategy.payment.PaymentStrategy;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.exceptions.ModelNotFoundException;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class DefaultPayPalCreditCardFacade implements PayPalCreditCardFacade {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCreditCardFacade.class);

    private static final String APPROVE = "approve";
    private static final String PAYER_ID = "payerId";

    private TokenRequestDirector tokenRequestDirector;
    private UserFacade userFacade;
    private PayPalPaymentInfoService payPalPaymentInfoService;
    private List<PaymentStrategy> paymentPopulatorStrategies;
    private PayPalAcceleratorCheckoutFacade payPalAcceleratorCheckoutFacade;
    private SessionService sessionService;
    protected UserService userService;
    protected ModelService modelService;
    protected DefaultPayPalCustomerAccountService customerAccountService;

    public CCSetupTokenData requestSetupToken(SetupTokenRequestData setupTokenRequestData) {
        CustomerModel customerModel = customerAccountService.getCustomerModel(setupTokenRequestData.getPayerId());
        PayPalCreditCardPaymentInfoModel emptyCreditCardPaymentInfo =
                customerAccountService.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel);

        PayPalSavePaymentForPurchaseLaterRequest payPalSavePaymentRequest = new PayPalSavePaymentForPurchaseLaterRequest();
        setupTokenRequestData.setPaymentMethodCode(emptyCreditCardPaymentInfo.getPk().getLongValueAsString());

        selectPaymentPopulationStrategy(setupTokenRequestData.getPaymentType())
                .populate(setupTokenRequestData, payPalSavePaymentRequest);

        PayPalSetupTokenResponse payPalSetupTokenResponse;
        try {
            payPalSetupTokenResponse =
                    payPalPaymentInfoService.getPaypalSetupToken(payPalSavePaymentRequest);
        } catch (Exception ex) {
            payPalPaymentInfoService.removePaymentInfoByPK(emptyCreditCardPaymentInfo.getPk().getLongValueAsString());
            throw ex;
        }

        if (StringUtils.isEmpty(customerModel.getVaultCustomerId())) {
            customerModel.setVaultCustomerId(payPalSetupTokenResponse.getCustomer().getId());
            modelService.save(customerModel);
        }

        // The setup token id will be stored temporary in the subscriptionId field until
        // receiving the payment token
        emptyCreditCardPaymentInfo.setSubscriptionId(payPalSetupTokenResponse.getId());
        modelService.save(emptyCreditCardPaymentInfo);

        return populatePaymentSetupDataBasedOnResponse(payPalSetupTokenResponse, emptyCreditCardPaymentInfo, setupTokenRequestData.getPaymentType());
    }

    private static CCSetupTokenData populatePaymentSetupDataBasedOnResponse(PayPalSetupTokenResponse payPalSetupTokenResponse,
                                                                            PayPalCreditCardPaymentInfoModel emptyCreditCardPaymentInfo,
                                                                            PayPalPaymentProvider paymentProvider) {
        CCSetupTokenData setupTokenData = new CCSetupTokenData();
        if (PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS.equals(paymentProvider)) {
            setupTokenData.setSelectedBillingAddress(emptyCreditCardPaymentInfo.getBillingAddress().getPk()
                    .getLongValueAsString());
        }
        setupTokenData.setCustomerId(payPalSetupTokenResponse.getCustomer().getId());
        setupTokenData.setStatus(payPalSetupTokenResponse.getStatus());
        setupTokenData.setId(payPalSetupTokenResponse.getId());
        setupTokenData.setApproveLink(payPalSetupTokenResponse.getLinks().stream()
                .filter(link -> link.getRel().equals(APPROVE))
                .map(PayPalPaymentLinks::getHref)
                .findFirst().orElse(StringUtils.EMPTY));
        setupTokenData.setPaymentInfoPK(emptyCreditCardPaymentInfo.getPk().getLongValueAsString());
        return setupTokenData;
    }

    public void requestPaymentToken(PaymentTokenData paymentTokenData) throws Exception {
        PayPalSavePaymentForPurchaseLaterRequest payPalSaveCardRequest = preparePaymentTokenRequest(paymentTokenData);
        PayPalCreditCardPaymentInfoModel creditCardPaymentInfo =
                payPalPaymentInfoService.getPaymentInfoByPK(paymentTokenData.getPaymentInfoPK()).orElseThrow(Exception::new);

        PayPalSetupTokenResponse payPalPaymentTokenResponse;
        try {
            payPalPaymentTokenResponse =
                    payPalPaymentInfoService.getPaypalPaymentToken(payPalSaveCardRequest);
        } catch (Exception ex) {
            payPalPaymentInfoService.removePaymentInfoByPK(paymentTokenData.getPaymentInfoPK());
            throw ex;
        }

        if (Objects.nonNull(payPalPaymentTokenResponse.getPaymentSource().getPaypal())) {
            final String payerId = getPayerId(paymentTokenData);
            CustomerModel customerModel = customerAccountService.getCustomerModel(payerId);
            payPalAcceleratorCheckoutFacade.createPayPalPaymentSubscription(payPalPaymentTokenResponse, true,
                    creditCardPaymentInfo, customerModel);
        } else {
            customerAccountService.updateStubCreditCardPaymentInfo(creditCardPaymentInfo, payPalPaymentTokenResponse);
        }
    }

    private String getPayerId(PaymentTokenData paymentTokenData) {
        if (StringUtils.isNotBlank(paymentTokenData.getPayerId())) {
            return paymentTokenData.getPayerId();
        } else {
            return sessionService.getAttribute(PAYER_ID);
        }
    }

    private PayPalSavePaymentForPurchaseLaterRequest preparePaymentTokenRequest(PaymentTokenData paymentTokenData) {
        PaypalGetPaymentTokenRequestBuilder setupTokenRequestBuilder = new PaypalGetPaymentTokenRequestBuilder();

        tokenRequestDirector.constructGetPaymentTokenRequest(setupTokenRequestBuilder, paymentTokenData);

        return setupTokenRequestBuilder.getResult();
    }

    public String getCardsExpirationStatus() {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        return getCardsExpirationStatus(customerModel);
    }

    public String getCardsExpirationStatus(final String customerUid) {
        ExpirationStatus result = ExpirationStatus.NOT_EXPIRED;
        CustomerModel customerModel = userService.getUserForUID(customerUid, CustomerModel.class);
        if (customerModel != null) {
            return getCardsExpirationStatus(customerModel);
        }
        return result.getCode();
    }

    public String getCardsExpirationStatus(CustomerModel customerModel) {
        ExpirationStatus result = ExpirationStatus.NOT_EXPIRED;
        List<PayPalCreditCardPaymentInfoModel> cards = customerModel.getPaymentInfos()
                .stream()
                .filter(PayPalCreditCardPaymentInfoModel.class::isInstance)
                .filter(paymentInfoModel -> !paymentInfoModel.getDuplicate())
                .filter(PaymentInfoModel::isSaved)
                .map(PayPalCreditCardPaymentInfoModel.class::cast)
                .collect(Collectors.toList());
        Predicate<ExpirationStatus> isExpiredStatus = expirationStatus -> cards
                .stream()
                .anyMatch(card -> card.getExpirationStatus().equals(expirationStatus));
        if (isExpiredStatus.test(ExpirationStatus.EXPIRED)) {
            result = ExpirationStatus.EXPIRED;
        } else if (isExpiredStatus.test(ExpirationStatus.EXPIRE_SOON) && customerModel.isHasNewExpireSoonCard()) {
            result = ExpirationStatus.EXPIRE_SOON;
            customerModel.setHasNewExpireSoonCard(false);
            modelService.save(customerModel);
        }
        return result.getCode();
    }

    public void deleteCreditCardFromPayPal(String paymentMethodId) {
        final CustomerModel currentCustomer = (CustomerModel) userService.getCurrentUser();
        final Optional<PaymentInfoModel> defaultPayment = Optional.ofNullable(currentCustomer.getDefaultPaymentInfo());
        final PayPalCreditCardPaymentInfoModel selectedPaymentInfo = (PayPalCreditCardPaymentInfoModel)
                customerAccountService.getCreditCardPaymentInfoForCode(currentCustomer, paymentMethodId);

        if(selectedPaymentInfo == null){
            throw new ModelNotFoundException("selected payment method not found for customer");
        }

        deletePayPalPaymentToken(selectedPaymentInfo.getSubscriptionId());
        userFacade.removeCCPaymentInfo(paymentMethodId);
        defaultPayment.filter(paymentInfoModel -> paymentInfoModel.getCode().equals(selectedPaymentInfo.getCode()))
                .ifPresent(paymentInfoModel -> setToDefaultLastAddedPayment(currentCustomer, paymentInfoModel));
    }

    private void deletePayPalPaymentToken(String subscriptionId) {
        if (!payPalPaymentInfoService.deletePaypalPaymentToken(subscriptionId)) {
            throw new PayPalCreditCardRemovalException();
        }
        LOG.info("Payment with subscriptionId - " + subscriptionId + " successfully removed from PayPal side");
    }


    public void deleteCreditCardFromWebhookEvent(String subscriptionId) {
        payPalPaymentInfoService.getCreditCardBySubscriptionId(subscriptionId)
                .ifPresent(creditCardPaymentInfoModel -> {
                    final CustomerModel customer = (CustomerModel) creditCardPaymentInfoModel.getUser();
                    deleteCCPaymentAndSetLastDefault(creditCardPaymentInfoModel, customer);
                });
    }

    protected void deleteCCPaymentAndSetLastDefault(PayPalCreditCardPaymentInfoModel creditCardPaymentInfoModel, CustomerModel customer) {
        final Optional<PaymentInfoModel> defaultPayment = Optional.ofNullable(customer.getDefaultPaymentInfo());
        customerAccountService.deleteCCPaymentInfo(customer, creditCardPaymentInfoModel);
        defaultPayment.filter(paymentInfoModel -> paymentInfoModel.getCode().equals(creditCardPaymentInfoModel.getCode()))
                .ifPresent(paymentInfoModel -> setToDefaultLastAddedPayment(customer, paymentInfoModel));
    }

    private void setToDefaultLastAddedPayment(CustomerModel currentCustomer, PaymentInfoModel selectedPaymentInfo) {
        Optional<PaymentInfoModel> lastPaymentInfoModel = currentCustomer.getPaymentInfos()
                .stream()
                .filter(PaymentInfoModel::isSaved)
                .filter(paymentInfoModel -> !paymentInfoModel.getDuplicate())
                .filter(paymentInfoModel -> !paymentInfoModel.getCode().equals(selectedPaymentInfo.getCode()))
                .reduce((firstPayment, secondPayment) -> secondPayment);
        lastPaymentInfoModel.ifPresent(currentCustomer::setDefaultPaymentInfo);
        modelService.save(currentCustomer);
    }

    private PaymentStrategy selectPaymentPopulationStrategy(PayPalPaymentProvider paymentProvider) {
        return paymentPopulatorStrategies.stream()
                .filter(strategy -> strategy.test(paymentProvider))
                .findAny()
                .orElseThrow(() ->
                        new RuntimeException("No suitable payment strategy found: strategy=" + paymentProvider));
    }

    public void setPaymentInfoService(DefaultPayPalPaymentInfoService paymentInfoService) {
        this.payPalPaymentInfoService = paymentInfoService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setTokenRequestDirector(TokenRequestDirector tokenRequestDirector) {
        this.tokenRequestDirector = tokenRequestDirector;
    }

    public void setCustomerAccountService(DefaultPayPalCustomerAccountService customerAccountService) {
        this.customerAccountService = customerAccountService;
    }

    public void setUserFacade(UserFacade userFacade) {
        this.userFacade = userFacade;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setPayPalPaymentInfoService(PayPalPaymentInfoService payPalPaymentInfoService) {
        this.payPalPaymentInfoService = payPalPaymentInfoService;
    }

    public void setPaymentPopulatorStrategies(List<PaymentStrategy> paymentPopulatorStrategies) {
        this.paymentPopulatorStrategies = paymentPopulatorStrategies;
    }

    public void setPayPalAcceleratorCheckoutFacade(PayPalAcceleratorCheckoutFacade payPalAcceleratorCheckoutFacade) {
        this.payPalAcceleratorCheckoutFacade = payPalAcceleratorCheckoutFacade;
    }

    public void setSessionService(SessionService sessionService) {
        this.sessionService = sessionService;
    }

    protected PayPalPaymentInfoService getPaymentInfoService() {
        return payPalPaymentInfoService;
    }

}
