/*

 */
package com.paypal.hybris.facade.facades.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.enums.LocalPaymentFunding;
import com.paypal.hybris.core.exception.PayPalPaymentInfoNotFoundException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.storedcredential.StoredCredentialStrategy;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.ApplePayAddressDetailsData;
import com.paypal.hybris.data.ApplePayLineItem;
import com.paypal.hybris.data.ApplePayOrderRequestData;
import com.paypal.hybris.data.ApplePaySessionDetailsData;
import com.paypal.hybris.data.AuthenticationResultData;
import com.paypal.hybris.data.CreditCardPaymentMethodData;
import com.paypal.hybris.data.NameData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalCheckoutData;
import com.paypal.hybris.data.PayPalConnectAddressData;
import com.paypal.hybris.data.PayPalCreateOrderData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalHostedFieldsCreateOrderData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalPaymentMethodData;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.PayPalVaultOrderRequestData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.StoredCredentialData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.service.impl.BreakdownCalculationService;
import de.hybris.platform.acceleratorfacades.order.impl.DefaultAcceleratorCheckoutFacade;
import de.hybris.platform.commercefacades.address.AddressVerificationFacade;
import de.hybris.platform.commercefacades.address.data.AddressVerificationResult;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.DeliveryOrderEntryGroupData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.commercefacades.user.UserFacade;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.address.AddressVerificationDecision;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.enums.CountryType;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.logging.log4j.util.Strings;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.paypal.hybris.core.constants.GeneratedPaypalcoreConstants.Enumerations.CreditCardType.PAYPAL;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.FIRST_PAYMENT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.GET_FROM_FILE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.ONE_TIME_PAYMENT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER;

/**
 * This method is a default implementation of the PayPalAcceleratorCheckoutFacade interface
 */
public class DefaultPayPalAcceleratorCheckoutFacade extends DefaultAcceleratorCheckoutFacade implements
        PayPalAcceleratorCheckoutFacade {

    private static final String POSSIBLE = "POSSIBLE";
    private static final String YES = "YES";
    private static final String UNKNOWN = "UNKNOWN";
    private static final String NO = "NO";
    public static final String N = "N";
    private static final Logger LOG = Logger.getLogger(DefaultPayPalAcceleratorCheckoutFacade.class);
    private static final String EMPTY = "";
    private static final String SPACE = " ";
    private static final String PAYPAL_CREDIT_CARD_PAYMENT_INFO = "PayPalCreditCardPaymentInfo";
    private static final String PAYPAL_ORDER_ID_PLACEHOLDER = "ORDER_ID_PLACEHOLDER";
    private static final String PAYPAL_PAYMENT_METHOD = "paypal";
    private static final String U = "U";
    private static final String B = "B";
    private static final String MERCHANT = "MERCHANT";
    private static final String CONSUMER = "CONSUMER";
    private static final String SUBTOTAL = "Subtotal";
    private static final String SHIPPING = "Shipping";
    private static final String TAX = "Tax";
    private static final String FINAL_PRICE_TYPE = "final";
    private static final String EXPRESS_CHECKOUT = "EXPRESS_CHECKOUT";

    private PayPalCustomerAccountService payPalCustomerAccountService;
    private PayPalPaymentService payPalPaymentService;
    private PayPalConfigurationService defaultPayPalConfigurationService;
    private CommonI18NService commonI18NService;
    private CartService cartService;
    private PayPalCartService payPalCartService;
    private AddressVerificationFacade addressVerificationFacade;
    private UserFacade userFacade;
    private Populator<PayPalAddressDetailsData, AddressData> payPalAddressDataPopulator;
    private Populator<AddressData, PayPalAddressDetailsData> payPalAddressDataReversePopulator;
    private Converter<ApplePayAddressDetailsData, AddressData> applePayAddressDataConverter;
    private Populator<ApplePayAddressDetailsData, PayPalAddressDetailsData> applePayPayPalAddressDataPopulator;
    private Populator<AddressData, ApplePayAddressDetailsData> applePayAddressDataReversePopulator;
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    private CustomerAccountService customerAccountService;
    private UserService userService;
    private CustomerFacade customerFacade;
    private PayPalPaymentInfoService payPalPaymentInfoService;
    private List<StoredCredentialStrategy> storedCredentialStrategies;
    private StoredCredentialStrategy emptyStoredCredentialStrategy;
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    private Converter<AbstractOrderModel, PayPalVaultOrderRequestData> vaultOrderRequestDataConverter;
    private BreakdownCalculationService breakdownCalculationService;

    @Override
    public boolean processExpressCheckout(final PayPalOrderDetailsData orderDetails, String paymentMethodType) {
        final AddressData shippingAddress = createAddressDataFromPayPalAddress(orderDetails.getShippingAddress(), true,
                getCurrentUserForCheckout());
        if (isPickupInStore()) {
            removeDeliveryAddressFromCart();
            setCheapestDeliveryModeForCheckout();
        } else if (isPayPalAddressValid(shippingAddress)) {
            removeDeliveryAddressFromCart();
            setDeliveryAddressForCheckout(shippingAddress);
            setCheapestDeliveryModeForCheckout();
        } else {
            return false;
        }
        updatePayPalOrder(orderDetails);
        final CCPaymentInfoData payPalPaymentSubscription = createPayPalPaymentSubscriptionForExpressCheckout(
                orderDetails, paymentMethodType);
        return setPaymentDetails(payPalPaymentSubscription.getId());
    }

    @Override
    public void updateCheckoutPaymentInfoOrderId(final String orderId) {
        final PaymentInfoModel paymentInfo = getCart().getPaymentInfo();
        if (paymentInfo instanceof PayPalCreditCardPaymentInfoModel) {
            ((PayPalCreditCardPaymentInfoModel) paymentInfo).setPayPalOrderId(orderId);
            getModelService().save(paymentInfo);
        }
    }

    @Override
    public boolean setSelectedPaymentInfoToCart(String paymentInfoId) {
        if (checkIfCurrentUserIsTheCartUser() && StringUtils.isNotBlank(paymentInfoId)) {
            final CartModel cartModel = getCart();
            final Optional<CreditCardPaymentInfoModel> clonedCcPaymentInfo = cloneSelectedPaymentInfo(paymentInfoId);
            if (clonedCcPaymentInfo.isEmpty()) {
                LOG.warn(String.format(
                        "Did not find CreditCardPaymentInfoModel for cart: %s &  paymentInfoId: %s. PaymentInfo Will not get set.",
                        cartModel, paymentInfoId));
            } else {
                final CommerceCheckoutParameter parameter = createCommerceCheckoutParameter(cartModel, true);
                parameter.setPaymentInfo(clonedCcPaymentInfo.get());
                return getCommerceCheckoutService().setPaymentInfo(parameter);
            }
        }
        return false;
    }

    private Optional<CreditCardPaymentInfoModel> cloneSelectedPaymentInfo(String paymentInfoId) {
        final CustomerModel currentUserForCheckout = getCurrentUserForCheckout();
        return Optional.ofNullable(getCustomerAccountService().getCreditCardPaymentInfoForCode(currentUserForCheckout, paymentInfoId))
                .map(this::clonePaymentInfoForCurrentUser);
    }

    private CreditCardPaymentInfoModel clonePaymentInfoForCurrentUser(CreditCardPaymentInfoModel originalPaymentInfo) {
        final CreditCardPaymentInfoModel clonedCcPaymentInfo = getModelService().clone(originalPaymentInfo);
        clonedCcPaymentInfo.setOriginal(originalPaymentInfo);
        clonedCcPaymentInfo.setDuplicate(true);
        clonedCcPaymentInfo.setCode(getCurrentUserForCheckout().getUid() + "_" + UUID.randomUUID());
        getModelService().save(clonedCcPaymentInfo);
        return clonedCcPaymentInfo;
    }

    @Override
    public boolean isPickupInStore() {
        final List<DeliveryOrderEntryGroupData> deliveryOrderGroups = getCheckoutCart().getDeliveryOrderGroups();
        return deliveryOrderGroups != null && deliveryOrderGroups.isEmpty();
    }

    @Override
    public boolean isLocalPaymentFlow() {
        return getCheckoutCart().getPaymentInfo() != null && getCheckoutCart().getPaymentInfo().getCardType().equals(CreditCardType.LOCAL_PAYMENT.getCode())
                && defaultPayPalConfigurationService.isCommitEnabled();
    }

    @Override
    public PayPalOrderDetailsData getPayPalOrderDetails(final String payPalOrderId) {
        return payPalPaymentService.getOrderDetails(payPalOrderId);
    }

    @Override
    public PayPalOrderDetailsData getPayPalOrderDetails() {
        final String payPalOrderId = getCheckoutCart().getPayPalOrderId();
        return payPalPaymentService.getOrderDetails(payPalOrderId);
    }

    public boolean isThreeDsVerificationSuccess(final AuthenticationResultData authenticationResultData) {
        return authenticationResultData == null || verifyAuthenticationResult(authenticationResultData);
    }

    private static boolean verifyAuthenticationResult(AuthenticationResultData authenticationResultData) {
        return !isLiabilityShiftUnknown(authenticationResultData.getLiabilityShift())
                && (isLiabilityShiftPossible(authenticationResultData.getLiabilityShift())
                || isLiabilityShiftedNotReady(authenticationResultData.getLiabilityShift(), authenticationResultData.getEnrollmentStatus())
                || isLiabilityShiftedUnavailable(authenticationResultData.getLiabilityShift(), authenticationResultData.getEnrollmentStatus())
                || isLiabilityShiftedBypassed(authenticationResultData.getLiabilityShift(), authenticationResultData.getEnrollmentStatus()));
    }

    private static boolean isLiabilityShiftPossible(String liabilityShift) {
        return POSSIBLE.equals(liabilityShift) || YES.equals(liabilityShift);
    }

    private static boolean isLiabilityShiftUnknown(String liabilityShift) {
        return UNKNOWN.equals(liabilityShift);
    }

    private static boolean isLiabilityShiftedNotReady(String liabilityShift, String enrolmentStatus) {
        return NO.equals(liabilityShift) && N.equals(enrolmentStatus);
    }


    private static boolean isLiabilityShiftedUnavailable(String liabilityShift, String enrolmentStatus) {
        return NO.equals(liabilityShift) && U.equals(enrolmentStatus);
    }

    private static boolean isLiabilityShiftedBypassed(String liabilityShift, String enrolmentStatus) {
        return NO.equals(liabilityShift) && B.equals(enrolmentStatus);
    }

    @Override
    public void setDeliveryAddressForCheckout(AddressData addressData) {
        userFacade.addAddress(addressData);
        setDeliveryAddress(addressData);
    }

    @Override
    public String getCurrentSessionUserUid() {
        return getCartService().getSessionCart().getUser().getUid();
    }

    @Override
    public String createPayPalOrder(PayPalCreateOrderData createOrderData) {
        PayPalAddressDetailsData addressDetailsData = null;
        PayPalOrderRequestData payPalOrderRequestData;
        String funding = createOrderData.getFunding();
        boolean isPickUpInStore = isPickupInStore();
        final CustomerModel currentCustomer = (CustomerModel) userService.getCurrentUser();
        ApplePayAddressDetailsData applePayAddressDetailsData = createOrderData.getApplePayShippingAddress();

        if (!isPickUpInStore) {
            addressDetailsData = new PayPalAddressDetailsData();
            if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(funding)) {
                applePayPayPalAddressDataPopulator.populate(applePayAddressDetailsData, addressDetailsData);
                AddressData addressData = applePayAddressDataConverter.convert(applePayAddressDetailsData);
                setDeliveryAddressForCheckout(addressData);
            } else {
                payPalAddressDataReversePopulator.populate(getCheckoutCart().getDeliveryAddress(), addressDetailsData);
            }
        }

        if (CreditCardType.PAYPAL_HOSTED_FIELDS_CARD.name().equalsIgnoreCase(funding)) {
            PayPalHostedFieldsCreateOrderData hostedFieldsData = createOrderData.getHostedFieldsData();
            StoredCredentialData storedCredentialData;
            if (hostedFieldsData.getIsShouldBeSaved()) {
                storedCredentialData = selectStoredCredentialStrategy(FIRST_PAYMENT).getStoredCredential();
            } else {
                storedCredentialData = selectStoredCredentialStrategy(ONE_TIME_PAYMENT).getStoredCredential();
                hostedFieldsData = null;
            }
            payPalOrderRequestData = GenericBuilder.of(PayPalHostedFieldsOrderRequestData::new)
                    .with(PayPalHostedFieldsOrderRequestData::setHostedFieldsData, hostedFieldsData)
                    .with(PayPalHostedFieldsOrderRequestData::setStoredCredential, storedCredentialData)
                    .build();
        } else if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(funding)) {
            StoredCredentialData storedCredentialData = selectStoredCredentialStrategy(ONE_TIME_PAYMENT).getStoredCredential();
            payPalOrderRequestData = GenericBuilder.of(ApplePayOrderRequestData::new)
                    .with(ApplePayOrderRequestData::setStoredCredential, storedCredentialData)
                    .with(ApplePayOrderRequestData::setName, applePayAddressDetailsData.getGivenName() + " " + applePayAddressDetailsData.getFamilyName())
                    .with(ApplePayOrderRequestData::setEmailAddress, applePayAddressDetailsData.getEmailAddress())
                    .with(ApplePayOrderRequestData::setPhoneNumber, applePayAddressDetailsData.getPhoneNumber())
                    .build();
        } else if (isLocalPaymentFunding(funding)) {
            payPalOrderRequestData = new PayPalOrderRequestData();
        } else {
            payPalOrderRequestData = createBasicPayPalVaultOrderRequestData();
            payPalOrderRequestData.setSavePaymentMethod(createOrderData.isSavePaymentMethod());
        }
        populateBasicOrderRequestData(currentCustomer, payPalOrderRequestData);
        payPalOrderRequestData.setShippingAddress(addressDetailsData);
        payPalOrderRequestData.setPickUpInStore(isPickUpInStore);

        final String orderId = payPalPaymentService.createOrder(payPalOrderRequestData);
        payPalCartService.setPayPalOrderIdForCurrentCart(orderId);
        return orderId;
    }

    @Override
    public String createPayPalOrderForExpressCheckout() {
        final CustomerModel currentCustomer = (CustomerModel) userService.getCurrentUser();
//        final Optional<AddressData> defaultShipmentAddress = Optional.ofNullable(customerFacade.getCurrentCustomer()).map(CustomerData::getDefaultShippingAddress).filter(AddressData::isVisibleInAddressBook);
        boolean isPickUpInStore = isPickupInStore();
        String shippingPreference = isPickUpInStore ? null : GET_FROM_FILE;

        final PayPalOrderRequestData orderRequestData = createBasicPayPalVaultOrderRequestData();
        orderRequestData.setCustomerId(currentCustomer.getVaultCustomerId());

//            commented to force a PayPal popup
//        if (payPalPaymentInfoService.isChangePaymentButtonActive() && defaultShipmentAddress.isPresent()) {
//            PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
//            payPalAddressDataReversePopulator.populate(defaultShipmentAddress.get(), payPalAddressDetailsData);
//            orderRequestData.setShippingAddress(payPalAddressDetailsData);
//            shippingPreference = SET_PROVIDED_ADDRESS;
//        }

        orderRequestData.setShippingPreference(shippingPreference);
        orderRequestData.setPickUpInStore(isPickUpInStore);

        final String orderId = payPalPaymentService.createOrder(orderRequestData);
        payPalCartService.setPayPalOrderIdForCurrentCart(orderId);
        return orderId;
    }

    private void populateBasicOrderRequestData(CustomerModel currentCustomer, PayPalOrderRequestData orderRequestData) {
        orderRequestDataConverter.convert(cartService.getSessionCart(), orderRequestData);
        orderRequestData.setIntent(defaultPayPalConfigurationService.getPayPalIntent());
        orderRequestData.setCustomerId(currentCustomer.getVaultCustomerId());

    }

    protected PayPalVaultOrderRequestData createBasicPayPalVaultOrderRequestData() {
        PayPalVaultOrderRequestData orderRequestData = vaultOrderRequestDataConverter.convert(cartService.getSessionCart());
        orderRequestData.setSaveOrderFlowActive(defaultPayPalConfigurationService.isSaveOrderFlow());
        orderRequestData.setIntent(defaultPayPalConfigurationService.getPayPalIntent());
        orderRequestData.setUsageType(MERCHANT);
        orderRequestData.setCustomerType(CONSUMER);
        return orderRequestData;
    }

    private StoredCredentialStrategy selectStoredCredentialStrategy(String flow) {
        return storedCredentialStrategies.stream()
                .filter(strategy -> strategy.test(flow))
                .findAny()
                .orElse(emptyStoredCredentialStrategy);
    }

    @Override
    public String createApplePayOrderForExpressCheckout(ApplePayAddressDetailsData applePayShippingAddress) {
        PayPalAddressDetailsData payPalAddressDetailsData = new PayPalAddressDetailsData();
        applePayPayPalAddressDataPopulator.populate(applePayShippingAddress, payPalAddressDetailsData);

        final ApplePayOrderRequestData applePayOrderRequestData = GenericBuilder.of(ApplePayOrderRequestData::new)
                .with(ApplePayOrderRequestData::setShippingAddress, payPalAddressDetailsData)
                .with(ApplePayOrderRequestData::setSaveOrderFlowActive, false)
                .with(ApplePayOrderRequestData::setIntent, defaultPayPalConfigurationService.getPayPalIntent())
                .with(ApplePayOrderRequestData::setStoredCredential, selectStoredCredentialStrategy(ONE_TIME_PAYMENT).getStoredCredential())
                .with(ApplePayOrderRequestData::setName, applePayShippingAddress.getGivenName() + " " + applePayShippingAddress.getFamilyName())
                .with(ApplePayOrderRequestData::setEmailAddress, applePayShippingAddress.getEmailAddress())
                .with(ApplePayOrderRequestData::setPhoneNumber, applePayShippingAddress.getPhoneNumber())
                .with(ApplePayOrderRequestData::setPickUpInStore, isPickupInStore())
                .build();
        orderRequestDataConverter.convert(cartService.getSessionCart(), applePayOrderRequestData);

        final String orderId = payPalPaymentService.createOrder(applePayOrderRequestData);
        payPalCartService.setPayPalOrderIdForCurrentCart(orderId);
        return orderId;
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscriptionForExpressCheckout(
            final PayPalOrderDetailsData orderDetails, String paymentMethodType) {
        final CreditCardType paymentType = setCreditCardType(paymentMethodType);
        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
                .with(CCPaymentInfoData::setPayPalOrderId, orderDetails.getOrderId())
                .with(CCPaymentInfoData::setSubscriptionId, PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER)
                .with(CCPaymentInfoData::setPayerId, orderDetails.getBuyerId())
                .with(CCPaymentInfoData::setAccountHolderName, orderDetails.getBuyer())
                .with(CCPaymentInfoData::setSaved, false)
                .with(CCPaymentInfoData::setPayerEmail, orderDetails.getShippingAddress().getEmail())
                .with(CCPaymentInfoData::setDefaultPaymentInfo, false)
                .with(CCPaymentInfoData::setCardType, paymentType.name())
                .build();

        final CustomerModel customer = getCurrentUserForCheckout();
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
                .createPayPalPaymentSubscription(customer, paymentInfoData,
                        createBillingAddress(orderDetails.getBillingAddress(), customer), null);
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public CCPaymentInfoData createPayPalPaymentSubscriptionForMarkCheckout(final String orderId,
                                                                            CreditCardType paymentType, String cardHolderName, boolean isShouldBeSaved) {

        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
                .with(CCPaymentInfoData::setPayPalOrderId, orderId)
                .with(CCPaymentInfoData::setSubscriptionId, PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER)
                .with(CCPaymentInfoData::setAccountHolderName, cardHolderName)
                .with(CCPaymentInfoData::setSaved, false)
                .with(CCPaymentInfoData::setDefaultPaymentInfo, false)
                .with(CCPaymentInfoData::setCardType, paymentType.name())
                .with(CCPaymentInfoData::setShouldBeSaved, isShouldBeSaved)
                .build();

        final AddressModel billingAddress = getModelService().create(AddressModel.class);
        if (!isPickupInStore()) {
            getAddressReversePopulator()
                    .populate(getCartFacade().getSessionCart().getDeliveryAddress(), billingAddress);
        }
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
                .createPayPalPaymentSubscription(getCurrentUserForCheckout(), paymentInfoData, billingAddress, null);
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }


    public PayPalCreditCardPaymentInfoModel createPayPalPaymentSubscription(final PayPalSetupTokenResponse payPalSetupTokenResponse,
                                                                            boolean savePaymentInfo,
                                                                            PayPalCreditCardPaymentInfoModel dumbPaymentInfo,
                                                                            CustomerModel customerModel) {
        final PayPalData payPalData = payPalSetupTokenResponse.getPaymentSource().getPaypal();

        if (hasNoDeliveryMode()) {
            setCheapestDeliveryModeForCheckout();
        }
        if (savePaymentInfo) {
            payPalCustomerAccountService.removeDuplicatePaymentMethod(payPalData.getEmailAddress());
        }
        final boolean shouldBeDefault = userFacade.getCCPaymentInfos(true)
                .stream().noneMatch(CCPaymentInfoData::isDefaultPaymentInfo);

        final CCPaymentInfoData paymentInfoData = GenericBuilder.of(CCPaymentInfoData::new)
                .with(CCPaymentInfoData::setPayerId, payPalData.getPayerId())
                .with(CCPaymentInfoData::setAccountHolderName, Optional.ofNullable(payPalData.getName())
                        .map(NameData::getFullName).orElse(Strings.EMPTY))
                .with(CCPaymentInfoData::setPayerEmail, payPalData.getEmailAddress())
                .with(CCPaymentInfoData::setPayPalOrderId, PAYPAL_ORDER_ID_PLACEHOLDER)
                .with(CCPaymentInfoData::setSubscriptionId, payPalSetupTokenResponse.getId())
                .with(CCPaymentInfoData::setSaved, savePaymentInfo)
                .with(CCPaymentInfoData::setDefaultPaymentInfo, shouldBeDefault && savePaymentInfo)
                .with(CCPaymentInfoData::setCardType, PAYPAL_PAYMENT_METHOD)
                .with(CCPaymentInfoData::setPMCustomerVaultId, payPalSetupTokenResponse.getCustomer().getId())
                .build();

        Optional<AddressPortableData> addressData = Optional.ofNullable(payPalData.getAddress());
        PayPalAddressDetailsData payPalAddressDetailsData;
        if (addressData.isPresent()) {
            AddressPortableData addressPortableData = addressData.get();
            payPalAddressDetailsData = GenericBuilder.of(PayPalAddressDetailsData::new)
                    .with(PayPalAddressDetailsData::setLine1, addressPortableData.getAddressLine1())
//                     should be uncommented after we start to receive again fields below from the PayPal API
//                    .with(PayPalAddressDetailsData::setLine2, addressPortableData.getAddressLine2())
                    .with(PayPalAddressDetailsData::setPostalCode, addressPortableData.getPostalCode())
                    .with(PayPalAddressDetailsData::setCountryCode, addressPortableData.getCountryCode())
//                     should be uncommented after we start to receive again fields below from the PayPal API
//                    .with(PayPalAddressDetailsData::setCity, addressPortableData.getAdminArea2())
//                    .with(PayPalAddressDetailsData::setRegion, addressPortableData.getAdminArea1())
                    .build();
        } else {
            payPalAddressDetailsData = GenericBuilder.of(PayPalAddressDetailsData::new).build();
        }

        return payPalCustomerAccountService
                .createPayPalPaymentSubscription(customerModel, paymentInfoData,
                        createBillingAddress(payPalAddressDetailsData, customerModel), dumbPaymentInfo);
    }


    @Override
    public CCPaymentInfoData updatePayPalPaymentSubscription(PayPalOrderDetailsData orderDetailsData,
                                                             CCPaymentInfoData paymentInfoData) {
        paymentInfoData.setPayerId(orderDetailsData.getBuyerId());
        if (paymentInfoData.getPayerEmail() == null) {
            paymentInfoData.setPayerEmail(orderDetailsData.getShippingAddress().getEmail());
        }

        final PaymentSourceData payment = orderDetailsData.getPayment();
        if (payment != null && payment.getCard() != null) {
            paymentInfoData.setCardNumber("************".concat(payment.getCard().getLastDigits()));
            paymentInfoData.setCardType(payment.getCard().getBrand());
        }

        final PayPalAddressDetailsData billingAddress = Optional.ofNullable(orderDetailsData.getBillingAddress())
                .orElse(orderDetailsData.getShippingAddress());


        final CreditCardPaymentInfoModel paymentInfoModel = getCustomerAccountService()
                .getCreditCardPaymentInfoForCode(getCurrentUserForCheckout(), paymentInfoData.getId());
        final CustomerModel customer = getCurrentUserForCheckout();
        final PayPalCreditCardPaymentInfoModel payPalPaymentSubscription = payPalCustomerAccountService
                .updatePayPalPaymentSubscription(customer, paymentInfoData,
                        createBillingAddress(billingAddress, customer),
                        (PayPalCreditCardPaymentInfoModel) paymentInfoModel);
        return getCreditCardPaymentInfoConverter().convert(payPalPaymentSubscription);
    }

    @Override
    public void addAddressForNewUser(final PayPalConnectAddressData addressData, final String payerId) {
        final AddressModel newAddress = getModelService().create(AddressModel.class);

        final CountryModel countryModel = getCommonI18NService().getCountry(addressData.getCountry());

        final CustomerModel customer = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);

        if (StringUtils.isNotBlank(addressData.getRegion())) {
            final String regionIsoCode = String.format("%s-%s", countryModel.getIsocode(), addressData.getRegion());
            final RegionModel regionModel = getCommonI18NService().getRegion(countryModel, regionIsoCode);
            newAddress.setRegion(regionModel);
            getModelService().save(regionModel);
        }

        final List<String> fullName = Arrays.asList(customer.getName().split(SPACE));
        newAddress.setFirstname(fullName.stream().findFirst().orElse(EMPTY));
        newAddress.setLastname(fullName.stream().skip(1).collect(Collectors.joining(" ")));
        newAddress.setEmail(customer.getContactEmail());
        newAddress.setCountry(countryModel);
        newAddress.setTown(addressData.getLocality());
        newAddress.setLine1(addressData.getStreet_address());
        newAddress.setPostalcode(addressData.getPostal_code());
        newAddress.setShippingAddress(true);

        customer.setDefaultShipmentAddress(newAddress);
        customer.setDefaultPaymentAddress(newAddress);

        getCustomerAccountService().saveAddressEntry(customer, newAddress);
    }

    @Override
    public boolean authorizePayment(final CartModel cartModel) {
        final CreditCardPaymentInfoModel creditCardPaymentInfoModel = cartModel == null ? null
                : (CreditCardPaymentInfoModel) cartModel.getPaymentInfo();
        if (creditCardPaymentInfoModel != null
                && StringUtils.isNotBlank(creditCardPaymentInfoModel.getSubscriptionId())) {
            final CommerceCheckoutParameter parameter = createCommerceCheckoutParameter(cartModel, true);
            parameter.setPaymentProvider(getPaymentProvider());
            final PaymentTransactionEntryModel paymentTransactionEntryModel = getCommerceCheckoutService()
                    .authorizePayment(parameter);
            return paymentTransactionEntryModel != null
                    && (TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())
                    || TransactionStatus.REVIEW.name().equals(paymentTransactionEntryModel.getTransactionStatus()));
        }
        return false;
    }

    @Override
    public OrderData placeOrderByCart(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            if (orderModel != null) {
                return getOrderConverter().convert(orderModel);
            }
        }
        return null;
    }

    @Override
    protected void afterPlaceOrder(CartModel cartModel, OrderModel orderModel) {
        if (orderModel != null) {
            getModelService().remove(cartModel);
            getModelService().refresh(orderModel);
        }
    }

    @Override
    public void prepareCartForCheckout(final CartModel cartModel) {
        if (cartModel != null) {
            getCommerceCheckoutService().calculateCart(createCommerceCheckoutParameter(cartModel, true));
        }
    }

    @Override
    public PayPalCheckoutData getPayPalCheckoutData(final String pageType) {
        final PayPalCheckoutData payPalCheckoutData = new PayPalCheckoutData();

        payPalCheckoutData.setPayPalPaymentMethod(generatePayPalMethodData());
        payPalCheckoutData.setCreditCardPaymentMethod(generateCreditCardPaymentMethod());

        return payPalCheckoutData;
    }

    @Override
    public CreditCardType setCreditCardType(String fundingSource) {
        CreditCardType paymentType;
        if (CreditCardType.PAYPAL.name().equalsIgnoreCase(fundingSource)
                || CreditCardType.PAYLATER.name().equalsIgnoreCase(fundingSource)
                || CreditCardType.CREDIT.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.PAYPAL;
        } else if (CreditCardType.VENMO.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.VENMO;
        } else if (CreditCardType.CARD.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.CARD;
        } else if (CreditCardType.PAYPAL_HOSTED_FIELDS_CARD.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.PAYPAL_HOSTED_FIELDS_CARD;
        } else if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(fundingSource)) {
            paymentType = CreditCardType.APPLEPAY;
        } else {
            paymentType = CreditCardType.LOCAL_PAYMENT;
        }

        return paymentType;
    }

    @Override
    public List<CCPaymentInfoData> getCCPaymentInfos(final boolean saved, final CustomerModel currentCustomer) {
        final List<CreditCardPaymentInfoModel> creditCards = getCustomerAccountService().getCreditCardPaymentInfos(currentCustomer, saved);
        final List<CCPaymentInfoData> ccPaymentInfos = new ArrayList<CCPaymentInfoData>();
        final PaymentInfoModel defaultPaymentInfoModel = currentCustomer.getDefaultPaymentInfo();
        for (final CreditCardPaymentInfoModel ccPaymentInfoModel : creditCards) {
            final CCPaymentInfoData paymentInfoData = getCreditCardPaymentInfoConverter().convert(ccPaymentInfoModel);
            if (paymentInfoData != null && ccPaymentInfoModel.equals(defaultPaymentInfoModel)) {
                paymentInfoData.setDefaultPaymentInfo(true);
                ccPaymentInfos.add(0, paymentInfoData);
            } else {
                ccPaymentInfos.add(paymentInfoData);
            }
        }
        return ccPaymentInfos;
    }

    @Override
    public List<CCPaymentInfoData> getAvailableCCPaymentInfosForCurrentUser() {
        return userFacade.getCCPaymentInfos(true)
                .stream()
                .filter(ccPaymentInfoData -> isHostedFieldsPaymentMethodAvailable(ccPaymentInfoData) || isPayPalPaymentMethodAvailable(ccPaymentInfoData))
                .toList();
    }

    private boolean isHostedFieldsPaymentMethodAvailable(CCPaymentInfoData paymentInfoData) {
        return PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS.equals(paymentInfoData.getPaymentProvider()) && defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled();
    }

    private boolean isPayPalPaymentMethodAvailable(CCPaymentInfoData paymentInfoData) {
        return PayPalPaymentProvider.PAYPAL.equals(paymentInfoData.getPaymentProvider()) && defaultPayPalConfigurationService.isPayPalVaultEnabled();
    }

    @Override
    public CCPaymentInfoData getCCPaymentInfoByOrderId(String payPayOrderId) {

        final Optional<PayPalCreditCardPaymentInfoModel> paymentInfo =
                payPalPaymentInfoService.getPaymentInfoByOrderId(payPayOrderId);

        if (paymentInfo.isPresent()) {
            return getCreditCardPaymentInfoConverter().convert(paymentInfo.get());
        } else {
            throw new PayPalPaymentInfoNotFoundException();
        }
    }

    @Override
    public ExpressCheckoutResult performExpressCheckout() {
        if (isExpressCheckoutEnabledForStore()) {
            final ExpressCheckoutResult expressCheckoutDeliveryResult = getExpressCheckoutDeliveryResult();
            if (expressCheckoutDeliveryResult != null) {
                return expressCheckoutDeliveryResult;
            }
            if (!setDefaultPaymentInfoForCheckout()) {
                return ExpressCheckoutResult.ERROR_PAYMENT_INFO;
            }
            final CartModel cartModel = getCart();
            if ((isPaymentMethodAndShippingAddressSaved(getCartFacade().getSessionCart())) && (cartModel.getPaymentInfo().getItemtype()
                    .equals(PAYPAL_CREDIT_CARD_PAYMENT_INFO))) {
                ((PayPalCreditCardPaymentInfoModel) cartModel.getPaymentInfo()).setPayPalOrderId(PAYPAL_ORDER_ID_PLACEHOLDER);
                ((PayPalCreditCardPaymentInfoModel) cartModel.getPaymentInfo()).setSubscriptionId(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);
                getModelService().save(cartModel.getPaymentInfo());
                getModelService().refresh(cartModel);
            }
            return ExpressCheckoutResult.SUCCESS;
        }
        return ExpressCheckoutResult.ERROR_NOT_AVAILABLE;
    }

    @Override
    public boolean authorizePayment(final String securityCode) {
        final CartModel cartModel = getCart();
        if (cartModel == null) {
            return false;
        }
        final PaymentInfoModel paymentInfoModel = cartModel.getPaymentInfo();
        return paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel creditCardPaymentInfoModel
                && creditCardPaymentInfoModel.isSaveOrderFlowActive()
                ? payPalPaymentService.savePayPalOrder((PayPalCreditCardPaymentInfoModel) paymentInfoModel, cartModel)
                : super.authorizePayment(securityCode);
    }

    @Override
    public boolean isPayPalAddressValid(final AddressData addressData) {
        final AddressVerificationResult<AddressVerificationDecision> verificationResult = addressVerificationFacade.verifyAddressData(addressData);

        if (!isCountrySupported(addressData.getCountry())) {
            verificationResult.setDecision(AddressVerificationDecision.REJECT);
        }

        return AddressVerificationDecision.ACCEPT.getDecisionString()
                .equals(verificationResult.getDecision().getDecisionString());
    }

    @Override
    public List<ApplePayLineItem> getApplePayLineItemsFromCart() {
        CartData cartData = getCartFacade().getSessionCart();

        ApplePayLineItem totalLineItem = new ApplePayLineItem();
        ApplePayLineItem subTotalLineItem = new ApplePayLineItem();
        ApplePayLineItem shippingLineItem = new ApplePayLineItem();
        ApplePayLineItem taxLineItem = new ApplePayLineItem();

        totalLineItem.setLabel(getBaseStoreService().getCurrentBaseStore().getName());
        subTotalLineItem.setLabel(SUBTOTAL);
        shippingLineItem.setLabel(SHIPPING);
        taxLineItem.setLabel(TAX);

        totalLineItem.setType(FINAL_PRICE_TYPE);
        subTotalLineItem.setType(FINAL_PRICE_TYPE);
        shippingLineItem.setType(FINAL_PRICE_TYPE);
        taxLineItem.setType(FINAL_PRICE_TYPE);

        totalLineItem.setAmount(getStringValueFromPriceData(cartData.getTotalPriceWithTax()));
        subTotalLineItem.setAmount(getStringValueFromPriceData(cartData.getSubTotal()));
        shippingLineItem.setAmount(getStringValueFromPriceData(cartData.getDeliveryCost()));
        taxLineItem.setAmount(getStringValueFromPriceData(cartData.getTotalTax()));

        return Arrays.asList(totalLineItem,
                subTotalLineItem, shippingLineItem, taxLineItem);
    }

    @Override
    public ApplePaySessionDetailsData createApplePaySessionDetails(String flow) {
        ApplePaySessionDetailsData applePaySessionDetailsData = new ApplePaySessionDetailsData();
        CartData cartData = getCartFacade().getSessionCart();

        applePaySessionDetailsData.setCurrency(cartData.getTotalPrice().getCurrencyIso());
        applePaySessionDetailsData.setLocale(commonI18NService.getCurrentLanguage().getIsocode());
        applePaySessionDetailsData.setLineItems(getApplePayLineItemsFromCart());
        applePaySessionDetailsData.setShippingContact(getShippingContact(flow));

        return applePaySessionDetailsData;
    }

    @Override
    public void updateApplePayShippingAddress(ApplePayAddressDetailsData applePayAddressDetailsData) {
        final AddressData addressData = Optional.ofNullable(applePayAddressDetailsData.getId())
                .map(id -> getAddressDataForId(id, false))
                .orElseGet(() -> applePayAddressDataConverter.convert(applePayAddressDetailsData));

        if (!userFacade.isAnonymousUser()) {
            userFacade.addAddress(addressData);
        }
        setDeliveryAddress(addressData);
        setCheapestDeliveryModeForCheckout();
        prepareCartForCheckout();
    }

    private ApplePayAddressDetailsData getShippingContact(String flow) {
        ApplePayAddressDetailsData applePayAddressDetailsData = new ApplePayAddressDetailsData();
        Optional<AddressData> defaultShippingAddress = Optional.ofNullable(getCustomerFacade().getCurrentCustomer()).map(CustomerData::getDefaultShippingAddress);
        Optional<AddressData> chosenShippingAddress = Optional.ofNullable(getCartFacade().getSessionCart()).map(CartData::getDeliveryAddress);

        if (EXPRESS_CHECKOUT.equals(flow)) {
            defaultShippingAddress.ifPresent(addressData -> applePayAddressDataReversePopulator.populate(addressData, applePayAddressDetailsData));
        } else {
            chosenShippingAddress.ifPresent(addressData -> applePayAddressDataReversePopulator.populate(addressData, applePayAddressDetailsData));
        }
        return applePayAddressDetailsData;
    }

    private static String getStringValueFromPriceData(PriceData priceData) {
        return String.valueOf(Optional.ofNullable(priceData)
                .map(PriceData::getValue)
                .orElse(BigDecimal.ZERO));
    }

    private void removeDeliveryAddressFromCart() {
        final AddressData previousSelectedAddress = getCheckoutCart().getDeliveryAddress();
        if (Objects.nonNull(previousSelectedAddress) && !previousSelectedAddress.isVisibleInAddressBook()) {
            userFacade.removeAddress(previousSelectedAddress);
        }
        removeDeliveryAddress();
    }

    protected void updatePayPalOrder(final PayPalOrderDetailsData orderDetailsData) {
        final BigDecimal payPalOrderAmountDouble = BigDecimal.valueOf(Double.parseDouble(orderDetailsData.getAmount()));

        CartModel currentCart = cartService.getSessionCart();
        final BigDecimal cartTotalPrice = breakdownCalculationService.calculateTotalAmount(currentCart);
        if (!cartTotalPrice.equals(payPalOrderAmountDouble)) {
            PayPalOrderRequestData orderRequestData = orderRequestDataConverter.convert(currentCart);
            payPalPaymentService.updateOrderAmountDetails(orderRequestData);
        }
    }

    protected boolean isCountrySupported(CountryData countryData) {
        return getCountries(CountryType.SHIPPING)
                .stream()
                .anyMatch(country -> country.getName().equalsIgnoreCase(countryData.getName())
                        || country.getIsocode().equalsIgnoreCase(countryData.getIsocode()));
    }

    private AddressModel createBillingAddress(final PayPalAddressDetailsData addressDetailsData, final CustomerModel customer) {
        final AddressModel billingAddress = getModelService().create(AddressModel.class);
        final AddressData billingAddressData = createAddressDataFromPayPalAddress(addressDetailsData, false, customer);
        getAddressReversePopulator().populate(billingAddressData, billingAddress);
        billingAddress.setEmail(billingAddressData.getEmail());
        return billingAddress;
    }

    private AddressData createAddressDataFromPayPalAddress(final PayPalAddressDetailsData payPalAddressDetailsData,
                                                           boolean isShipping, final CustomerModel customer) {
        final AddressData addressData = new AddressData();
        payPalAddressDataPopulator.populate(payPalAddressDetailsData, addressData);

        final TitleModel title = customer.getTitle();
        if (title != null) {
            addressData.setTitle(title.getName());
            addressData.setTitleCode(title.getCode());
        }
        final boolean hasNoDefaultAddress = userFacade.getDefaultAddress() == null && isShipping;
        addressData.setDefaultAddress(hasNoDefaultAddress);
        addressData.setVisibleInAddressBook(hasNoDefaultAddress);
        addressData.setShippingAddress(isShipping);
        addressData.setBillingAddress(!isShipping);
        return addressData;
    }

    private PayPalPaymentMethodData generatePayPalMethodData() {
        final PayPalPaymentMethodData paymentMethodData = new PayPalPaymentMethodData();
        paymentMethodData.setPayPalExpressEnabled(defaultPayPalConfigurationService.isExpressCheckoutEnabled());
        paymentMethodData.setIsPayPalPaymentAmountLimited(payPalPaymentInfoService.isSavedPayPalAccountsAmountLimited());
        paymentMethodData.setIsChangePaymentButtonActive(payPalPaymentInfoService.isChangePaymentButtonActive());
        return paymentMethodData;
    }

    private CreditCardPaymentMethodData generateCreditCardPaymentMethod() {
        final CreditCardPaymentMethodData paymentMethodData = new CreditCardPaymentMethodData();
        paymentMethodData.setHostedFieldsEnable(defaultPayPalConfigurationService.isPayPalHostedFieldsEnabled());
        paymentMethodData.setVaultFlowEnable(defaultPayPalConfigurationService.isPayPalHostedFieldsVaultEnabled());
        paymentMethodData.setIsCreditCardsAmountLimited(payPalPaymentInfoService.isSavedCreditCardsAmountLimited());
        return paymentMethodData;
    }

    private boolean isPaymentMethodAndShippingAddressSaved(final CartData cartData) {
        final boolean hasPayment = hasSavedPayPalPayment() || hasActivePayPalPaymentInTheSession(cartData);
        final boolean hasShipping = userFacade.getDefaultAddress() != null || cartData.getDeliveryAddress() != null;
        return hasPayment && hasShipping;
    }

    private boolean hasSavedPayPalPayment() {
        return !userFacade.isAnonymousUser() &&
                userFacade.getCCPaymentInfos(true).stream().anyMatch(a -> PAYPAL.equals(a.getCardType()));
    }

    private static boolean hasActivePayPalPaymentInTheSession(final CartData cartData) {
        return cartData.getPaymentInfo() != null && PAYPAL.equals(cartData.getPaymentInfo().getCardType());
    }

    private static boolean isLocalPaymentFunding(String funding) {
        try {
            return funding.equalsIgnoreCase(LocalPaymentFunding.valueOf(funding.toUpperCase()).getCode());
        } catch (Exception e) {
            return false;
        }
    }

    public void setPayPalCustomerAccountService(PayPalCustomerAccountService payPalCustomerAccountService) {
        this.payPalCustomerAccountService = payPalCustomerAccountService;
    }

    public void setAddressVerificationFacade(AddressVerificationFacade addressVerificationFacade) {
        this.addressVerificationFacade = addressVerificationFacade;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setUserFacade(UserFacade userFacade) {
        this.userFacade = userFacade;
    }

    public void setPayPalAddressDataPopulator(
            Populator<PayPalAddressDetailsData, AddressData> payPalAddressDataPopulator) {
        this.payPalAddressDataPopulator = payPalAddressDataPopulator;
    }

    public void setPayPalAddressDataReversePopulator(
            Populator<AddressData, PayPalAddressDetailsData> payPalAddressDataReversePopulator) {
        this.payPalAddressDataReversePopulator = payPalAddressDataReversePopulator;
    }

    public void setApplePayAddressDataConverter(Converter<ApplePayAddressDetailsData, AddressData> applePayAddressDataConverter) {
        this.applePayAddressDataConverter = applePayAddressDataConverter;
    }

    public void setApplePayPayPalAddressDataPopulator(Populator<ApplePayAddressDetailsData, PayPalAddressDetailsData> applePayPayPalAddressDataPopulator) {
        this.applePayPayPalAddressDataPopulator = applePayPayPalAddressDataPopulator;
    }

    public void setApplePayAddressDataReversePopulator(Populator<AddressData, ApplePayAddressDetailsData> applePayAddressDataReversePopulator) {
        this.applePayAddressDataReversePopulator = applePayAddressDataReversePopulator;
    }

    @Override
    public CartService getCartService() {
        return cartService;
    }

    @Override
    public void setCartService(final CartService cartService) {
        this.cartService = cartService;
    }

    public void setPayPalCustomerAccountDao(PayPalCustomerAccountDao payPalCustomerAccountDao) {
        this.payPalCustomerAccountDao = payPalCustomerAccountDao;
    }

    @Override
    public CustomerAccountService getCustomerAccountService() {
        return customerAccountService;
    }

    @Override
    public void setCustomerAccountService(CustomerAccountService customerAccountService) {
        this.customerAccountService = customerAccountService;
    }

    @Override
    public CommonI18NService getCommonI18NService() {
        return commonI18NService;
    }

    @Override
    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

    public CustomerFacade getCustomerFacade() {
        return customerFacade;
    }

    public void setCustomerFacade(CustomerFacade customerFacade) {
        this.customerFacade = customerFacade;
    }

    @Override
    public UserService getUserService() {
        return userService;
    }

    @Override
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setPayPalCartService(PayPalCartService payPalCartService) {
        this.payPalCartService = payPalCartService;
    }

    public void setStoredCredentialStrategies(List<StoredCredentialStrategy> storedCredentialStrategies) {
        this.storedCredentialStrategies = storedCredentialStrategies;
    }

    public void setEmptyStoredCredentialStrategy(StoredCredentialStrategy emptyStoredCredentialStrategy) {
        this.emptyStoredCredentialStrategy = emptyStoredCredentialStrategy;
    }

    public void setPayPalPaymentInfoService(PayPalPaymentInfoService payPalPaymentInfoService) {
        this.payPalPaymentInfoService = payPalPaymentInfoService;
    }

    public void setOrderRequestDataConverter(Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter) {
        this.orderRequestDataConverter = orderRequestDataConverter;
    }

    public void setVaultOrderRequestDataConverter(Converter<AbstractOrderModel, PayPalVaultOrderRequestData> vaultOrderRequestDataConverter) {
        this.vaultOrderRequestDataConverter = vaultOrderRequestDataConverter;
    }

    public void setBreakdownCalculationService(BreakdownCalculationService breakdownCalculationService) {
        this.breakdownCalculationService = breakdownCalculationService;
    }
}
