package com.paypal.hybris.core.util.builder;

import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.function.BiConsumer;
import java.util.function.Supplier;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.ignoreStubs;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

public class GenericBuilderTest {

    private static final String TEST_VALUE = "TestValue";
    private static final String FIRST = "First";
    private static final String SECOND = "Second";

    private static class TestObject {
        private String property;

        public void setProperty(String property) {
            this.property = property;
        }

        public String getProperty() {
            return property;
        }
    }

    @Mock
    private Supplier<TestObject> supplier;

    @Mock
    private BiConsumer<TestObject, String> consumer;

    @InjectMocks
    private GenericBuilder<TestObject> builder;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        when(supplier.get()).thenReturn(new TestObject());
    }

    @Test
    public void shouldBuildObjectUsingSupplier() {
        builder = new GenericBuilder<>(supplier);

        TestObject result = builder.build();

        verify(supplier).get();
        verifyNoMoreInteractions(ignoreStubs(supplier));
        assertNotNull(result);
    }

    @Test
    public void shouldApplyModifierToBuiltObject() {
        ArgumentCaptor<TestObject> captor = ArgumentCaptor.forClass(TestObject.class);

        GenericBuilder<TestObject> builder = GenericBuilder.of(supplier)
                .with(consumer, TEST_VALUE);

        TestObject result = builder.build();

        verify(consumer).accept(captor.capture(), eq(TEST_VALUE));
        TestObject capturedObject = captor.getValue();
        assertEquals(result, capturedObject);
    }

    @Test
    public void shouldAllowChainingOfModifiers() {
        GenericBuilder<TestObject> builder = GenericBuilder.of(supplier);

        builder.with((obj, val) -> obj.setProperty(val), FIRST)
                .with((obj, val) -> obj.setProperty(val), SECOND);

        TestObject result = builder.build();

        assertEquals(SECOND, result.getProperty());
    }
}