package com.paypal.hybris.core.util;

import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalAttributesData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalVaultData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.product.data.PriceData;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

import java.math.BigDecimal;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

@UnitTest
@RunWith(MockitoJUnitRunner.StrictStubs.class)
public class PayPalCommandsUtilTest {

    private static final String CARD_CUSTOMER_ID = "cardCustomerId";
    private static final String PAYPAL_CUSTOMER_ID = "payPalCustomerId";
    private static final String CARD_VAULT_ID = "cardVaultId";
    private static final String PAY_PAL_VAULT_ID = "payPalVaultId";
    private static final String USD = "USD";
    private static final String ITEMS_AMOUNT = "10.00";
    private static final String TAX_AMOUNT = "9.00";
    private static final String DEFAULT_ZERO_VALUE = "0.0";
    private PayPalOrderResponseData cardOrderResponseData;
    private PayPalOrderResponseData payPalOrderResponseData;

    @Before
    public void setUp() throws Exception {
        cardOrderResponseData = new PayPalOrderResponseData();
        PaymentSourceData cardPaymentSource = new PaymentSourceData();
        CardData cardData = new CardData();
        PayPalAttributesData cardAttributesData = new PayPalAttributesData();
        PayPalVaultData cardVaultData = new PayPalVaultData();
        PayPalCustomerData cardCustomerData = new PayPalCustomerData();

        cardVaultData.setId(CARD_VAULT_ID);
        cardVaultData.setCustomer(cardCustomerData);
        cardCustomerData.setId(CARD_CUSTOMER_ID);
        cardAttributesData.setVault(cardVaultData);
        cardData.setAttributes(cardAttributesData);
        cardPaymentSource.setCard(cardData);
        cardOrderResponseData.setPaymentSource(cardPaymentSource);

        payPalOrderResponseData = new PayPalOrderResponseData();
        PaymentSourceData payPalPaymentSource = new PaymentSourceData();
        PayPalData payPalData = new PayPalData();
        PayPalAttributesData payPalAttributesData = new PayPalAttributesData();
        PayPalVaultData payPalVaultData = new PayPalVaultData();
        PayPalCustomerData payPalCustomerData = new PayPalCustomerData();

        payPalVaultData.setId(PAY_PAL_VAULT_ID);
        payPalVaultData.setCustomer(payPalCustomerData);
        payPalCustomerData.setId(PAYPAL_CUSTOMER_ID);
        payPalAttributesData.setVault(payPalVaultData);
        payPalData.setAttributes(payPalAttributesData);
        payPalPaymentSource.setPaypal(payPalData);
        payPalOrderResponseData.setPaymentSource(payPalPaymentSource);
    }

    @Test
    public void shouldReturnNullCustomerIdWhenPayPalOrderResponseDataIsEmpty() {
        String customerId = PayPalCommandsUtil.getCustomerIdFromPayPalOrderResponseData(new PayPalOrderResponseData());
        assertNull(customerId);
    }

    @Test
    public void shouldReturnNullVaultIdWhenPayPalOrderResponseDataIsEmpty() {
        String customerId = PayPalCommandsUtil.getVaultIdFromPayPalOrderResponseData(new PayPalOrderResponseData());
        assertNull(customerId);
    }

    @Test
    public void shouldReturnCustomerIdWhenCardPresent() {
        String customerId = PayPalCommandsUtil.getCustomerIdFromPayPalOrderResponseData(cardOrderResponseData);
        assertEquals(CARD_CUSTOMER_ID, customerId);
    }

    @Test
    public void shouldReturnCustomerIdWhenPayPalPresent() {
        String customerId = PayPalCommandsUtil.getCustomerIdFromPayPalOrderResponseData(payPalOrderResponseData);
        assertEquals(PAYPAL_CUSTOMER_ID, customerId);
    }

    @Test
    public void shouldReturnVaultIdWhenCardPresent() {
        String customerId = PayPalCommandsUtil.getVaultIdFromPayPalOrderResponseData(cardOrderResponseData);
        assertEquals(CARD_VAULT_ID, customerId);
    }

    @Test
    public void shouldReturnVaultIdWhenPayPalPresent() {
        String customerId = PayPalCommandsUtil.getVaultIdFromPayPalOrderResponseData(payPalOrderResponseData);
        assertEquals(PAY_PAL_VAULT_ID, customerId);
    }

    @Test
    public void shouldReturnValueFromSimpleAmountWhenSimpleAmountIsNotNull() {
        SimpleAmount simpleAmount = new SimpleAmount();
        simpleAmount.setValue(ITEMS_AMOUNT);

        String value = PayPalCommandsUtil.getPriceValue(simpleAmount);

        assertEquals(ITEMS_AMOUNT, value);
    }

    @Test
    public void shouldReturnZeroValueWhenSimpleAmountIsNull() {
        SimpleAmount simpleAmount = null;
        String value = PayPalCommandsUtil.getPriceValue(simpleAmount);

        assertEquals(DEFAULT_ZERO_VALUE, value);
    }

    @Test
    public void shouldCreateSimpleAmount() {
        SimpleAmount simpleAmount = PayPalCommandsUtil.createAmount(9d, USD);

        assertNotNull(simpleAmount);
        assertEquals(TAX_AMOUNT, simpleAmount.getValue());
        assertEquals(USD, simpleAmount.getCurrencyCode());
    }

    @Test
    public void shouldCreatePriceData() {
        SimpleAmount priceData = PayPalCommandsUtil.createAmount(BigDecimal.TEN, USD);

        assertNotNull(priceData);
        assertEquals(ITEMS_AMOUNT, priceData.getValue());
        assertEquals(USD, priceData.getCurrencyCode());
    }

    @Test
    public void shouldReturnValueFromPriceDataWhenPriceDataIsNotNull() {
        PriceData amount = new PriceData();
        amount.setValue(BigDecimal.ONE);

        String value = PayPalCommandsUtil.getPriceValue(amount);

        assertEquals(BigDecimal.ONE.toString(), value);
    }

    @Test
    public void shouldReturnZeroValueWhenPriceDataIsNull() {
        PriceData amount = null;
        String value = PayPalCommandsUtil.getPriceValue(amount);

        assertEquals(DEFAULT_ZERO_VALUE, value);
    }
}
