package com.paypal.hybris.core.strategy.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commerceservices.strategies.GenerateMerchantTransactionCodeStrategy;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Currency;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalVaultedPaymentStrategyTest {

    private static final String CODE = "CODE";
    private static final String TEST_FLOW = "testFlow";
    private static final String USD = "USD";

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private CartModel cartModel;

    @Mock
    private PaymentTransactionEntryModel paymentTransactionEntry;

    @Mock
    private PaymentTransactionModel paymentTransaction;

    @Mock
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel;

    @Mock
    private PaymentInfoModel paymentInfoModel;

    @Mock
    private I18NService i18NService;

    @Mock
    private Currency currency;

    @Mock
    private ModelService modelService;

    @Mock
    private PayPalPaymentService payPalPaymentService;

    @Mock
    private GenerateMerchantTransactionCodeStrategy generateMerchantTransactionCodeStrategy;

    @InjectMocks
    private DefaultPayPalVaultedPaymentStrategy unit;

    private PayPalCommerceCheckoutParameter parameter;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        parameter = GenericBuilder.of(PayPalCommerceCheckoutParameter::new)
                .with(PayPalCommerceCheckoutParameter::setCart, cartModel)
                .with(PayPalCommerceCheckoutParameter::setAuthorizationAmount, BigDecimal.valueOf(123))
                .with(PayPalCommerceCheckoutParameter::setPaymentProvider, PaymentProvider.PAYPAL.getCode())
                .with(PayPalCommerceCheckoutParameter::setStoredCredentialFlow, TEST_FLOW)
                .build();

        when(payPalCreditCardPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
    }

    @Test
    public void shouldNotAuthorizePaymentAmountWhenPaymentInfoIsNotCreditCard() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        assertNull(unit.authorizePaymentAmount(parameter));
    }

    @Test
    public void shouldCapturePaymentAmount() {
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(cartModel.getCurrency().getIsocode()).thenReturn(USD);

        when(i18NService.getBestMatchingJavaCurrency(USD)).thenReturn(currency);
        when(generateMerchantTransactionCodeStrategy.generateCode(cartModel)).thenReturn(CODE);

        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(payPalPaymentService.createStubAuthorize(CODE, parameter.getAuthorizationAmount(), currency, parameter.getPaymentProvider(), payPalCreditCardPaymentInfoModel)).thenReturn(paymentTransactionEntry);
        when(paymentTransactionEntry.getPaymentTransaction()).thenReturn(paymentTransaction);

        PaymentTransactionEntryModel result = unit.authorizePaymentAmount(parameter);

        verify(payPalPaymentService).captureVaultedPayment(paymentTransaction, TEST_FLOW, PaymentProvider.PAYPAL_HOSTED_FIELDS);
        verify(modelService).saveAll(cartModel, paymentTransaction);

        assertNotNull(result);
    }

    @Test
    public void shouldAuthorizePaymentAmountWithStatusReview() {
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(cartModel.getCurrency().getIsocode()).thenReturn(USD);

        when(i18NService.getBestMatchingJavaCurrency(USD)).thenReturn(currency);
        when(generateMerchantTransactionCodeStrategy.generateCode(cartModel)).thenReturn(CODE);

        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);

        when(payPalPaymentService.authorizeVaultedPayment(CODE, parameter.getAuthorizationAmount(), currency, cartModel, parameter.getPaymentProvider(), TEST_FLOW, PaymentProvider.PAYPAL_HOSTED_FIELDS)).thenReturn(paymentTransactionEntry);

        when(paymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.REVIEW.name());
        when(paymentTransactionEntry.getPaymentTransaction()).thenReturn(paymentTransaction);

        PaymentTransactionEntryModel result = unit.authorizePaymentAmount(parameter);

        verify(modelService).saveAll(cartModel, paymentTransaction);

        assertNotNull(result);
    }

    @Test
    public void shouldAuthorizePaymentAmountWithoutStatusReviewOrAccepted() {
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(cartModel.getCurrency().getIsocode()).thenReturn(USD);

        when(i18NService.getBestMatchingJavaCurrency(USD)).thenReturn(currency);
        when(generateMerchantTransactionCodeStrategy.generateCode(cartModel)).thenReturn(CODE);

        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);

        when(payPalPaymentService.authorizeVaultedPayment(CODE, parameter.getAuthorizationAmount(), currency, cartModel, parameter.getPaymentProvider(), TEST_FLOW, PaymentProvider.PAYPAL_HOSTED_FIELDS)).thenReturn(paymentTransactionEntry);

        when(paymentTransactionEntry.getTransactionStatus()).thenReturn(TransactionStatus.ERROR.name());
        when(paymentTransactionEntry.getPaymentTransaction()).thenReturn(paymentTransaction);

        PaymentTransactionEntryModel result = unit.authorizePaymentAmount(parameter);

        assertNotNull(result);
    }
}
