/**
 *
 */
package com.paypal.hybris.core.strategy;

import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.impl.DefaultPayPalCommercePaymentAuthorizationStrategy;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.commerceservices.strategies.GenerateMerchantTransactionCodeStrategy;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.Currency;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultPayPalCommercePaymentAuthorizationStrategyTest {

    private static final String PROVIDER = "provider";
    private static final String USD = "USD";
    private static final String CODE = "code";
    private static final String ORDER_ID = "orderId";

    @Mock
    private ModelService modelService;

    @Mock
    private GenerateMerchantTransactionCodeStrategy codeStrategy;

    @Mock
    private I18NService i18nService;

    @Mock
    private CommerceCheckoutParameter parameter;

    @Mock
    private PaymentTransactionEntryModel entryModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;

    @Mock
    private CartModel cartModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private PaymentInfoModel infoModel;

    @Mock
    private BigDecimal amount;

    @Mock
    private PayPalPaymentService payPalPaymentService;

    @Mock
    private PaymentService paymentService;

    @InjectMocks
    DefaultPayPalCommercePaymentAuthorizationStrategy unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldNotAuthorizePaymentAmount() {
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(amount);
        when(cartModel.getPaymentInfo()).thenReturn(infoModel);

        Assert.assertNull(unit.authorizePaymentAmount(parameter));

    }

    @Test
    public void shouldAuthorizePaymentAmountWhenTransactionStatusIsNotAccepted() {
        final Currency currency = Currency.getInstance(USD);

        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(BigDecimal.ONE);
        when(parameter.getPaymentProvider()).thenReturn(PROVIDER);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(paymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);

        when(cartModel.getCurrency()).thenReturn(currencyModel);
        when(i18nService.getBestMatchingJavaCurrency(cartModel.getCurrency().getIsocode())).thenReturn(currency);

        when(codeStrategy.generateCode(cartModel)).thenReturn(CODE);

        when(payPalPaymentService.authorizePayment(CODE, BigDecimal.ONE, currency, cartModel, PROVIDER)).thenReturn(entryModel);
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);

        Assert.assertNotNull(unit.authorizePaymentAmount(parameter));
    }

    @Test
    public void shouldAuthorizePaymentAmountWhenTransactionStatusIsAccepted() {
        final Currency currency = Currency.getInstance(USD);

        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(BigDecimal.ONE);
        when(parameter.getPaymentProvider()).thenReturn(PROVIDER);
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        when(paymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);

        when(cartModel.getCurrency()).thenReturn(currencyModel);
        when(i18nService.getBestMatchingJavaCurrency(cartModel.getCurrency().getIsocode())).thenReturn(currency);

        when(codeStrategy.generateCode(cartModel)).thenReturn(CODE);

        when(payPalPaymentService.authorizePayment(CODE, BigDecimal.ONE, currency, cartModel, PROVIDER)).thenReturn(entryModel);
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);

        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        Assert.assertNotNull(unit.authorizePaymentAmount(parameter));

        verify(modelService).saveAll(cartModel, transactionModel);
    }

}
