/**
 *
 */
package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.dao.PaymentTransactionsDao;
import de.hybris.backoffice.AmountData;
import de.hybris.backoffice.PaypalWebhookData;
import de.hybris.backoffice.RelatedIdData;
import de.hybris.backoffice.ResourceData;
import de.hybris.backoffice.SupplementaryData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordercancel.OrderCancelEntry;
import de.hybris.platform.ordercancel.OrderCancelRecordsHandler;
import de.hybris.platform.ordercancel.OrderCancelRequest;
import de.hybris.platform.ordercancel.OrderCancelResponse;
import de.hybris.platform.ordercancel.OrderCancelResponse.ResponseStatus;
import de.hybris.platform.ordercancel.OrderStatusChangeStrategy;
import de.hybris.platform.ordercancel.exceptions.OrderCancelRecordsHandlerException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPaymentTransactionsServiceTest {

    private static final String TRANSACTION_ID = "transactionId";
    private static final String AUTHORIZATION_ID = "authorizationId";
    private static final String CAPTURE_ID = "captureId";
    private static final String TEST_REQUEST_ID = "testRequestId";
    private static final String RESOURCE_ID = "resourceId";

    private static final String SUCCESFULL_STATUS = "SUCCESFULL";
    private static final BigDecimal AMOUNT_TEN = BigDecimal.TEN;
    private static final BigDecimal AMOUNT_ZERO = BigDecimal.ZERO;

    @Mock
    private PaymentTransactionsDao transactionsDao;

    @Mock
    private ModelService modelService;

    @Mock
    private OrderCancelRecordsHandler orderCancelRecordsHandler;

    @Mock
    private OrderStatusChangeStrategy completeCancelStatusChangeStrategy;

    @Mock
    private DefaultPayPalPaymentService paypalPaymentService;

    @Mock
    private BusinessProcessService businessProcessService;

    @Mock
    private PaymentTransactionEntryModel transactionEntryModel;

    @Mock
    private OrderModel orderModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private PaymentTransactionEntryModel newTransactionEntryModel;

    @Mock
    private OrderCancelRequest request;

    @Mock
    private OrderCancelEntry orderCancelEntry;

    @Mock
    private AbstractOrderEntryModel abstractOrderEntryModel;

    @Mock
    private OrderModel order;

    @Mock
    private OrderEntryModel orderEntry;

    @Mock
    private OrderEntryModel orderEntry2;

    @Spy
    @InjectMocks
    private DefaultPaymentTransactionsService unit;

    private PaypalWebhookData webhookData;

    private ResourceData resource;

    private AmountData amountData;

    private RelatedIdData relatedIdData;

    private SupplementaryData supplementaryData;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        webhookData = new PaypalWebhookData();
        resource = new ResourceData();
        amountData = new AmountData();
        supplementaryData = new SupplementaryData();
        relatedIdData = new RelatedIdData();

        resource.setId(RESOURCE_ID);
        webhookData.setResource(resource);
        resource.setSupplementaryData(supplementaryData);
        resource.setAmount(amountData);
        supplementaryData.setRelatedIds(relatedIdData);
    }

    @Test
    public void getTransactionEntryByIdNotNull() {
        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(transactionEntryModel);

        assertNotNull(unit.getTransactionEntryById(TRANSACTION_ID));
    }

    @Test
    public void getTransactionEntryByIdIsNull() {

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(null);

        assertNull(unit.getTransactionEntryById(TRANSACTION_ID));
    }

    @Test
    public void voidPaymentTransactionWithTransactionEntryNull() {
        resource.setId(null);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(null);

        unit.doOnVoidPerformed(webhookData);

        verify(modelService, never()).save(transactionEntryModel);

    }

    @Test
    public void voidPaymentTransactionWithAuthorizeTypeTransactionEntryNotNull() {

        ArrayList<AbstractOrderEntryModel> orderEntries = new ArrayList<>();
        orderEntries.add(orderEntry);
        ArrayList<PaymentTransactionModel> paymentTransactions = new ArrayList<>();
        paymentTransactions.add(transactionModel);
        ArrayList<PaymentTransactionEntryModel> paymentTransactionEntries = new ArrayList<>();
        paymentTransactionEntries.add(transactionEntryModel);
        resource.setId(TRANSACTION_ID);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);
        when(order.getEntries()).thenReturn(orderEntries);
        when(orderEntry.getOrder()).thenReturn(order);
        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(order.getPaymentTransactions()).thenReturn(paymentTransactions);
        when(transactionModel.getEntries()).thenReturn(paymentTransactionEntries);

        when(paypalPaymentService.getCancelableTransactions(order)).thenReturn(paymentTransactionEntries);
        when(paypalPaymentService.getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CANCEL)).thenReturn("newCode");
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(newTransactionEntryModel);
        when(paypalPaymentService.isOrderCanceled(order)).thenReturn(true);

        unit.doOnVoidPerformed(webhookData);

        verify(modelService).save(transactionEntryModel);
        verify(modelService).save(order);
        verify(modelService).save(newTransactionEntryModel);
        verify(modelService).refresh(order);

    }

    @Test
    public void shouldCatchOrderCancelRecordsHandlerException() throws OrderCancelRecordsHandlerException {
        resource.setId(TRANSACTION_ID);
        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);

        doReturn(request).when(unit).buildCancelRequest(order);
        when(request.getOrder()).thenReturn(order);

        when(order.getEntries()).thenReturn(Collections.emptyList());


        when(orderCancelRecordsHandler.updateRecordEntry(any())).thenThrow(OrderCancelRecordsHandlerException.class);

        unit.doOnVoidPerformed(webhookData);

        verify(modelService).refresh(order);
        verify(modelService, never()).save(transactionEntryModel);
        verify(modelService, never()).save(order);
        verify(modelService, never()).save(newTransactionEntryModel);

    }

    @Test
    public void voidPaymentTransactionWithAuthorizeTypeTransactionEntryNotNullAndCompleteCancelStatusChangeStrategyNull() {

        unit.setCompleteCancelStatusChangeStrategy(null);

        ArrayList<PaymentTransactionModel> paymentTransactions = new ArrayList<>();
        paymentTransactions.add(transactionModel);
        ArrayList<PaymentTransactionEntryModel> paymentTransactionEntries = new ArrayList<>();
        paymentTransactionEntries.add(transactionEntryModel);
        resource.setId(TRANSACTION_ID);

        doReturn(request).when(unit).buildCancelRequest(order);
        when(request.getOrder()).thenReturn(order);

        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);
        when(order.getEntries()).thenReturn(Collections.emptyList());
        when(orderEntry.getOrder()).thenReturn(order);
        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(order.getPaymentTransactions()).thenReturn(paymentTransactions);
        when(transactionModel.getEntries()).thenReturn(paymentTransactionEntries);

        when(paypalPaymentService.getCancelableTransactions(order)).thenReturn(paymentTransactionEntries);
        when(paypalPaymentService.getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CANCEL)).thenReturn("newCode");
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(newTransactionEntryModel);
        when(paypalPaymentService.isOrderCanceled(order)).thenReturn(true);

        unit.doOnVoidPerformed(webhookData);

        verify(modelService).save(transactionEntryModel);
        verify(modelService).save(order);
        verify(modelService).save(newTransactionEntryModel);
        verify(modelService).refresh(order);

    }

    @Test
    public void voidPaymentTransactionWithAuthorizeTypeTransactionEntryNotNullAndHasLivingEntries() {

        ArrayList<PaymentTransactionModel> paymentTransactions = new ArrayList<>();
        paymentTransactions.add(transactionModel);
        ArrayList<PaymentTransactionEntryModel> paymentTransactionEntries = new ArrayList<>();
        paymentTransactionEntries.add(transactionEntryModel);
        resource.setId(TRANSACTION_ID);

        doReturn(request).when(unit).buildCancelRequest(order);
        when(request.getOrder()).thenReturn(order);


        when(transactionsDao.getTransactionEntryById(TRANSACTION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);
        when(order.getEntries()).thenReturn(Collections.emptyList());
        when(orderEntry.getOrder()).thenReturn(order);
        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(order.getPaymentTransactions()).thenReturn(paymentTransactions);
        when(transactionModel.getEntries()).thenReturn(paymentTransactionEntries);

        when(paypalPaymentService.getCancelableTransactions(order)).thenReturn(paymentTransactionEntries);
        when(paypalPaymentService.getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CANCEL)).thenReturn("newCode");
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(newTransactionEntryModel);
        when(paypalPaymentService.isOrderCanceled(order)).thenReturn(true);

        unit.doOnVoidPerformed(webhookData);

        verify(modelService).save(transactionEntryModel);
        verify(modelService).save(order);
        verify(modelService).save(newTransactionEntryModel);
        verify(modelService).refresh(order);

    }

    @Test
    public void shouldDoOnRefundPerformed() {
        amountData.setValue("123.11");

        when(transactionsDao.getTransactionEntryById(CAPTURE_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(order);
        when(transactionEntryModel.getAmount()).thenReturn(new BigDecimal(125));

        when(transactionEntryModel.getRequestId()).thenReturn(TEST_REQUEST_ID);
        when(order.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(transactionEntryModel));

        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.REFUND_PARTIAL);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getSubscriptionID()).thenReturn(TEST_REQUEST_ID);

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(newTransactionEntryModel);

        unit.doOnRefundPerformed(webhookData, CAPTURE_ID);

        verify(modelService).create(PaymentTransactionEntryModel.class);
        verify(modelService).saveAll(order, newTransactionEntryModel);

    }

    @Test
    public void makeInternalResponsePartialStatus() {
        List<OrderCancelEntry> orderCancelEntryList = new ArrayList<>();
        orderCancelEntryList.add(orderCancelEntry);

        when(request.isPartialCancel()).thenReturn(true);
        when(request.getOrder()).thenReturn(orderModel);
        when(request.getEntriesToCancel()).thenReturn(orderCancelEntryList);
        when(orderCancelEntry.getOrderEntry()).thenReturn(abstractOrderEntryModel);
        when(abstractOrderEntryModel.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, true, (String) null);

        OrderCancelResponse.ResponseStatus actual = response.getResponseStatus();
        OrderCancelResponse.ResponseStatus expected = OrderCancelResponse.ResponseStatus.partial;

        assertEquals(expected, actual);
    }

    @Test
    public void shouldNotCapturePerformedTransactionWhenEntryNull() {
        relatedIdData.setAuthorizationId(null);

        unit.doOnCapturePerformed(webhookData);
        verify(modelService, never()).save(transactionEntryModel);
    }

    @Test
    public void updatePaymentTransactionCaptureTransactionEntryNotNullTypeCapture() {
        relatedIdData.setAuthorizationId(AUTHORIZATION_ID);
        amountData.setValue("123.11");

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionsDao.getTransactionEntryById(AUTHORIZATION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionEntryModel.getAmount()).thenReturn(new BigDecimal(125));
        when(transactionModel.getOrder()).thenReturn(orderModel);
        when(transactionModel.getPlannedAmount()).thenReturn(new BigDecimal("100"));
        when(transactionModel.getEntries()).thenReturn(List.of(transactionEntryModel));

        unit.doOnCapturePerformed(webhookData);

        verify(modelService).saveAll(any(), any());
    }

    @Test
    public void updatePaymentTransactionCaptureTransactionEntryNotNullTypeWhenPartialCaptureAndOrderNotFullyCaptured() {
        relatedIdData.setAuthorizationId(AUTHORIZATION_ID);
        amountData.setValue("123.11");

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionsDao.getTransactionEntryById(AUTHORIZATION_ID)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionEntryModel.getAmount()).thenReturn(new BigDecimal(-125));
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(transactionModel.getOrder()).thenReturn(orderModel);
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.ZERO);
        when(transactionModel.getEntries()).thenReturn(List.of(transactionEntryModel));


        unit.doOnCapturePerformed(webhookData);

        verify(modelService).saveAll(any(), any());
    }

    @Test
    public void updatePaymentTransactionCaptureTransactionWhenCaptureTransactionEmpty() {
        relatedIdData.setAuthorizationId(AUTHORIZATION_ID);
        amountData.setValue("123.11");

        when(transactionsDao.getTransactionEntryById(AUTHORIZATION_ID)).thenReturn(transactionEntryModel);
        when(transactionsDao.getTransactionEntryById(RESOURCE_ID)).thenReturn(transactionEntryModel);


        unit.doOnCapturePerformed(webhookData);

        verify(modelService, never()).saveAll(any(), any());
    }

    @Test
    public void makeInternalResponseFullStatus() {
        when(request.isPartialCancel()).thenReturn(false);
        when(request.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, true, (String) null);

        OrderCancelResponse.ResponseStatus actual = response.getResponseStatus();
        OrderCancelResponse.ResponseStatus expected = OrderCancelResponse.ResponseStatus.full;

        assertEquals(expected, actual);
    }

    @Test
    public void shouldMakeInternalResponseErrorStatusWhenCancelIsNotSuccessful() {
        when(request.isPartialCancel()).thenReturn(false);
        when(request.getOrder()).thenReturn(orderModel);
        when(request.getEntriesToCancel()).thenReturn(List.of(orderCancelEntry));
        when(orderCancelEntry.getOrderEntry()).thenReturn(abstractOrderEntryModel);
        when(abstractOrderEntryModel.getOrder()).thenReturn(orderModel);

        OrderCancelResponse response = unit.makeInternalResponse(request, false,  null);

        assertEquals(ResponseStatus.error, response.getResponseStatus());
    }

    @Test
    public void buildCancelRequestWithNullOrder() {
        assertNull(unit.buildCancelRequest(null));
    }

    @Test
    public void buildCancelRequestWithNotNullOrder() {
        List<AbstractOrderEntryModel> abstractOrderEntryModelList = new ArrayList<>();
        abstractOrderEntryModelList.add(abstractOrderEntryModel);

        when(orderCancelEntry.getOrderEntry()).thenReturn(abstractOrderEntryModel);
        when(abstractOrderEntryModel.getOrder()).thenReturn(orderModel);
        when(orderModel.getEntries()).thenReturn(abstractOrderEntryModelList);

        assertNotNull(unit.buildCancelRequest(orderModel));
    }

    @Test
    public void createOrderCancelEntryListNotEmpty() {
        List<OrderCancelEntry> orderCancelEntryList = new ArrayList<>();

        unit.createOrderCancelEntry(orderCancelEntryList, abstractOrderEntryModel);

        assertFalse(orderCancelEntryList.isEmpty());
    }

    @Test
    public void shouldReturnCaptureTransactionEntryFromOrder() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(transactionEntryModel, newTransactionEntryModel));
        when(transactionEntryModel.getTransactionStatusDetails()).thenReturn(SUCCESFULL_STATUS);
        when(newTransactionEntryModel.getTransactionStatusDetails()).thenReturn(SUCCESFULL_STATUS);
        when(newTransactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        List<PaymentTransactionEntryModel> result = unit.getCaptureTransactionEntries(orderModel);
        assertEquals(1, result.size());
        assertEquals(PaymentTransactionType.CAPTURE, result.get(0).getType());
    }

    @Test
    public void shouldReturnOnlyCapturedTransactions() {
        doReturn(List.of(transactionEntryModel, newTransactionEntryModel)).when(unit).getCaptureTransactionEntries(orderModel);

        when(transactionEntryModel.getAmount()).thenReturn(AMOUNT_TEN);
        doReturn(AMOUNT_ZERO).when(unit).calculateRefundedAmount(transactionEntryModel);

        when(newTransactionEntryModel.getAmount()).thenReturn(AMOUNT_TEN);
        doReturn(AMOUNT_TEN).when(unit).calculateRefundedAmount(newTransactionEntryModel);

        List<PaymentTransactionEntryModel> result = unit.getTransactionsToRefund(orderModel);
        assertFalse(result.isEmpty());
        assertEquals(1, result.size());
        assertEquals(transactionEntryModel, result.get(0));
    }

}
