package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.commands.impl.DefaultPayPalSendTrackingInfoCommand;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalTrackingInfoData;
import com.paypal.hybris.data.PayPalTrackingInfoRequestData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordersplitting.model.ConsignmentEntryModel;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.List;
import java.util.Set;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalTrackingInfoServiceTest {

    private static final String AMOUNT_VALUE = "123.45";
    private static final String USD = "USD";
    private static final String SKU = "12345678";
    private static final String PRODUCT_NAME = "productName";
    private static final String TRACKING_NUMBER = "1234asdf";
    private static final String CARRIER = "carrier";
    private static final String CAPTURE_ID = "captureId";
    private static final String ORDER_ID = "orderId";
    private static final String PAYPAL_PROVIDER = "PayPal";
    private static final String COMPLETED_STATUS = "COMPLETED";
    private AutoCloseable closable;
    @Mock
    private Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter;
    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;
    @Mock
    private CommandFactory commandFactory;
    @Mock
    private DefaultPayPalSendTrackingInfoCommand sendTrackingInfoCommand;
    @Mock
    private ConsignmentModel consignmentModel;
    @Mock
    private ConsignmentEntryModel consignmentEntryModel;
    @Mock
    private AbstractOrderEntryModel orderEntryModel;
    @Mock
    private PaymentTransactionModel transactionModel;
    @Mock
    private PaymentTransactionEntryModel captureTransactionEntryModel;
    @Mock
    private OrderModel orderModel;
    @Mock
    private PaymentTransactionsService paymentTransactionsService;

    @InjectMocks
    @Spy
    private DefaultPayPalTrackingInfoService unit;

    @Before
    public void setup() {
        closable = MockitoAnnotations.openMocks(this);
        when(captureTransactionEntryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(transactionModel.getOrder()).thenReturn(orderModel);
        when(orderModel.getPayPalOrderId()).thenReturn(ORDER_ID);
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transactionModel));
    }

    @Test
    public void shouldReturnOrderItemsFromConsignment() {
        OrderEntryData orderEntryData = new OrderEntryData();
        OrderItemData orderItemData = new OrderItemData();

        when(consignmentModel.getConsignmentEntries()).thenReturn(Set.of(consignmentEntryModel));
        when(consignmentEntryModel.getOrderEntry()).thenReturn(orderEntryModel);
        when(orderItemDataConverter.convert(orderEntryModel)).thenReturn(orderItemData);

        List<OrderItemData> orderItemsFromConsignment = unit.getOrderItemsFromConsignment(consignmentModel);
        assertEquals(orderItemData, orderItemsFromConsignment.get(0));
    }

    @Test
    public void shouldPerformSendTrackingInfoCommand() throws CommandNotSupportedException {
        OrderItemData orderItemData = new OrderItemData();
        SimpleAmount amount = new SimpleAmount();

        orderItemData.setSku(SKU);
        orderItemData.setName(PRODUCT_NAME);
        orderItemData.setAmount(amount);
        amount.setValue(AMOUNT_VALUE);
        amount.setCurrencyCode(USD);

        doReturn(List.of(orderItemData)).when(unit).getOrderItemsFromConsignment(consignmentModel);
        doReturn(List.of(captureTransactionEntryModel)).when(paymentTransactionsService).getCaptureTransactionEntries(orderModel);
        when(captureTransactionEntryModel.getRequestId()).thenReturn(CAPTURE_ID);
        when(transactionModel.getRequestId()).thenReturn(ORDER_ID);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalSendTrackingInfoCommand.class)).thenReturn(sendTrackingInfoCommand);
        when(sendTrackingInfoCommand.perform(any(PayPalTrackingInfoData.class))).thenReturn(COMPLETED_STATUS);

        String result = unit.sendTrackingInfoToPayPalForConsignment(captureTransactionEntryModel, consignmentModel, CARRIER, TRACKING_NUMBER);
        assertEquals(COMPLETED_STATUS, result);
        verify(sendTrackingInfoCommand, times(1)).perform(ArgumentMatchers.argThat(payPalTrackingInfoData -> {
            PayPalTrackingInfoRequestData trackingInfo = payPalTrackingInfoData.getTrackingInfo();
            OrderItemData itemData = trackingInfo.getItems().get(0);
            return ORDER_ID.equals(payPalTrackingInfoData.getOrderId())
                    && SKU.equals(itemData.getSku())
                    && PRODUCT_NAME.equals(itemData.getName())
                    && AMOUNT_VALUE.equals(itemData.getAmount().getValue())
                    && USD.equals(itemData.getAmount().getCurrencyCode())
                    && CAPTURE_ID.equals(trackingInfo.getCaptureId())
                    && TRACKING_NUMBER.equals(trackingInfo.getTrackingNumber())
                    && CARRIER.equals(trackingInfo.getCarrier());
        }));
    }

    @Test
    public void shouldReturnNullWhenSendTrackingInfoCommandFailed() throws CommandNotSupportedException {
        doReturn(List.of(new OrderItemData())).when(unit).getOrderItemsFromConsignment(consignmentModel);
        doReturn(List.of(captureTransactionEntryModel)).when(paymentTransactionsService).getCaptureTransactionEntries(orderModel);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalSendTrackingInfoCommand.class)).thenReturn(sendTrackingInfoCommand);
        when(sendTrackingInfoCommand.perform(any(PayPalTrackingInfoData.class))).thenThrow(AdapterException.class);

        String result = unit.sendTrackingInfoToPayPalForConsignment(captureTransactionEntryModel, consignmentModel, CARRIER, TRACKING_NUMBER);
        assertNull(result);
        verify(sendTrackingInfoCommand, times(1)).perform(ArgumentMatchers.any(PayPalTrackingInfoData.class));
    }

    @After
    public void close() throws Exception {
        closable.close();
    }

}
