/*

 */
package com.paypal.hybris.core.service.impl;

import com.paypal.http.HttpRequest;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCaptureIntentCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalConvertBAToPaymentTokensCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetCardDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetEventCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetOrderDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalProcessVaultedPaymentCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalReauthorizationRequestCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalSaveOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalUpdateOrderAmountCommand;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.exception.PayPalHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.exception.PayPalProcessVaultedPaymentException;
import com.paypal.hybris.core.exception.PayPalRefundAdapterException;
import com.paypal.hybris.core.exception.PayPalVoidAdapterException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.results.PayPalVaultedPaymentResult;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.strategy.storedcredential.StoredCredentialStrategy;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalConvertBAToPaymentTokensRequestData;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderProcessRequestData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.payments.Refund;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.RefundResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.methods.CardPaymentService;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.web.client.HttpClientErrorException;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Currency;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.FIRST_PAYMENT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalPaymentServiceTest {

    private static final String CURRENCY = "USD";
    private static final String PAYMENT_PROVIDER = "PaymentProvider";
    private static final String REQUEST_ID = "RequestId";
    private static final String REQUEST_TOKEN = "RequestToken";
    private static final String SUBSCRIPTION_ID = "subscriptionID";
    private static final String ORDER_ID = "orderId";
    private static final String AMOUNT = "10.00";
    private static final String SAVE_ORDER_ID = "saveOrderId";
    private static final String PAYMENT_TOKEN = "paymentToken";
    private static final String PAYPAL_INTENT_CAPTURE = "capture";
    private static final String CUSTOMER_ID = "customerId";
    private static final String NAME = "name";
    private static final String EXPIRY = "2025-12";
    private static final String ENTRY_CODE = "entryCode";
    private static final String CANCEL_ENTRY_CODE = "null-CANCEL-2";
    private static final String MERCHANT_TRANSACTION = "merchantTransaction";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private static final String NEW_ENTRY_CODE = UUID.randomUUID().toString();
    private static final String EXCEPTION_MESSAGE = "ERROR";
    private static final String NEW_ENTRY_CODE_FOR_CAPTURE = "null-CAPTURE-2";
    private static final String EXPECTED_FAILURE_REASON = "Status code: '400', Status text: '400 BAD_REQUEST', Message from PayPal: 'Something went wrong, please try again.'";
    private static final String EXPECTED_FAILURE_REASON_CAPTURE = "Status code: '404', Status text: 'NOT_FOUND', Message from PayPal: 'ERROR'";
    private static final String PAYER_EMAIL = "payerEmail";
    private static final String VAULT_RESULT = "{\"intent\":null,\"purchase_units\":null,\"payment_source\":null}";
    private static final String CAPTURE_FAILED_RESULT = "{\"merchantTransactionCode\":null,\"requestId\":null,\"requestToken\":null,\"currency\":null,\"totalAmount\":null,\"paymentProvider\":null,\"subscriptionID\":null}";
    private static final String REQUEST_FIELD = "Request field";
    private static final String RESPONSE_FIELD = "Response field";
    private static final String REQUEST_FIELD_EDITED = "Requestfield";
    private static final String RESPONSE_FIELD_EDITED = "Responsefield";
    private static final String CANCEL_REQUEST = "{\"path\":\"path\",\"verb\":\"POST\",\"body\":null,\"responseClass\":\"java.lang.Void\",\"headers\":{\"mHeaders\":{},\"keyMapping\":{}}}";
    private static final String REFUND_REQUEST = "{\"path\":\"path\",\"verb\":\"POST\",\"body\":null,\"responseClass\":\"com.paypal.payments.Refund\",\"headers\":{\"mHeaders\":{},\"keyMapping\":{}}}";

    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;

    @Mock
    private CommandFactory commandFactory;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private DefaultPayPalSaveOrderCommand payPalSaveOrderCommand;

    @Mock
    private DefaultPayPalCaptureIntentCommand payPalCaptureIntentCommand;

    @Mock
    private DefaultPayPalReauthorizationRequestCommand payPalReauthorizationRequestCommand;

    @Mock
    private DefaultPayPalUpdateOrderAmountCommand payPalUpdateOrderAmountCommand;

    @Mock
    private DefaultPayPalCreateOrderCommand payPalCreateOrderCommand;

    @Mock
    private DefaultPayPalProcessVaultedPaymentCommand payPalProcessHostedFieldsCommand;

    @Mock
    private DefaultPayPalGetOrderDetailsCommand payPalGetOrderDetailsCommand;

    @Mock
    private DefaultPayPalGetCardDetailsCommand payPalGetCardDetailsCommand;

    @Mock
    private DefaultPayPalConvertBAToPaymentTokensCommand payPalConvertBAToPaymentTokensCommand;

    @Mock
    private DefaultPayPalGetEventCommand payPalGetEventCommand;

    @Mock
    private UserService userService;

    @Mock
    private CustomerModel customerModel;

    @Mock
    private ModelService modelService;

    @Mock
    private CardPaymentService cardPaymentService;

    @Mock
    private DefaultPayPalPaymentInfoService paymentInfoService;

    @Mock
    private CommonI18NService commonI18NService;

    @Mock
    private PaymentTransactionEntryModel selectedEntry;
    @Mock
    private PaymentTransactionEntryModel vaultedEntry;

    @Mock
    private PaymentTransactionEntryModel cancelledEntry;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private AddressModel addressModel;

    @Mock
    private SubscriptionAuthorizationRequest subscriptionAuthorizationExpectedRequest;

    @Mock
    private CaptureRequest captureExpectedRequest;

    @Mock
    private CartModel cart;

    @Mock
    private OrderModel orderModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel;

    @Mock
    private List<StoredCredentialStrategy> storedCredentialStrategies;

    @Mock
    private StoredCredentialStrategy storedCredentialStrategy;

    @Mock
    private PayPalCustomerAccountService payPalCustomerAccountService;
    @Mock
    private PayPalRefundAdapterException refundException;
    @Mock
    private PayPalVoidAdapterException voidException;
    @Mock
    private DefaultPaymentTransactionsService paymentTransactionsService;
    @Mock
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    @Spy
    @InjectMocks
    private DefaultPayPalPaymentService unit;

    private CardData cardData;

    private RefundResult refundResult;

    private PayPalCaptureResult captureResult;

    private PayPalAuthorizationResult authorizationResult;

    private PayPalHostedFieldsOrderRequestData hostedFieldsProcessRequestData;

    private PayPalVaultedPaymentResult hostedFieldsProcessResultData;

    private PayPalOrderRequestData payPalOrderRequestData;

    private PayPalAddressDetailsData detailsData;

    private PayPalGetCardDetailsResponseData payPalGetCardDetailsResponseData;

    private PayPalConvertBAToPaymentTokensRequestData payPalConvertBAToPaymentTokensRequestData;
    private Date createTime;
    private Date updateTime;

    @Before
    public void setUp() throws CommandNotSupportedException {
        MockitoAnnotations.initMocks(this);
        createTime = new Date();
        updateTime = new Date();

        cardData = new CardData();
        refundResult = new RefundResult();
        captureResult = new PayPalCaptureResult();
        authorizationResult = new PayPalAuthorizationResult();
        authorizationResult.setRequestField(REQUEST_FIELD);
        authorizationResult.setResponseField(RESPONSE_FIELD);
        detailsData = new PayPalAddressDetailsData();
        hostedFieldsProcessRequestData = new PayPalHostedFieldsOrderRequestData();
        hostedFieldsProcessResultData = new PayPalVaultedPaymentResult();
        hostedFieldsProcessResultData.setRequest(REQUEST_FIELD);
        hostedFieldsProcessResultData.setResponse(RESPONSE_FIELD);
        hostedFieldsProcessResultData.setRequestId(REQUEST_ID);
        hostedFieldsProcessResultData.setTransactionStatus(TransactionStatus.ACCEPTED);
        hostedFieldsProcessResultData.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
        hostedFieldsProcessResultData.setDebugId(PAYPAL_DEBUG_ID_VALUE);
        hostedFieldsProcessResultData.setCreateTime(createTime);
        hostedFieldsProcessResultData.setUpdateTime(updateTime);
        payPalOrderRequestData = new PayPalOrderRequestData();
        payPalOrderRequestData.setOrderId(ORDER_ID);
        payPalOrderRequestData.setCurrency(CURRENCY);
        payPalOrderRequestData.setAmount(AMOUNT);
        payPalGetCardDetailsResponseData = new PayPalGetCardDetailsResponseData();
        payPalConvertBAToPaymentTokensRequestData = new PayPalConvertBAToPaymentTokensRequestData();
        final List<PaymentTransactionEntryModel> entryModels = new ArrayList<>();
        entryModels.add(selectedEntry);
        captureResult.setResponseField(RESPONSE_FIELD);
        captureResult.setRequestField(REQUEST_FIELD);
        when(refundException.getMessage()).thenReturn(EXCEPTION_MESSAGE);

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(selectedEntry);
        when(selectedEntry.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(selectedEntry.getPaymentTransaction()).thenReturn(transactionModel);
        when(selectedEntry.getCurrency()).thenReturn(currencyModel);
        when(commonI18NService.getCurrency(CURRENCY)).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY);
        when(selectedEntry.getRequestId()).thenReturn(REQUEST_ID);
        when(selectedEntry.getRequestToken()).thenReturn(REQUEST_TOKEN);
        when(transactionModel.getEntries()).thenReturn(List.of(selectedEntry));
        when(transactionModel.getEntries()).thenReturn(entryModels);
        when(transactionModel.getInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(transactionModel.getOrder()).thenReturn(orderModel);
        when(cardPaymentService.capture(any())).thenReturn(captureResult);
        when(transactionModel.getRequestId()).thenReturn(REQUEST_ID);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);

        when(payPalGetCardDetailsCommand.perform(PAYMENT_TOKEN)).thenReturn(Optional.of(cardData));
        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.TRUE);
        when(selectedEntry.getAmount()).thenReturn(BigDecimal.TEN);
        when(paymentTransactionsService.calculateRefundedAmount(selectedEntry)).thenReturn(BigDecimal.ONE);
        when(transactionModel.getPaymentProvider()).thenReturn(PAYMENT_PROVIDER);
        when(selectedEntry.getSubscriptionID()).thenReturn(SUBSCRIPTION_ID);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        captureResult.setTransactionStatus(TransactionStatus.ACCEPTED);
        captureResult.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);
        captureResult.setCustomerId(CUSTOMER_ID);
        captureResult.setPaymentToken(PAYMENT_TOKEN);
        cardData.setExpiry(EXPIRY);
        cardData.setName(NAME);
    }

    @Test
    public void shouldConvertBillingAgreementTokenToVault() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalConvertBAToPaymentTokensCommand.class)).thenReturn(payPalConvertBAToPaymentTokensCommand);
        when(payPalConvertBAToPaymentTokensCommand.perform(payPalConvertBAToPaymentTokensRequestData)).thenReturn(Optional.of(payPalGetCardDetailsResponseData));

        unit.convertBillingAgreementTokenToVault(payPalConvertBAToPaymentTokensRequestData);

        verify(payPalConvertBAToPaymentTokensCommand).perform(payPalConvertBAToPaymentTokensRequestData);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenDefaultPayPalConvertBAToPaymentTokensCommandNotSupported() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalConvertBAToPaymentTokensCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.convertBillingAgreementTokenToVault(payPalConvertBAToPaymentTokensRequestData);

        verify(payPalConvertBAToPaymentTokensCommand, never()).perform(payPalConvertBAToPaymentTokensRequestData);
    }

    @Test
    public void shouldRefund(){
        final PaymentTransactionType transactionType = PaymentTransactionType.REFUND_PARTIAL;
        final BigDecimal amount = BigDecimal.TEN;

        when(selectedEntry.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(unit.getNewPaymentTransactionEntryCode(transactionModel, transactionType)).thenReturn(null);
        when(cardPaymentService.refundFollowOn(any())).thenReturn(refundResult);
        refundResult.setTransactionStatus(TransactionStatus.ACCEPTED);
        refundResult.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);

        unit.refund(selectedEntry, amount);

        verify(modelService).attach(any());
        verify(modelService).saveAll(eq(orderModel), any());
        verify(modelService).refresh(transactionModel);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenRefundNotAvailable() {
        final BigDecimal amount = BigDecimal.TEN;
        when(selectedEntry.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        unit.refund(selectedEntry, amount);
    }

    @Test
    public void shouldRefundWithException(){
        final PaymentTransactionType transactionType = PaymentTransactionType.REFUND_PARTIAL;
        final BigDecimal amount = BigDecimal.TEN;
        when(selectedEntry.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(unit.getNewPaymentTransactionEntryCode(transactionModel, transactionType)).thenReturn(null);

        unit.refund(selectedEntry, amount);

        verify(modelService).attach(any());
        verify(modelService).saveAll(eq(orderModel), any());
        verify(modelService).refresh(transactionModel);
    }

    @Test
    public void shouldSavePayPalOrder() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalSaveOrderCommand.class)).thenReturn(payPalSaveOrderCommand);
        when(payPalCreditCardPaymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);
        when(payPalSaveOrderCommand.perform(ORDER_ID)).thenReturn(SAVE_ORDER_ID);
        doNothing().when(modelService).saveAll(payPalCreditCardPaymentInfoModel, cart);

        boolean result = unit.savePayPalOrder(payPalCreditCardPaymentInfoModel, cart);

        assertTrue(result);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPayPalSaveOrderCommand.class);
        verify(payPalCreditCardPaymentInfoModel).getPayPalOrderId();
        verify(payPalSaveOrderCommand).perform(ORDER_ID);
        verify(modelService).saveAll(payPalCreditCardPaymentInfoModel, cart);
    }

    @Test
    public void shouldAuthorizePaymentAndSetCustomerVaultIdWhenItIsEmpty() throws CommandNotSupportedException {
        Currency currency = Currency.getInstance(CURRENCY);

        when(cart.getDeliveryAddress()).thenReturn(null);
        when(cart.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(null);
        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.AUTHORIZATION);

        when(cardPaymentService.authorize(any(SubscriptionAuthorizationRequest.class))).thenReturn(authorizationResult);

        populateAuthorizationResult(currency);

        populateCardData();

        when(payPalCreditCardPaymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);

        PaymentTransactionEntryModel result = unit.authorizePayment(MERCHANT_TRANSACTION, BigDecimal.TEN, currency, cart, PAYMENT_PROVIDER);

        verify(customerModel).setVaultCustomerId(CUSTOMER_ID);
        verify(payPalCreditCardPaymentInfoModel, never()).setPMCustomerVaultId(CUSTOMER_ID);
        verify(modelService).save(customerModel);
        verify(modelService).save(payPalCreditCardPaymentInfoModel);
        verify(modelService).save(selectedEntry);
        verify(modelService).save(transactionModel);

        assertEquals(selectedEntry, result);
    }

    private void populateCardData() {
        cardData.setExpiry(EXPIRY);
        cardData.setName(NAME);
    }

    private void populateAuthorizationResult(Currency currency) {
        authorizationResult.setCustomerId(CUSTOMER_ID);
        authorizationResult.setPaymentToken(PAYMENT_TOKEN);
        authorizationResult.setTotalAmount(BigDecimal.TEN);
        authorizationResult.setCurrency(currency);
        authorizationResult.setRequestId(REQUEST_ID);
        authorizationResult.setTransactionStatus(TransactionStatus.ACCEPTED);
        authorizationResult.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);
    }

    @Test
    public void shouldAuthorizePaymentAndNotSetCustomerVaultIdWhenItIsPresent() throws CommandNotSupportedException {
        Currency currency = Currency.getInstance(CURRENCY);

        when(cart.getDeliveryAddress()).thenReturn(null);
        when(cart.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL);

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        when(payPalGetCardDetailsCommand.perform(PAYMENT_TOKEN)).thenReturn(Optional.of(cardData));

        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(CUSTOMER_ID);
        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.AUTHORIZATION);

        when(cardPaymentService.authorize(any(SubscriptionAuthorizationRequest.class))).thenReturn(authorizationResult);
        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.TRUE);

        populateAuthorizationResult(currency);

        populateCardData();

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(selectedEntry);
        when(payPalCreditCardPaymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);

        PaymentTransactionEntryModel result = unit.authorizePayment(MERCHANT_TRANSACTION, BigDecimal.TEN, currency, cart, PAYMENT_PROVIDER);

        verify(customerModel, never()).setVaultCustomerId(CUSTOMER_ID);
        verify(payPalCreditCardPaymentInfoModel, never()).setPMCustomerVaultId(CUSTOMER_ID);
        verify(modelService).save(payPalCreditCardPaymentInfoModel);
        verify(modelService).save(selectedEntry);
        verify(modelService).save(transactionModel);

        verify(payPalCustomerAccountService).removeDuplicatePaymentMethod(PAYER_EMAIL);

        assertEquals(selectedEntry, result);
    }

    @Test
    public void shouldAuthorizePaymentAndSetPMCustomerVaultIdWhenPayPal() throws CommandNotSupportedException {
        Currency currency = Currency.getInstance(CURRENCY);

        when(cart.getDeliveryAddress()).thenReturn(null);
        when(cart.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL);

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        when(payPalGetCardDetailsCommand.perform(PAYMENT_TOKEN)).thenReturn(Optional.of(cardData));

        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(CUSTOMER_ID);
        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.AUTHORIZATION);

        when(cardPaymentService.authorize(any(SubscriptionAuthorizationRequest.class))).thenReturn(authorizationResult);
        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.TRUE);

        populateAuthorizationResult(currency);

        populateCardData();

        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(selectedEntry);
        when(payPalCreditCardPaymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID);
        when(payPalCreditCardPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        PaymentTransactionEntryModel result = unit.authorizePayment(MERCHANT_TRANSACTION, BigDecimal.TEN, currency, cart, PAYMENT_PROVIDER);

        verify(customerModel, never()).setVaultCustomerId(CUSTOMER_ID);
        verify(payPalCreditCardPaymentInfoModel).setPMCustomerVaultId(CUSTOMER_ID);
        verify(modelService).save(payPalCreditCardPaymentInfoModel);
        verify(modelService).save(selectedEntry);
        verify(modelService).save(transactionModel);

        verify(payPalCustomerAccountService).removeDuplicatePaymentMethod(PAYER_EMAIL);

        assertEquals(selectedEntry, result);
    }

    @Test
    public void shouldReturnTrueWhenOrderCanceled() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(selectedEntry.getType()).thenReturn(PaymentTransactionType.CANCEL);
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        assertTrue(unit.isOrderCanceled(orderModel));

        verify(modelService).refresh(orderModel);
    }

    @Test
    public void shouldReturnFalseWhenOrderNotCanceled() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        assertFalse(unit.isOrderCanceled(orderModel));

        verify(modelService).refresh(orderModel);
    }

    @Test
    public void shouldCapture(){
        captureResult.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);

        PaymentTransactionEntryModel entry = unit.capture(transactionModel);

        verify(modelService).attach(entry);
        verify(modelService).save(transactionModel);
        verify(modelService).save(entry);
    }

    @Test
    public void shouldCaptureHostedFieldsPayment() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalProcessVaultedPaymentCommand.class)).thenReturn(payPalProcessHostedFieldsCommand);
        when(payPalProcessHostedFieldsCommand.perform(any())).thenReturn(hostedFieldsProcessResultData);

        when(payPalCreditCardPaymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID);
        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);

        hostedFieldsProcessResultData.setOrderId(ORDER_ID);
        hostedFieldsProcessResultData.setTransactionStatus(TransactionStatus.ACCEPTED);
        hostedFieldsProcessResultData.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);

        when(storedCredentialStrategies.stream()).thenReturn(Stream.of(storedCredentialStrategy));
        when(storedCredentialStrategy.test(FIRST_PAYMENT)).thenReturn(Boolean.TRUE);

        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CAPTURE);

        unit.captureVaultedPayment(transactionModel, FIRST_PAYMENT, PaymentProvider.PAYPAL_HOSTED_FIELDS);

        verify(orderRequestDataConverter).convert(eq(orderModel), any(PayPalHostedFieldsOrderRequestData.class));
        verify(orderModel).setPayPalOrderId(ORDER_ID);
        verify(modelService).saveAll(any(), eq(selectedEntry), eq(orderModel));
    }

    @Test
    public void shouldAuthorizeHostedFieldPayment() throws CommandNotSupportedException {
        Currency currency = Currency.getInstance(CURRENCY);
        when(commandFactory.createCommand(DefaultPayPalProcessVaultedPaymentCommand.class)).thenReturn(payPalProcessHostedFieldsCommand);
        when(payPalProcessHostedFieldsCommand.perform(any())).thenReturn(hostedFieldsProcessResultData);

        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(selectedEntry);

        when(cart.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);

        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.AUTHORIZATION);
        when(storedCredentialStrategies.stream()).thenReturn(Stream.of(storedCredentialStrategy));
        when(storedCredentialStrategy.test(FIRST_PAYMENT)).thenReturn(Boolean.TRUE);

        when(payPalCreditCardPaymentInfoModel.getSubscriptionId()).thenReturn(SUBSCRIPTION_ID);
        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);

        hostedFieldsProcessResultData.setOrderId(ORDER_ID);
        hostedFieldsProcessResultData.setTransactionStatus(TransactionStatus.ACCEPTED);
        hostedFieldsProcessResultData.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);
        hostedFieldsProcessResultData.setRequestId(REQUEST_ID);

        PaymentTransactionEntryModel result = unit.authorizeVaultedPayment(MERCHANT_TRANSACTION, BigDecimal.TEN, currency, cart, PAYMENT_PROVIDER, FIRST_PAYMENT, PaymentProvider.PAYPAL_HOSTED_FIELDS);

        verify(orderRequestDataConverter).convert(eq(cart), any());
        verify(cart).setPayPalOrderId(ORDER_ID);
        verify(modelService).save(transactionModel);
        verify(modelService).saveAll(selectedEntry, cart);
        verify(modelService).refresh(transactionModel);

        assertEquals(selectedEntry, result);
    }

    @Test
    public void shouldCaptureWhenPayPalIntent() throws CommandNotSupportedException {
        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(payPalCaptureIntentCommand.perform(any())).thenReturn(captureResult);
        when(commandFactory.createCommand(DefaultPayPalCaptureIntentCommand.class)).thenReturn(payPalCaptureIntentCommand);
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        captureResult.setTransactionStatusDetails(TransactionStatusDetails.TIMEOUT);
        captureResult.setCustomerId(CUSTOMER_ID);
        captureResult.setPaymentToken(PAYMENT_TOKEN);
        populateCardData();

        PaymentTransactionEntryModel entry = unit.capture(transactionModel);

        verify(modelService).save(transactionModel);
        verify(modelService).refresh(transactionModel);
        verify(modelService).save(entry);
        verify(modelService).attach(entry);
        verify(modelService).save(customerModel);
        verify(modelService).save(payPalCreditCardPaymentInfoModel);
    }

    @Test(expected = PayPalHttpClientErrorException.class)
    public void shouldThrowPayPalHttpClientErrorExceptionWhenCaptureFails() throws CommandNotSupportedException {
        HttpHeaders headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.BAD_REQUEST,
                HttpStatus.BAD_REQUEST.toString(), headers, new byte[]{}, StandardCharsets.UTF_8);

        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(commandFactory.createCommand(DefaultPayPalCaptureIntentCommand.class)).thenReturn(payPalCaptureIntentCommand);
        when(payPalCaptureIntentCommand.perform(any()))
                .thenThrow(new PayPalHttpClientErrorException(parentException, captureExpectedRequest));

        PayPalHttpClientErrorException e = new PayPalHttpClientErrorException(parentException, captureExpectedRequest);

        PaymentTransactionEntryModel entryModel = unit.capture(transactionModel);

        verify(unit, never()).createPaymentTransactionEntry(any(), any(), any());
        verify(unit).createFailedCapturePaymentTransactionEntry(e, transactionModel, NEW_ENTRY_CODE);
        verify(modelService).attach(entryModel);
        verify(modelService).save(entryModel);
    }

    @Test
    public void shouldCreateFailedTransactionEntryFromReceivedException() {
        HttpHeaders headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.BAD_REQUEST,
                HttpStatus.BAD_REQUEST.toString(), headers, new byte[]{}, StandardCharsets.UTF_8);

        PayPalHttpClientErrorException e = new PayPalHttpClientErrorException(parentException, captureExpectedRequest);

        PaymentTransactionEntryModel entryModel
                = unit.createFailedCapturePaymentTransactionEntry(e, transactionModel, NEW_ENTRY_CODE);

        assertEquals(PaymentTransactionType.CAPTURE, entryModel.getType());
        assertEquals(TransactionStatus.ERROR.name(), entryModel.getTransactionStatus());
        assertEquals(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name(), entryModel.getTransactionStatusDetails());
        assertEquals(transactionModel, entryModel.getPaymentTransaction());
        assertEquals(NEW_ENTRY_CODE, entryModel.getCode());
        assertEquals(CAPTURE_FAILED_RESULT, entryModel.getRequest());
        assertEquals(StringUtils.EMPTY, entryModel.getResponse());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, entryModel.getDebugId());
        assertEquals(EXPECTED_FAILURE_REASON, entryModel.getFailureReason());
    }

    @Test
    public void shouldCreateStubAuthorize() {
        final String merchantTransactionCode = UUID.randomUUID().toString();
        final String paymentProvider = UUID.randomUUID().toString();
        final String subscriptionId = UUID.randomUUID().toString();
        final String newEntryCode = UUID.randomUUID().toString();
        final BigDecimal amount = BigDecimal.valueOf(10);
        Currency currency = Currency.getInstance(CURRENCY);

        when(modelService.create(PaymentTransactionModel.class)).thenReturn(transactionModel);
        when(payPalCreditCardPaymentInfoModel.getSubscriptionId()).thenReturn(subscriptionId);
        doReturn(newEntryCode).when(unit).getNewPaymentTransactionEntryCode(any(), any());
        when(transactionModel.getPlannedAmount()).thenReturn(amount);
        when(transactionModel.getCurrency()).thenReturn(currencyModel);

        PaymentTransactionEntryModel actualTransactionModel = unit.createStubAuthorize(merchantTransactionCode, amount, currency, paymentProvider, payPalCreditCardPaymentInfoModel);

        assertEquals(transactionModel.getPlannedAmount(), actualTransactionModel.getAmount());
        assertEquals(transactionModel.getCurrency(), actualTransactionModel.getCurrency());
    }

    @Test
    public void shouldInvokeReauthorizeCommand() throws CommandNotSupportedException {
        when(subscriptionAuthorizationExpectedRequest.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactory.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactory.createCommand(DefaultPayPalReauthorizationRequestCommand.class)).thenReturn(payPalReauthorizationRequestCommand);
        when(payPalReauthorizationRequestCommand.perform(subscriptionAuthorizationExpectedRequest)).thenReturn(authorizationResult);

        unit.reauthorize(subscriptionAuthorizationExpectedRequest);

        verify(payPalReauthorizationRequestCommand).perform(subscriptionAuthorizationExpectedRequest);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenReauthorizeCommandNotSupported() throws CommandNotSupportedException {
        when(subscriptionAuthorizationExpectedRequest.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactory.getPaymentProvider()).thenReturn(PAYPAL_PROVIDER_NAME);
        when(commandFactory.createCommand(DefaultPayPalReauthorizationRequestCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.reauthorize(subscriptionAuthorizationExpectedRequest);
    }

    @Test
    public void shouldInvokeCreateOrderCommand() throws CommandNotSupportedException {
        final String currency = UUID.randomUUID().toString();
        final String orderAmount = UUID.randomUUID().toString();
        when(commandFactory.createCommand(DefaultPayPalCreateOrderCommand.class)).thenReturn(payPalCreateOrderCommand);
        unit.createOrder(GenericBuilder.of(PayPalOrderRequestData::new)
            .with(PayPalOrderRequestData::setCurrency, currency)
            .with(PayPalOrderRequestData::setSaveOrderFlowActive, false)
            .with(PayPalOrderRequestData::setAmount, orderAmount)
            .with(PayPalOrderRequestData::setShippingAddress, detailsData)
            .build());

        verify(payPalCreateOrderCommand).perform(any(PayPalOrderRequestData.class));
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenCreateOrderCommandNotSupported() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalCreateOrderCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.createOrder(new PayPalOrderRequestData());
    }


    @Test
    public void shouldInvokeCaptureIntentCommand() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalCaptureIntentCommand.class)).thenReturn(payPalCaptureIntentCommand);

        unit.doCaptureIntentRequest(captureExpectedRequest);

        verify(payPalCaptureIntentCommand).perform(captureExpectedRequest);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenCommandNotSupportedExceptionDuringInvokeCaptureIntentCommand() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalCaptureIntentCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.doCaptureIntentRequest(captureExpectedRequest);
    }

    @Test
    public void shouldDoHostedFieldsPaymentRequest() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalProcessVaultedPaymentCommand.class)).thenReturn(payPalProcessHostedFieldsCommand);
        when(payPalProcessHostedFieldsCommand.perform(hostedFieldsProcessRequestData)).thenReturn(hostedFieldsProcessResultData);

        PayPalVaultedPaymentResult result = unit.doVaultedPaymentRequest(hostedFieldsProcessRequestData);

        assertEquals(hostedFieldsProcessResultData, result);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPayPalProcessVaultedPaymentCommand.class);
        verify(payPalProcessHostedFieldsCommand).perform(hostedFieldsProcessRequestData);
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldSaveFailedTransactionEntryAndThrowPayPalProcessPaymentExceptionWhenPaymentRequestFails() throws CommandNotSupportedException {
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.NOT_FOUND, HttpStatus.NOT_FOUND.name());
        PayPalOrderProcessRequestData request = new PayPalOrderProcessRequestData();
        PayPalProcessVaultedPaymentException exception =
                new PayPalProcessVaultedPaymentException(parentException, request, EXCEPTION_MESSAGE);

        doThrow(exception).when(unit).doVaultedPaymentRequest(any());

        unit.captureVaultedPayment(transactionModel, FIRST_PAYMENT, PaymentProvider.PAYPAL_HOSTED_FIELDS);

        verify(modelService).save(transactionModel);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPayPalProcessVaultedPaymentCommand.class);
    }

    @Test
    public void shouldCreatePaymentTransactionEntryForCaptureVaultedPayment() {
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(vaultedEntry);
        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CAPTURE);

        unit.createTransactionEntryForCaptureVaultedPayment(transactionModel, selectedEntry, hostedFieldsProcessResultData);

        verify(vaultedEntry).setAmount(BigDecimal.TEN);
        verify(vaultedEntry).setCurrency(currencyModel);
        verify(vaultedEntry).setType(PaymentTransactionType.CAPTURE);
        verify(vaultedEntry).setPaymentTransaction(transactionModel);
        verify(vaultedEntry).setCode(ENTRY_CODE);
        verify(vaultedEntry).setTransactionStatus(TransactionStatus.ACCEPTED.name());
        verify(vaultedEntry).setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.name());
        verify(vaultedEntry).setDebugId(PAYPAL_DEBUG_ID_VALUE);
        verify(vaultedEntry).setRequestId(REQUEST_ID);
        verify(vaultedEntry).setCreateTime(createTime);
        verify(vaultedEntry).setUpdateTime(updateTime);
        verify(vaultedEntry).setRequest(REQUEST_FIELD_EDITED);
        verify(vaultedEntry).setResponse(RESPONSE_FIELD_EDITED);
    }

    @Test
    public void shouldPopulateFailedTransactionEntryWhenPayPalVaultedPaymentExceptionIsThrown() throws CommandNotSupportedException {
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.NOT_FOUND, HttpStatus.NOT_FOUND.name());
        PayPalOrderProcessRequestData request = new PayPalOrderProcessRequestData();
        PayPalProcessVaultedPaymentException exception =
                new PayPalProcessVaultedPaymentException(parentException, request, EXCEPTION_MESSAGE);

        doThrow(exception).when(unit).doVaultedPaymentRequest(any());

        PaymentTransactionEntryModel entry = unit.createFailedVaultedTransactionEntryModel(transactionModel,exception);

        assertEquals(PaymentTransactionType.CAPTURE, entry.getType());
        assertEquals(transactionModel, entry.getPaymentTransaction());
        assertEquals(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name(), entry.getTransactionStatusDetails());
        assertEquals(StringUtils.EMPTY, entry.getDebugId());
        assertEquals(NEW_ENTRY_CODE_FOR_CAPTURE, entry.getCode());
        assertEquals(VAULT_RESULT, entry.getRequest());
        assertEquals(EXPECTED_FAILURE_REASON_CAPTURE, entry.getFailureReason());
    }

    @Test
    public void shouldInvokeGetEventCommand() throws CommandNotSupportedException {
        final String eventId = UUID.randomUUID().toString();
        when(commandFactory.createCommand(DefaultPayPalGetEventCommand.class)).thenReturn(payPalGetEventCommand);

        unit.getEventById(eventId);

        verify(payPalGetEventCommand).perform(eventId);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenCommandNotSupported() throws CommandNotSupportedException {
        final String eventId = UUID.randomUUID().toString();
        when(commandFactory.createCommand(DefaultPayPalGetEventCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.getEventById(eventId);
    }

    @Test
    public void shouldSetCancelStatus() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transactionModel));
        when(selectedEntry.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        doReturn(Boolean.TRUE).when(unit).isOrderCanceled(orderModel);

        doReturn(selectedEntry).when(unit).cancel(selectedEntry);

        unit.doCancel(orderModel);

        verify(modelService).save(selectedEntry);
        verify(modelService).save(orderModel);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenPayPalVoidAdapterExceptionIsThrown() {
        doReturn(ENTRY_CODE).when(unit).getNewPaymentTransactionEntryCode(transactionModel, PaymentTransactionType.CANCEL);
        HttpRequest<Void> cancel = new HttpRequest<>("path", "POST", Void.class);
        PayPalVoidAdapterException exception =
                new PayPalVoidAdapterException(new IOException(EXPECTED_FAILURE_REASON), cancel);

        doThrow(exception).when(unit).getPaypalVoidResult(ENTRY_CODE, selectedEntry);

        unit.cancel(selectedEntry);
    }

    @Test
    public void shouldGetOrderDetails() throws CommandNotSupportedException {
        final String expectedRequest = UUID.randomUUID().toString();
        when(commandFactory.createCommand(DefaultPayPalGetOrderDetailsCommand.class)).thenReturn(payPalGetOrderDetailsCommand);

        unit.getOrderDetails(expectedRequest);

        verify(payPalGetOrderDetailsCommand).perform(expectedRequest);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenTheErrorsDuringPerformGetOrderDetailsCommand() throws CommandNotSupportedException {
        final String expectedRequest = UUID.randomUUID().toString();
        when(commandFactory.createCommand(DefaultPayPalGetOrderDetailsCommand.class)).thenReturn(payPalGetOrderDetailsCommand);

        when(payPalGetOrderDetailsCommand.perform(expectedRequest)).thenThrow(HttpClientErrorException.class);

        unit.getOrderDetails(expectedRequest);
    }

    @Test
    public void shouldGetCardData() throws CommandNotSupportedException {
        final Optional<CardData> optionalCardData = Optional.of(cardData);
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        when(payPalGetCardDetailsCommand.perform(PAYMENT_TOKEN)).thenReturn(optionalCardData);

        Optional<CardData> result = unit.getCardData(PAYMENT_TOKEN);

        assertEquals(optionalCardData, result);
        assertEquals(cardData, result.get());
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPayPalGetCardDetailsCommand.class);
        verify(payPalGetCardDetailsCommand).perform(PAYMENT_TOKEN);
    }


    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenErrorDuringGetCardData() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalGetCardDetailsCommand.class)).thenReturn(payPalGetCardDetailsCommand);
        when(payPalGetCardDetailsCommand.perform(PAYMENT_TOKEN)).thenThrow(HttpClientErrorException.class);

        unit.getCardData(PAYMENT_TOKEN);
    }

    @Test
    public void shouldReturnTrueWhenUpdateOrderAmountDetailsIsSuccessful() throws CommandNotSupportedException {
        when(commandFactory.createCommand(DefaultPayPalUpdateOrderAmountCommand.class))
                .thenReturn(payPalUpdateOrderAmountCommand);
        when(payPalUpdateOrderAmountCommand.perform(payPalOrderRequestData)).thenReturn(true);

        assertTrue(unit.updateOrderAmountDetails(payPalOrderRequestData));
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenUpdateOrderAmountDetailsFailed() throws CommandNotSupportedException {
        payPalOrderRequestData.setOrderId(UUID.randomUUID().toString());
        payPalOrderRequestData.setCurrency(UUID.randomUUID().toString());
        payPalOrderRequestData.setAmount(UUID.randomUUID().toString());
        when(commandFactory.createCommand(DefaultPayPalUpdateOrderAmountCommand.class))
                .thenReturn(payPalUpdateOrderAmountCommand);
        doThrow(new RuntimeException()).when(payPalUpdateOrderAmountCommand).perform(payPalOrderRequestData);

        unit.updateOrderAmountDetails(payPalOrderRequestData);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenUpdateOrderAmountDetailsIsNotCreated() throws CommandNotSupportedException {
        payPalOrderRequestData.setOrderId(UUID.randomUUID().toString());
        payPalOrderRequestData.setCurrency(UUID.randomUUID().toString());
        payPalOrderRequestData.setAmount(UUID.randomUUID().toString());
        doThrow(CommandNotSupportedException.class).when(commandFactory)
                .createCommand(DefaultPayPalUpdateOrderAmountCommand.class);

        unit.updateOrderAmountDetails(payPalOrderRequestData);
    }

    @Test
    public void shouldPopulateFailedTransactionEntryWithRequestAndFailureReasonWhenRefundAdapterExceptionIsThrown() {
        HttpRequest<Refund> refund = new HttpRequest<>("path", "POST", Refund.class);
        Date date = new Date();
        when(refundException.getRequest()).thenReturn(refund);
        when(refundException.getParentException()).thenReturn(new IOException(EXPECTED_FAILURE_REASON));
        when(refundException.getCreateDate()).thenReturn(date);
        when(refundException.getDebugId()).thenReturn(PAYPAL_DEBUG_ID_VALUE);

        PaymentTransactionEntryModel entry = unit.createFailedRefundTransaction(selectedEntry, BigDecimal.TEN,
                PaymentTransactionType.REFUND_PARTIAL, ENTRY_CODE, refundException);

        assertEquals(currencyModel, entry.getCurrency());
        assertEquals(PaymentTransactionType.REFUND_PARTIAL, entry.getType());
        assertEquals(transactionModel, entry.getPaymentTransaction());
        assertEquals(BigDecimal.TEN, entry.getAmount());
        assertEquals(REQUEST_ID, entry.getRequestId());
        assertEquals(REQUEST_TOKEN, entry.getRequestToken());
        assertEquals(TransactionStatus.REJECTED.name(), entry.getTransactionStatus());
        assertEquals(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name(), entry.getTransactionStatusDetails());
        assertEquals(ENTRY_CODE, entry.getCode());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, entry.getDebugId());
        assertEquals(date, entry.getCreateTime());

        assertEquals(REFUND_REQUEST, entry.getRequest());
        assertEquals(EXPECTED_FAILURE_REASON, entry.getFailureReason());
    }

    @Test
    public void shouldPopulateFailedCancelTransactionEntryWhenPayPalVoidAdapterExceptionOccurs() {
        HttpRequest<Void> cancel = new HttpRequest<>("path", "POST", Void.class);
        Date date = new Date();
        when(voidException.getRequest()).thenReturn(cancel);
        when(voidException.getParentException()).thenReturn(new IOException(EXPECTED_FAILURE_REASON));
        when(voidException.getCreateDate()).thenReturn(date);
        when(voidException.getDebugId()).thenReturn(PAYPAL_DEBUG_ID_VALUE);
        when(voidException.getMessage()).thenReturn(EXCEPTION_MESSAGE);

        PaymentTransactionEntryModel entry = unit.getPopulatedFailedCancelTransactionEntry(voidException, selectedEntry);

        assertEquals(PaymentTransactionType.CANCEL, entry.getType());
        assertEquals(transactionModel, entry.getPaymentTransaction());
        assertEquals(BigDecimal.TEN, entry.getAmount());
        assertEquals(TransactionStatus.ERROR.name(), entry.getTransactionStatus());
        assertEquals(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name(), entry.getTransactionStatusDetails());
        assertEquals(CANCEL_ENTRY_CODE, entry.getCode());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, entry.getDebugId());
        assertEquals(date, entry.getCreateTime());

        assertEquals(CANCEL_REQUEST, entry.getRequest());
        assertEquals(EXPECTED_FAILURE_REASON, entry.getFailureReason());
    }

}
