/**
 *
 */
package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.commands.impl.DefaultPayPalDeletePaymentTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPaypalCreatePaymentTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPaypalCreateSetupTokenCommand;
import com.paypal.hybris.core.dao.impl.DefaultPayPalPaymentInfoDao;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.SavedPaymentMethodsLimit;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.util.ExpiredDateValidatorUtil;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalPaymentInfoServiceTest {
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String ORDER_ID= "orderId";
    private static final String PAYPAL_PROVIDER_NAME = "PayPal";
    private static final String PK = "pk";
    private static final String VAULT_CUSTOMER_ID = "vault customer id";
    private static final String OTHER_CUSTOMER_VAULT_CUSTOMER_ID = "other customer vault customer id";

    @Mock
    private DefaultPayPalPaymentInfoDao payPalPaymentInfoDao;

    @Mock
    private ModelService modelService;

    @Mock
    private UserService userService;

    @Mock
    private PaymentInfoModel defaultPaymentInfoModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;
    @Mock
    private PayPalCreditCardPaymentInfoModel duplicatedPaymentInfoModel;

    @Mock
    private PayPalSavePaymentForPurchaseLaterRequest payPalSaveCardRequest;

    @Mock
    private CommandFactory commandFactory;

    @Mock
    private DefaultPaypalCreateSetupTokenCommand command;

    @Mock
    private DefaultPaypalCreatePaymentTokenCommand defaultPaypalCreatePaymentTokenCommand;

    @Mock
    private DefaultPayPalDeletePaymentTokenCommand defaultPayPalDeletePaymentTokenCommand;

    @Mock
    private CustomerModel customer;

    @Mock
    private PayPalSetupTokenResponse setupTokenResponse;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private ExpiredDateValidatorUtil expiredDateValidatorUtil;

    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;

    @InjectMocks
    private DefaultPayPalPaymentInfoService unit;

    private Map<SavedPaymentMethodsLimit, Integer> savedPaymentMethodsLimitIntegerMap;

    private List<PaymentInfoModel> payPalCreditCardPaymentInfoModels;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);

        savedPaymentMethodsLimitIntegerMap = new HashMap<>();
        payPalCreditCardPaymentInfoModels = new ArrayList<>();

        savedPaymentMethodsLimitIntegerMap.put(SavedPaymentMethodsLimit.ONE, 1);
        savedPaymentMethodsLimitIntegerMap.put(SavedPaymentMethodsLimit.TWO, 2);
        savedPaymentMethodsLimitIntegerMap.put(SavedPaymentMethodsLimit.THREE, 3);
        savedPaymentMethodsLimitIntegerMap.put(SavedPaymentMethodsLimit.FOUR, 4);
        savedPaymentMethodsLimitIntegerMap.put(SavedPaymentMethodsLimit.FIVE, 5);

        payPalCreditCardPaymentInfoModels.add(paymentInfoModel);
        payPalCreditCardPaymentInfoModels.add(paymentInfoModel);

        unit.setSavedPaymentMethodsLimitIntegerMap(savedPaymentMethodsLimitIntegerMap);
    }

    @Test
    public void shouldReturnTrueWhenPayPalAccountsLimited() {
        when(defaultPayPalConfigurationService.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.ONE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(payPalCreditCardPaymentInfoModels);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        assertTrue(unit.isSavedPayPalAccountsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenPayPalAccountsNotLimited() {
        when(defaultPayPalConfigurationService.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(payPalCreditCardPaymentInfoModels);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        assertFalse(unit.isSavedPayPalAccountsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenUserHasNotSavedPayments() {
        when(defaultPayPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.TRUE);

        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(Collections.singletonList(paymentInfoModel));
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.FALSE);

        assertFalse(unit.isChangePaymentButtonActive());
    }

    @Test
    public void shouldReturnFalseWhenUserHasNotSavedPayPalPayments() {
        when(defaultPayPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.TRUE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);
        when(customer.getPaymentInfos()).thenReturn(Collections.singletonList(paymentInfoModel));
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        assertFalse(unit.isChangePaymentButtonActive());
    }

    @Test
    public void shouldReturnFalseWhenChangePaymentButtonDisabled() {
        when(defaultPayPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.FALSE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);

        assertFalse(unit.isChangePaymentButtonActive());
    }

    @Test
    public void shouldReturnTrueWhenChangePaymentButtonActiveAndPMAttachedToTheSameCustomer() {
        when(defaultPayPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.TRUE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);
        when(customer.getPaymentInfos()).thenReturn(Collections.singletonList(paymentInfoModel));
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getDuplicate()).thenReturn(Boolean.FALSE);
        when(paymentInfoModel.getPMCustomerVaultId()).thenReturn(VAULT_CUSTOMER_ID);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        assertTrue(unit.isChangePaymentButtonActive());
    }

    @Test
    public void shouldReturnFalseWhenChangePaymentButtonActiveAndPMAttachedToDifferentCustomers() {
        when(defaultPayPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.TRUE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);
        when(customer.getPaymentInfos()).thenReturn(Collections.singletonList(paymentInfoModel));
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getDuplicate()).thenReturn(Boolean.FALSE);
        when(paymentInfoModel.getPMCustomerVaultId()).thenReturn(OTHER_CUSTOMER_VAULT_CUSTOMER_ID);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        assertFalse(unit.isChangePaymentButtonActive());
    }

    @Test
    public void shouldReturnFalseWhenPayPalAccountsUnlimited() {
        when(defaultPayPalConfigurationService.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);

        assertFalse(unit.isSavedPayPalAccountsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenThereNotPayPalAccounts() {
        when(defaultPayPalConfigurationService.getSavedPayPalAccountsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(Collections.emptyList());

        assertFalse(unit.isSavedPayPalAccountsAmountLimited());
    }

    @Test
    public void shouldReturnTrueWhenCreditCardLimited() {
        when(defaultPayPalConfigurationService.getSavedCreditCardsLimit()).thenReturn(SavedPaymentMethodsLimit.ONE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(payPalCreditCardPaymentInfoModels);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        assertTrue(unit.isSavedCreditCardsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenCreditCardsNotLimited() {
        when(defaultPayPalConfigurationService.getSavedCreditCardsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(payPalCreditCardPaymentInfoModels);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        assertFalse(unit.isSavedCreditCardsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenCreditCardsNotLimitedButAreDuplicated() {
        payPalCreditCardPaymentInfoModels.add(duplicatedPaymentInfoModel);
        when(defaultPayPalConfigurationService.getSavedCreditCardsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(payPalCreditCardPaymentInfoModels);
        when(paymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        when(duplicatedPaymentInfoModel.getDuplicate()).thenReturn(true);

        assertFalse(unit.isSavedCreditCardsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenCreditCardsUnlimited() {
        when(defaultPayPalConfigurationService.getSavedCreditCardsLimit()).thenReturn(SavedPaymentMethodsLimit.UNLIMITED);

        assertFalse(unit.isSavedCreditCardsAmountLimited());
    }

    @Test
    public void shouldReturnFalseWhenThereNotCreditCards() {
        when(defaultPayPalConfigurationService.getSavedCreditCardsLimit()).thenReturn(SavedPaymentMethodsLimit.THREE);
        when(userService.getCurrentUser()).thenReturn(customer);
        when(customer.getPaymentInfos()).thenReturn(Collections.emptyList());

        assertFalse(unit.isSavedCreditCardsAmountLimited());
    }

    @Test
    public void shouldGetPaypalSetupToken() throws CommandNotSupportedException {

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPaypalCreateSetupTokenCommand.class)).thenReturn(command);
        when(command.perform(payPalSaveCardRequest)).thenReturn(setupTokenResponse);

        PayPalSetupTokenResponse result = unit.getPaypalSetupToken(payPalSaveCardRequest);

        assertEquals(setupTokenResponse, result);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPaypalCreateSetupTokenCommand.class);
        verify(command).perform(payPalSaveCardRequest);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenCommandNotSupportedWhenGetPaypalSetupToken() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPaypalCreateSetupTokenCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.getPaypalSetupToken(payPalSaveCardRequest);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenCommandNotSupportedWhenGetPaypalPaymentToken() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPaypalCreatePaymentTokenCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.getPaypalPaymentToken(payPalSaveCardRequest);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenCommandNotSupportedWhenDeletePaypalPaymentToken() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalDeletePaymentTokenCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.deletePaypalPaymentToken(SUBSCRIPTION_ID);
    }


    @Test
    public void getCreditCardBySubscriptionIdTest() {
        Optional<PayPalCreditCardPaymentInfoModel> optionalPaymentInfoModel = Optional.of(paymentInfoModel);

        when(payPalPaymentInfoDao.findCardBySubscriptionId(SUBSCRIPTION_ID)).thenReturn(optionalPaymentInfoModel);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.getCreditCardBySubscriptionId(SUBSCRIPTION_ID);

        assertEquals(optionalPaymentInfoModel, result);
        assertEquals(paymentInfoModel, optionalPaymentInfoModel.get());
        verify(payPalPaymentInfoDao).findCardBySubscriptionId(SUBSCRIPTION_ID);
    }

    @Test
    public void getPaymentInfoByOrderIdTest() {
        Optional<PayPalCreditCardPaymentInfoModel> optionalPaymentInfoModel = Optional.of(paymentInfoModel);

        when(payPalPaymentInfoDao.findPaymentInfoByPayPalOrderId(ORDER_ID)).thenReturn(optionalPaymentInfoModel);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.getPaymentInfoByOrderId(ORDER_ID);

        assertEquals(optionalPaymentInfoModel, result);
        assertEquals(paymentInfoModel, optionalPaymentInfoModel.get());
        verify(payPalPaymentInfoDao).findPaymentInfoByPayPalOrderId(ORDER_ID);
    }

    @Test
    public void getPaypalPaymentTokenTest() throws CommandNotSupportedException {

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPaypalCreatePaymentTokenCommand.class)).thenReturn(defaultPaypalCreatePaymentTokenCommand);
        when(defaultPaypalCreatePaymentTokenCommand.perform(payPalSaveCardRequest)).thenReturn(setupTokenResponse);

        PayPalSetupTokenResponse result = unit.getPaypalPaymentToken(payPalSaveCardRequest);

        assertEquals(setupTokenResponse, result);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPaypalCreatePaymentTokenCommand.class);
        verify(defaultPaypalCreatePaymentTokenCommand).perform(payPalSaveCardRequest);
    }

    @Test
    public void deletePaypalPaymentTokenTest() throws CommandNotSupportedException {

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalDeletePaymentTokenCommand.class)).thenReturn(defaultPayPalDeletePaymentTokenCommand);
        when(defaultPayPalDeletePaymentTokenCommand.perform(SUBSCRIPTION_ID)).thenReturn(true);

        boolean result = unit.deletePaypalPaymentToken(SUBSCRIPTION_ID);

        assertTrue(result);
        verify(commandFactoryRegistry).getFactory(PAYPAL_PROVIDER_NAME);
        verify(commandFactory).createCommand(DefaultPayPalDeletePaymentTokenCommand.class);
        verify(defaultPayPalDeletePaymentTokenCommand).perform(SUBSCRIPTION_ID);
    }

    @Test
    public void updateExpirationStatusShouldSetExpiredTest() {
        final int expireBetween = 6;
        final long monthsToExpiration = -1;

        when(defaultPayPalConfigurationService.getPayPalExpireBetween()).thenReturn(expireBetween);
        when(expiredDateValidatorUtil.getMonthsToExpiration(paymentInfoModel.getValidToMonth(), paymentInfoModel.getValidToYear())).thenReturn(monthsToExpiration);

        unit.updateExpirationStatus(paymentInfoModel);

        verify(defaultPayPalConfigurationService).getPayPalExpireBetween();
        verify(expiredDateValidatorUtil).getMonthsToExpiration(any(), any());
        verify(paymentInfoModel).setExpirationStatus(ExpirationStatus.EXPIRED);
    }

    @Test
    public void updateExpirationStatusShouldSetExpireSoonTest() {
        final int expireBetween = 6;
        final long monthsToExpiration = 3;

        when(defaultPayPalConfigurationService.getPayPalExpireBetween()).thenReturn(expireBetween);
        when(expiredDateValidatorUtil.getMonthsToExpiration(paymentInfoModel.getValidToMonth(), paymentInfoModel.getValidToYear())).thenReturn(monthsToExpiration);

        when(paymentInfoModel.getUser()).thenReturn(customer);
        doNothing().when(modelService).save(customer);

        unit.updateExpirationStatus(paymentInfoModel);

        verify(defaultPayPalConfigurationService).getPayPalExpireBetween();
        verify(expiredDateValidatorUtil).getMonthsToExpiration(any(), any());
        verify(paymentInfoModel).setExpirationStatus(ExpirationStatus.EXPIRE_SOON);
        verify(paymentInfoModel).getUser();
        verify(customer).setHasNewExpireSoonCard(true);
        verify(modelService).save(customer);
    }

    @Test
    public void updateExpirationStatusShouldSetNotExpiredTest() {
        final int expireBetween = 6;
        final long monthsToExpiration = 9;

        when(defaultPayPalConfigurationService.getPayPalExpireBetween()).thenReturn(expireBetween);
        when(expiredDateValidatorUtil.getMonthsToExpiration(paymentInfoModel.getValidToMonth(), paymentInfoModel.getValidToYear())).thenReturn(monthsToExpiration);

        unit.updateExpirationStatus(paymentInfoModel);

        verify(defaultPayPalConfigurationService).getPayPalExpireBetween();
        verify(expiredDateValidatorUtil).getMonthsToExpiration(any(), any());
        verify(paymentInfoModel).setExpirationStatus(ExpirationStatus.NOT_EXPIRED);
    }

    @Test
    public void shouldRemovePaymentInfoByPK() {
        unit.removePaymentInfoByPK(PK);

        verify(payPalPaymentInfoDao).removePaymentInfoByPK(PK);
    }

    @Test
    public void shouldgetPaymentInfoByPK() {
        when(payPalPaymentInfoDao.findPaymentInfoByPK(PK)).thenReturn(Optional.of(paymentInfoModel));

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.getPaymentInfoByPK(PK);

        assertTrue(result.isPresent());
        assertEquals(paymentInfoModel, result.get());
        verify(payPalPaymentInfoDao).findPaymentInfoByPK(PK);
    }

    @Test
    public void shouldReturnTrueWhenSaveOrderFlowActive() {
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(true);
        assertTrue(unit.isPayPalSaveOrderFlowActive(paymentInfoModel));
    }

    @Test
    public void shouldReturnFalseWhenSaveOrderFlowInactive() {
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(false);
        assertFalse(unit.isPayPalSaveOrderFlowActive(paymentInfoModel));
    }

    @Test
    public void shouldReturnFalseForNonPayPalPaymentInfoModel() {
        assertFalse(unit.isPayPalSaveOrderFlowActive(defaultPaymentInfoModel));
    }
}
