/**
 *
 */
package com.paypal.hybris.core.service.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.http.HttpRequest;
import com.paypal.hybris.core.commands.impl.DefaultPayPalAuthorizeSavedOrderCommand;
import com.paypal.hybris.core.exception.PayPalAuthorizeAdapterException;
import com.paypal.hybris.core.exception.PayPalProcessHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.orders.Order;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

import de.hybris.platform.store.BaseStoreModel;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.web.client.HttpClientErrorException;

@UnitTest
public class DefaultPayPalManualAuthorizationServiceTest {

    public static final String PURCHASE_UNIT_ID = "id";
    private static final String CURRENCY_ISO_CODE_TEST = "USD";

    private static final String ORDER_ID_TEST = "testOrderId";
    private static final String TEST_USER_UID = "testUserUID";
    private static final String REQUEST_ID = "requestId";
    private static final String REQUEST_TOKEN = "requestToken";
    private static final String EXCEPTION_MESSAGE = "message";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private static final String BODY = "body";
    private static final String FAILURE_REASON = "Status code: '400', Status text: '400 BAD_REQUEST', Message from PayPal: '400 400 BAD_REQUEST'";

    @Mock
    private ModelService modelService;

    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;

    @Mock
    private PayPalPaymentService payPalPaymentService;

    @Mock
    private PayPalConfigurationService payPalConfigurationService;

    @Mock
    private I18NService i18nService;

    @Mock
    private BaseSiteModel baseSiteModel;

    @Mock
    private BaseStoreModel baseStoreModel;

    @Mock
    private CommandFactory commandFactory;

    @Mock
    private SubscriptionAuthorizationCommand command;

    @Mock
    private DefaultPayPalAuthorizeSavedOrderCommand savedOrderCommand;

    @Mock
    private AuthorizationResult result;

    @Mock
    private OrderModel orderModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private UserModel userModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;

    @Mock
    private PaymentTransactionModel paymentTransactionModel;
    @Mock
    private Converter<OrderModel, OrderData> orderConverter;
    @Mock
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    @Mock
    private AddressModel addressModel;
    @Mock
    private SubscriptionAuthorizationRequest request;

    @Spy
    @InjectMocks
    private DefaultPayPalManualAuthorizationService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        when(orderModel.getUser()).thenReturn(userModel);
        when(userModel.getUid()).thenReturn(TEST_USER_UID);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(orderRequestDataConverter.convert(orderModel)).thenReturn(new PayPalOrderRequestData());

        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID_TEST);
        when(paymentInfoModel.getPurchaseUnitId()).thenReturn(PURCHASE_UNIT_ID);

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);
        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        OrderData orderData = new OrderData();
        when(orderConverter.convert(orderModel)).thenReturn(orderData);

    }

    @Test
    public void shouldDoAuthorizationWithSuccessfulResult() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(SubscriptionAuthorizationCommand.class)).thenReturn(command);
        when(command.perform(any())).thenReturn(result);
        when(result.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED);
        when(result.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL);

        BigDecimal amount = BigDecimal.TEN;

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);

        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(payPalPaymentService.createOrder(GenericBuilder.of(PayPalOrderRequestData::new)
            .with(PayPalOrderRequestData::setCurrency, CURRENCY_ISO_CODE_TEST)
            .with(PayPalOrderRequestData::setAmount, amount.toString())
            .with(PayPalOrderRequestData::setSaveOrderFlowActive, false)
            .build()))
            .thenReturn(ORDER_ID_TEST);

        unit.doAuthorization(orderModel, amount);

        verify(paymentTransactionModel).setPlannedAmount(any());
        verify(modelService).save(paymentTransactionModel);
    }

    @Test
    public void shouldDoAuthorizationWhenSaveOrderFlowActive() throws CommandNotSupportedException {
        when(modelService.create(PaymentTransactionModel.class)).thenReturn(paymentTransactionModel);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalAuthorizeSavedOrderCommand.class)).thenReturn(savedOrderCommand);
        when(savedOrderCommand.perform(any())).thenReturn(result);

        when(result.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED);
        when(result.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL);
        when(result.getTotalAmount()).thenReturn(BigDecimal.ONE);
        when(result.getRequestId()).thenReturn(REQUEST_ID);
        when(result.getRequestToken()).thenReturn(REQUEST_TOKEN);
        when(result.getPaymentProvider()).thenReturn(PayPalPaymentProvider.PAYPAL.name());


        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID_TEST);
        when(paymentInfoModel.getPurchaseUnitId()).thenReturn(PURCHASE_UNIT_ID);

        BigDecimal amount = BigDecimal.TEN;

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);

        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(userModel.getUid()).thenReturn(TEST_USER_UID);


        unit.doAuthorization(orderModel, amount);

        verify(paymentTransactionModel).setPlannedAmount(any());
        verify(modelService).save(paymentTransactionModel);
    }

    @Test
    public void shouldDoAuthorizationWhenThrowsCommandNotSupportedException() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(SubscriptionAuthorizationCommand.class))
            .thenThrow(CommandNotSupportedException.class);

        BigDecimal amount = BigDecimal.TEN;

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);

        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        unit.doAuthorization(orderModel, amount);
        verify(modelService).refresh(paymentTransactionModel);
        verify(orderModel).getPaymentTransactions();
    }

    @Test
    public void shouldDoAuthorizationWithSaveOrderFlowWhenThrowsCommandNotSupportedException() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalAuthorizeSavedOrderCommand.class))
            .thenThrow(CommandNotSupportedException.class);

        BigDecimal amount = BigDecimal.TEN;

        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(paymentInfoModel.getPayPalOrderId()).thenReturn(ORDER_ID_TEST);
        when(paymentInfoModel.getPurchaseUnitId()).thenReturn(PURCHASE_UNIT_ID);

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);

        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        unit.doAuthorization(orderModel, amount);
        verify(modelService).refresh(paymentTransactionModel);
        verify(orderModel).getPaymentTransactions();
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenAuthorizeSavedOrderCommandFinishedWithCommandNotSupportedException()
            throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalAuthorizeSavedOrderCommand.class))
                .thenThrow(CommandNotSupportedException.class);

        unit.executeAuthorizeSavedOrderCommand(request, PURCHASE_UNIT_ID);
    }

    @Test
    public void shouldSaveFailedTransactionEntryWithRequestAndFailureReasonWhenAdapterExceptionOccurs() {
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        HttpRequest<Order> request = new HttpRequest<>("path", "post", Order.class);
        doThrow(new PayPalAuthorizeAdapterException(new IOException(EXCEPTION_MESSAGE), request)).when(unit)
                .executeAuthorizeSavedOrderCommand(any(), any());

        PaymentTransactionEntryModel entry = unit.doAuthorization(orderModel, BigDecimal.TEN);
        verify(modelService).save(paymentTransactionModel);
        verify(modelService).save(entry);
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldNewPayPalProcessPaymentExceptionWhenProcessPaymentExceptionOccurs() {
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        HttpHeaders headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.BAD_REQUEST,
                HttpStatus.BAD_REQUEST.toString(), headers, new byte[]{}, StandardCharsets.UTF_8);

        PayPalProcessHttpClientErrorException exception = new PayPalProcessHttpClientErrorException(parentException, request, EXCEPTION_MESSAGE);

        doThrow(exception).when(unit).executeSubscriptionAuthorizeCommand(any());

        unit.doAuthorization(orderModel, BigDecimal.TEN);
    }

    @Test
    public void shouldCreateEntryModelWithResponseAndDebugIdWhenPayPalProcessHttpClientErrorExceptionThrown() {

        HttpHeaders headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.BAD_REQUEST,
                HttpStatus.BAD_REQUEST.toString(), headers, BODY.getBytes(), StandardCharsets.UTF_8);

        PayPalProcessHttpClientErrorException exception = new PayPalProcessHttpClientErrorException(parentException, request, EXCEPTION_MESSAGE);

        PaymentTransactionEntryModel entry = unit.handlePayPalSDKError(orderModel, exception, BigDecimal.TEN);

        assertEquals(paymentTransactionModel, entry.getPaymentTransaction());
        assertEquals(PaymentTransactionType.AUTHORIZATION, entry.getType());
        assertEquals(request(), entry.getRequest());
        assertEquals(BODY, entry.getResponse());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, entry.getDebugId());
        assertEquals(FAILURE_REASON, entry.getFailureReason());

        verify(modelService).save(paymentTransactionModel);
        verify(modelService).save(entry);
    }

    @Test
    public void shouldCreateEntryModelWithoutResponseBodyAndDebugIdWhenAdapterExceptionThrown() {
        HttpRequest<Order> httpRequest = new HttpRequest<>("path", "GET", Order.class);
        PayPalAuthorizeAdapterException exception = new PayPalAuthorizeAdapterException(new IOException(EXCEPTION_MESSAGE), httpRequest);

        PaymentTransactionEntryModel entry = unit.handlePayPalSDKError(orderModel, exception, BigDecimal.TEN);

        assertEquals(paymentTransactionModel, entry.getPaymentTransaction());
        assertEquals(PaymentTransactionType.AUTHORIZATION, entry.getType());
        System.out.println(entry.getRequest());
        assertEquals(httpRequest(), entry.getRequest());
        assertEquals(null, entry.getResponse());
        assertEquals(null, entry.getDebugId());
        assertEquals(EXCEPTION_MESSAGE, entry.getFailureReason());

        verify(modelService).save(paymentTransactionModel);
        verify(modelService).save(entry);

    }

    @Test
    public void shouldDoAuthorizationWithSuccessfulResultAndAddressNotNull() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(SubscriptionAuthorizationCommand.class)).thenReturn(command);
        when(command.perform(any())).thenReturn(result);
        when(result.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED);
        when(result.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL);

        BigDecimal amount = BigDecimal.TEN;

        List<PaymentTransactionModel> paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);

        when(orderModel.getSite()).thenReturn(baseSiteModel);
        when(baseSiteModel.getStores()).thenReturn(List.of(baseStoreModel));
        when(payPalConfigurationService.getPayPalIntent(baseStoreModel)).thenReturn(PAYPAL_INTENT_CAPTURE);

        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE_TEST);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(orderModel.getDeliveryAddress()).thenReturn(addressModel);
        when(payPalPaymentService.createOrder(GenericBuilder.of(PayPalOrderRequestData::new)
            .with(PayPalOrderRequestData::setCurrency, CURRENCY_ISO_CODE_TEST)
            .with(PayPalOrderRequestData::setAmount, amount.toString())
            .with(PayPalOrderRequestData::setSaveOrderFlowActive, false)
            .build()))
            .thenReturn(ORDER_ID_TEST);

        unit.doAuthorization(orderModel, amount);

        verify(paymentTransactionModel).setPlannedAmount(any());
        verify(modelService).save(paymentTransactionModel);
    }

    private String request() {
        return """
                {
                  "merchantTransactionCode" : null,
                  "subscriptionID" : null,
                  "currency" : null,
                  "totalAmount" : null,
                  "shippingInfo" : null,
                  "cv2" : null,
                  "paymentProvider" : null
                }""";
    }
    private String httpRequest() {
        return """
                {
                  "path" : "path",
                  "verb" : "GET",
                  "body" : null,
                  "responseClass" : "com.paypal.orders.Order",
                  "headers" : {
                    "mHeaders" : { },
                    "keyMapping" : { }
                  }
                }""";
    }


}
