/*

 */
package com.paypal.hybris.core.service.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.data.PayPalPaymentSource;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.SetupTokenRequestData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.impl.DefaultCustomerAccountService;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.util.Strings;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.Collections;
import java.util.List;
import java.util.UUID;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertSame;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalCustomerAccountServiceTest extends DefaultCustomerAccountService {

    private static final String PAYER_ID =  "payerID";
    private static final String ACCESS_TOKEN =  "accessToken";
    private static final String LAST_DIGITS = "1234";
    private static final String VISA = "Visa";
    private static final String ID = "id";
    private static final String CREDIT = "CREDIT";
    private static final String NAME_ON_CARD = "nameOnCard";
    private static final String EXPIRY_YEAR = "2020";
    private static final String EXPIRY_MONTH = "02";
    private static final String ADDRESS_CODE = "addressCode";
    private static final String TEST_EXPIRY = "12-2020";
    private static final String NAME = "name";
    private static final String PM_CUSTOMER_VAULT_ID = "PM customer vault id";
    private static final String PAYER_EMAIL = "payer email";
    private static final String SECOND_PAYER_EMAIL = "second payer email";

    @Mock
    private CustomerModel customerModel;

    @Mock
    private ModelService modelService;

    @Mock
    private UserService userService;

    @Mock
    private DefaultPayPalPaymentInfoService paymentInfoService;

    @Mock
    private AddressModel addressModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel mockInfo;

    @Mock
    private PayPalSetupTokenResponse payPalPaymentTokenResponse;

    @Mock
    private PayPalPaymentSource paymentSource;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private PayPalCustomerAccountDao payPalCustomerAccountDao;

    @Mock
    private Converter<CustomerModel, CustomerData> customerConverter;

    @Mock
    private PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel;

    @Spy
    @InjectMocks
    private DefaultPayPalCustomerAccountService unit;

    private CardData cardData;

    private PayPalGetCardDetailsResponseData payPalGetCardDetailsResponseData;

    private PayPalCustomerData payPalCustomerData;

    private PaymentSourceData paymentSourceData;

    private CCPaymentInfoData ccPaymentInfoData;

    private CCPaymentInfoData ccRefreshPaymentInfoData;

    private SetupTokenRequestData setupTokenRequestData;

    private CreditCardAddressData creditCardAddressData;

    private CustomerData customerData;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);
        when(customerModel.getUid()).thenReturn(UUID.randomUUID().toString());
        when(userService.getCurrentUser()).thenReturn(customerModel);

        cardData = new CardData();
        customerData = new CustomerData();
        payPalCustomerData = new PayPalCustomerData();
        paymentSourceData = new PaymentSourceData();
        setupTokenRequestData = new SetupTokenRequestData();
        creditCardAddressData = new CreditCardAddressData();
        payPalGetCardDetailsResponseData = new PayPalGetCardDetailsResponseData();

        ccPaymentInfoData = setUpPaymentData();
        ccRefreshPaymentInfoData = setUpPaymentData();
    }

    private CCPaymentInfoData setUpPaymentData() {
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setPayerEmail(UUID.randomUUID().toString());
        paymentInfoData.setAccountHolderName(UUID.randomUUID().toString());
        paymentInfoData.setPayerId(UUID.randomUUID().toString());
        paymentInfoData.setSubscriptionId(UUID.randomUUID().toString());
        paymentInfoData.setCardType(CreditCardType.PAYPAL.getCode());
        return paymentInfoData;
    }

    @Before
    public void setUpPayPalInfoModel() {
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(ccRefreshPaymentInfoData.getPayerEmail());
        when(payPalPaymentInfoModel.getPayPalOrderId()).thenReturn(ccRefreshPaymentInfoData.getPayPalOrderId());
        when(payPalPaymentInfoModel.getPayerId()).thenReturn(ccRefreshPaymentInfoData.getPayerId());
        when(payPalPaymentInfoModel.getSubscriptionId()).thenReturn(ccRefreshPaymentInfoData.getSubscriptionId());
        when(payPalPaymentInfoModel.getCcOwner()).thenReturn(ccRefreshPaymentInfoData.getAccountHolderName());
        when(payPalPaymentInfoModel.getType()).thenReturn(CreditCardType.PAYPAL);
    }

    @Test
    public void shouldUpdateStubCreditCardPaymentInfo() {
        when(payPalPaymentInfoModel.getSubscriptionId()).thenReturn(ID);
        when(payPalPaymentInfoModel.getNumber()).thenReturn(LAST_DIGITS);
        when(payPalPaymentInfoModel.getType()).thenReturn(CreditCardType.CREDIT);

        when(payPalPaymentTokenResponse.getId()).thenReturn(ID);
        when(payPalPaymentTokenResponse.getPaymentSource()).thenReturn(paymentSource);
        when(paymentSource.getCard()).thenReturn(cardData);
        cardData.setLastDigits(LAST_DIGITS);
        cardData.setBrand(CREDIT);

        PayPalCreditCardPaymentInfoModel result = unit.updateStubCreditCardPaymentInfo(payPalPaymentInfoModel, payPalPaymentTokenResponse);

        verify(payPalPaymentInfoModel).setSaved(true);
        verify(modelService).saveAll(payPalPaymentInfoModel);

        assertEquals(payPalPaymentInfoModel, result);
    }

    @Test
    public void shouldCreateStubCreditCardPaymentInfoWhenPaymentProviderHostedFields() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class)).thenReturn(payPalPaymentInfoModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(false);

        when(customerModel.getPaymentInfos()).thenReturn(List.of(payPalPaymentInfoModel));
        when(customerModel.getUid()).thenReturn(ID);
        setupTokenRequestData.setCreditCardData(creditCardAddressData);
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS);
        creditCardAddressData.setNameOnCard(NAME_ON_CARD);
        creditCardAddressData.setExpiryYear(EXPIRY_YEAR);
        creditCardAddressData.setExpiryMonth(EXPIRY_MONTH);
        creditCardAddressData.setSelectedAddressCode(ADDRESS_CODE);

        doReturn(addressModel).when(unit).getAddressForCode(customerModel, ADDRESS_CODE);
        when(addressModel.getVisibleInAddressBook()).thenReturn(Boolean.TRUE);
        when(modelService.clone(addressModel)).thenReturn(addressModel);
        when(addressModel.getOwner()).thenReturn(null);

        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.TRUE);

        unit.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel);

        verify(paymentInfoService).updateExpirationStatus(payPalPaymentInfoModel);
        verify(modelService).clone(addressModel);
        verify(modelService).saveAll(payPalPaymentInfoModel, customerModel);
    }

    @Test
    public void shouldCreateStubCreditCardPaymentInfoWhenPaymentProviderPayPal() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class)).thenReturn(payPalPaymentInfoModel);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(false);

        when(customerModel.getPaymentInfos()).thenReturn(List.of(payPalPaymentInfoModel));
        when(customerModel.getUid()).thenReturn(ID);
        setupTokenRequestData.setCreditCardData(creditCardAddressData);
        setupTokenRequestData.setPaymentType(PayPalPaymentProvider.PAYPAL);
        creditCardAddressData.setNameOnCard(NAME_ON_CARD);
        creditCardAddressData.setExpiryYear(EXPIRY_YEAR);
        creditCardAddressData.setExpiryMonth(EXPIRY_MONTH);
        creditCardAddressData.setSelectedAddressCode(ADDRESS_CODE);

        doReturn(addressModel).when(unit).getAddressForCode(customerModel, ADDRESS_CODE);
        when(addressModel.getVisibleInAddressBook()).thenReturn(Boolean.TRUE);
        when(modelService.clone(addressModel)).thenReturn(addressModel);
        when(addressModel.getOwner()).thenReturn(null);

        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.TRUE);

        unit.createStubCreditCardPaymentInfo(setupTokenRequestData, customerModel);

        verify(modelService).saveAll(payPalPaymentInfoModel, customerModel);
    }


    @Test
    public void shouldUpdateCreditCardInfo() {
        when(payPalPaymentInfoModel.getUser()).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(null);
        payPalGetCardDetailsResponseData.setCustomer(payPalCustomerData);
        payPalCustomerData.setId(ID);

        when(defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()).thenReturn(Boolean.FALSE);
        payPalGetCardDetailsResponseData.setPaymentSource(paymentSourceData);
        payPalGetCardDetailsResponseData.setId(ID);
        paymentSourceData.setCard(cardData);
        cardData.setExpiry(TEST_EXPIRY);
        cardData.setName(NAME);
        cardData.setLastDigits(LAST_DIGITS);

        unit.updateCreditCardInfo(payPalGetCardDetailsResponseData, payPalPaymentInfoModel);

        verify(modelService).saveAll(payPalPaymentInfoModel, customerModel);
    }

    @Test
    public void shouldCorrectPopulatePayPalPaymentSubscription() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class))
            .thenReturn(payPalPaymentInfoModel);
        when(modelService.clone(addressModel)).thenReturn(addressModel);
        ccPaymentInfoData.setDefaultPaymentInfo(Boolean.TRUE);
        ccPaymentInfoData.setPMCustomerVaultId(PM_CUSTOMER_VAULT_ID);

        final PayPalCreditCardPaymentInfoModel actualInfo = unit
            .createPayPalPaymentSubscription(customerModel, ccPaymentInfoData, addressModel, null);

        assertEquals(payPalPaymentInfoModel.getPayerEmail(), actualInfo.getPayerEmail());
        assertEquals(payPalPaymentInfoModel.getPayPalOrderId(), actualInfo.getPayPalOrderId());
        assertEquals(payPalPaymentInfoModel.getPayerId(), actualInfo.getPayerId());
        assertEquals(payPalPaymentInfoModel.getSubscriptionId(), actualInfo.getSubscriptionId());
        assertEquals(payPalPaymentInfoModel.getCcOwner(), actualInfo.getCcOwner());
        assertEquals(CreditCardType.PAYPAL, payPalPaymentInfoModel.getType());
        assertEquals(payPalPaymentInfoModel.getPMCustomerVaultId(), actualInfo.getPMCustomerVaultId());
    }

    @Test
    public void shouldCorrectPopulatePayPalPaymentSubscriptionWhenSaveOrderFlowEnabled() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class))
                .thenReturn(payPalPaymentInfoModel);
        when(modelService.clone(addressModel)).thenReturn(addressModel);
        when(defaultPayPalConfigurationService.isSaveOrderFlow()).thenReturn(Boolean.TRUE);
        ccPaymentInfoData.setDefaultPaymentInfo(Boolean.TRUE);

        final PayPalCreditCardPaymentInfoModel actualInfo = unit
                .createPayPalPaymentSubscription(customerModel, ccPaymentInfoData, addressModel, null);

        assertEquals(payPalPaymentInfoModel.getPayerEmail(), actualInfo.getPayerEmail());
        assertEquals(payPalPaymentInfoModel.getPayPalOrderId(), actualInfo.getPayPalOrderId());
        assertEquals(payPalPaymentInfoModel.getPayerId(), actualInfo.getPayerId());
        assertEquals(payPalPaymentInfoModel.getSubscriptionId(), actualInfo.getSubscriptionId());
        assertEquals(payPalPaymentInfoModel.getCcOwner(), actualInfo.getCcOwner());
        assertEquals(CreditCardType.PAYPAL, payPalPaymentInfoModel.getType());
        verify(actualInfo).setSaveOrderFlowActive(Boolean.TRUE);
    }

    @Test
    public void shouldCorrectSavePayPalPaymentSubscription() {
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class)).thenReturn(mockInfo);
        when(modelService.clone(addressModel)).thenReturn(addressModel);

        unit.createPayPalPaymentSubscription(customerModel, ccPaymentInfoData, addressModel, null);

        verify(modelService).saveAll(addressModel, mockInfo);
        verify(modelService, times(2)).refresh(customerModel);
    }

    @Test
    public void shouldSetSaveOrderFlowActiveOnCreatePayPalPaymentSubscriptionForVenmo(){
        when(modelService.create(PayPalCreditCardPaymentInfoModel.class))
                .thenReturn(payPalPaymentInfoModel);
        when(defaultPayPalConfigurationService.isSaveOrderFlow()).thenReturn(true);

        unit.createPayPalPaymentSubscription(customerModel,ccPaymentInfoData,addressModel,null);

        verify(payPalPaymentInfoModel).setSaveOrderFlowActive(true);
    }

    @Test
    public void shouldCorrectUpdatePayPAlPaymentSubscription() {

        final PayPalCreditCardPaymentInfoModel updatedPayPalPaymentInfoModel = unit
            .updatePayPalPaymentSubscription(customerModel, ccRefreshPaymentInfoData, addressModel, payPalPaymentInfoModel);

        verify(modelService).saveAll(addressModel, payPalPaymentInfoModel);
        verify(modelService).saveAll(customerModel, payPalPaymentInfoModel);
        verify(modelService, times(2)).refresh(customerModel);

        assertNotEquals(updatedPayPalPaymentInfoModel.getPayerEmail(), ccPaymentInfoData.getPayerEmail());
        assertNotEquals(updatedPayPalPaymentInfoModel.getCcOwner(), ccPaymentInfoData.getAccountHolderName());
        assertNotEquals(updatedPayPalPaymentInfoModel.getPayerId(), ccPaymentInfoData.getPayerId());
        assertNotEquals(updatedPayPalPaymentInfoModel.getSubscriptionId(), ccPaymentInfoData.getSubscriptionId());

        assertEquals(updatedPayPalPaymentInfoModel.getPayerEmail(), ccRefreshPaymentInfoData.getPayerEmail());
        assertEquals(updatedPayPalPaymentInfoModel.getCcOwner(), ccRefreshPaymentInfoData.getAccountHolderName());
        assertEquals(updatedPayPalPaymentInfoModel.getPayerId(), ccRefreshPaymentInfoData.getPayerId());
        assertEquals(updatedPayPalPaymentInfoModel.getSubscriptionId(), ccRefreshPaymentInfoData.getSubscriptionId());
        assertEquals(updatedPayPalPaymentInfoModel.getType().getCode(), ccRefreshPaymentInfoData.getCardType());
    }

    @Test
    public void setTempPasswordTest(){
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);

        unit.setTempPassword(PAYER_ID);

        verify(modelService).save(customerModel);
    }

    @Test
    public void clearTempPasswordTest(){
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);

        unit.clearTempPassword(PAYER_ID);

        verify(modelService).save(customerModel);
    }

    @Test
    public void setAccessTokenForCustomerTest(){
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);

        unit.setAccessTokenForCustomer(ACCESS_TOKEN, PAYER_ID);

        verify(customerModel).setPayPalAccessToken(ACCESS_TOKEN);
        verify(modelService).save(customerModel);
    }
    @Test
    public void setDefaultAccessTokenTest(){
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);

        unit.setDefaultAccessToken(PAYER_ID);

        verify(customerModel).setPayPalAccessToken(StringUtils.EMPTY);
        verify(modelService).save(customerModel);
    }

    @Test
    public void shouldReturnCustomerDataByUid() {
        when(userService.getUserForUID(ID, CustomerModel.class)).thenReturn(customerModel);
        when(customerConverter.convert(customerModel)).thenReturn(customerData);

        assertEquals(customerData, unit.getCustomerDataByUid(ID));
    }

    @Test
    public void getCustomerByIdTest(){
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(PAYER_ID)).thenReturn(customerModel);
        assertEquals(unit.getCustomerModelByPayerId(PAYER_ID), customerModel);
    }

    @Test
    public void shouldGetCustomerByPayerId() {
        final String payerId = UUID.randomUUID().toString();
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId)).thenReturn(customerModel);
        when(customerConverter.convert(customerModel)).thenReturn(customerData);

        assertEquals(customerData, unit.getCustomerDataByPayerId(payerId));
    }

    @Test
    public void shouldNotGetCustomerByPayerId() {
        final String payerId = UUID.randomUUID().toString();
        when(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId)).thenReturn(null);

        assertFalse(unit.isCustomerWithPayerIdExist(payerId));
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldFailIfCustomerIsNull(){
        unit.setDefaultAddressEntry(null, addressModel);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldFailIfAddressIsNull(){
        unit.setDefaultAddressEntry(customerModel, null);
    }

    @Test
    public void shouldSetAsDefaultIfAddressIsVisible(){
        when(addressModel.getVisibleInAddressBook()).thenReturn(Boolean.TRUE);
        when(customerModel.getAddresses()).thenReturn(List.of(addressModel));
        unit.setDefaultAddressEntry(customerModel, addressModel);
        verify(customerModel, times(1)).setDefaultPaymentAddress(addressModel);
        verify(customerModel, times(1)).setDefaultShipmentAddress(addressModel);
    }

    @Test
    public void shouldNotSetAsDefaultIfAddressIsNotVisible(){
        when(addressModel.getVisibleInAddressBook()).thenReturn(Boolean.FALSE);
        unit.setDefaultAddressEntry(customerModel, addressModel);
        verify(customerModel, never()).setDefaultPaymentAddress(addressModel);
        verify(customerModel, never()).setDefaultShipmentAddress(addressModel);
    }


    @Test
    public void shouldRemoveDuplicatePMIfItExistsByPayerEmail() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL);
        when(payPalPaymentInfoModel.getDuplicate()).thenReturn(false);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(true);
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.removeDuplicatePaymentMethod(PAYER_EMAIL);

        verify(unit).deleteCCPaymentInfo(customerModel, payPalPaymentInfoModel);
    }

    @Test
    public void shouldNotRemoveDuplicatePMIfItNotExistsByPayerEmail() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.emptyList());

        unit.removeDuplicatePaymentMethod(PAYER_EMAIL);

        verify(unit, never()).deleteCCPaymentInfo(any(), any());
    }

    @Test
    public void shouldNotRemovePMWhenPMIsHostedFields() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getDuplicate()).thenReturn(false);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(true);
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);

        unit.removeDuplicatePaymentMethod(Strings.EMPTY);

        verify(unit, never()).deleteCCPaymentInfo(any(), any());
    }

    @Test
    public void shouldNotRemovePMWhenPMIsDuplicate() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL);
        when(payPalPaymentInfoModel.getDuplicate()).thenReturn(true);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(true);
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.removeDuplicatePaymentMethod(PAYER_EMAIL);

        verify(unit, never()).deleteCCPaymentInfo(any(), any());
    }

    @Test
    public void shouldNotRemovePMWhenPMIsNotSaved() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(PAYER_EMAIL);
        when(payPalPaymentInfoModel.getDuplicate()).thenReturn(false);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(false);
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.removeDuplicatePaymentMethod(PAYER_EMAIL);

        verify(unit, never()).deleteCCPaymentInfo(any(), any());
    }

    @Test
    public void shouldNotRemovePMWhenPayerEmailIsDifferent() {
        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(payPalPaymentInfoModel));
        when(payPalPaymentInfoModel.getPayerEmail()).thenReturn(SECOND_PAYER_EMAIL);
        when(payPalPaymentInfoModel.getDuplicate()).thenReturn(false);
        when(payPalPaymentInfoModel.isSaved()).thenReturn(false);
        when(payPalPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL);

        unit.removeDuplicatePaymentMethod(PAYER_EMAIL);

        verify(unit, never()).deleteCCPaymentInfo(any(), any());
    }

    @Test
    public void shouldRetrieveCustomerFromUserServiceWhenPayerIdIsNotPresentOnCustomer() {
        when(unit.getCustomerModelByPayerId(PAYER_ID)).thenReturn(null);
        when(userService.getCurrentUser()).thenReturn(customerModel);

        CustomerModel returnedCustomerModel = unit.getCustomerModel(PAYER_ID);

        verify(userService, times(1)).getCurrentUser();
        assertSame(customerModel, returnedCustomerModel);
    }

}
