/*

 */
package com.paypal.hybris.core.service.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.enums.PayPalPageType;
import com.paypal.hybris.core.commands.impl.DefaultPayPalConnectExchangeCodeToAccessTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalConnectGetUserInformationCommand;
import com.paypal.hybris.core.exception.PayPalConnectException;
import com.paypal.hybris.data.PayPalConnectAccessTokenData;
import com.paypal.hybris.data.PayPalConnectUserData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultPayPalConnectServiceTest {

    private static final String AUTHORIZATION_TOKEN = "authorizationToken";
    private static final String ACCESS_TOKEN = "accessToken";
    private static final String ID_TOKEN = "idToken";

    @Mock
    private CommandFactoryRegistry commandFactoryRegistry;

    @Mock
    private CommandFactory commandFactory;

    @Mock
    private DefaultPayPalConnectExchangeCodeToAccessTokenCommand command;

    @Mock
    private DefaultPayPalConnectGetUserInformationCommand defaultPayPalConnectGetUserInformationCommand;

    @Mock
    private PayPalConnectAccessTokenData data;

    @Mock
    private PayPalConnectUserData payPalConnectUserData;

    @Mock
    private DefaultPayPalConfigurationService payPalConfigurationService;

    @InjectMocks
    private DefaultPayPalConnectService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void exchangeAuthorizationCodeToAccessTokenShouldReturnAccessToken() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectExchangeCodeToAccessTokenCommand.class)).thenReturn(command);
        when(command.perform(any())).thenReturn(data);
        when(data.getAccess_token()).thenReturn(ACCESS_TOKEN);

        assertEquals(ACCESS_TOKEN, unit.exchangeAuthorizationCodeToAccessToken(AUTHORIZATION_TOKEN));
        verify(command).perform(any());
    }

    @Test
    public void shouldReturnIdToken() throws CommandNotSupportedException {
        when(payPalConfigurationService.isChangePaymentButtonEnabled()).thenReturn(Boolean.TRUE);
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectExchangeCodeToAccessTokenCommand.class)).thenReturn(command);
        when(command.perform(any())).thenReturn(data);
        when(data.getId_token()).thenReturn(ID_TOKEN);

        assertEquals(ID_TOKEN, unit.getIdToken(PayPalPageType.ACCOUNT));
        verify(command).perform(any());
    }

    @Test(expected = PayPalConnectException.class)
    public void shouldThrowPayPalConnectExceptionWhenCommandNotSupported() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectExchangeCodeToAccessTokenCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.getIdToken(PayPalPageType.ACCOUNT);
    }

    @Test(expected = PayPalConnectException.class)
    public void exchangeAuthorizationCodeToAccessTokenShouldReturnAccessTokenWithException() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectExchangeCodeToAccessTokenCommand.class)).thenReturn(command);
        when(command.perform(any())).thenReturn(data);
        when(data.getAccess_token()).thenThrow(AdapterException.class);

        unit.exchangeAuthorizationCodeToAccessToken(AUTHORIZATION_TOKEN);

    }

    @Test
    public void getUserDataByAccessTokenShouldReturnData() throws CommandNotSupportedException {
        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectGetUserInformationCommand.class)).thenReturn(defaultPayPalConnectGetUserInformationCommand);
        when(defaultPayPalConnectGetUserInformationCommand.perform(ACCESS_TOKEN)).thenReturn(payPalConnectUserData);

        assertEquals(payPalConnectUserData, unit.getUserDataByAccessToken(ACCESS_TOKEN));
        verify(defaultPayPalConnectGetUserInformationCommand).perform(ACCESS_TOKEN);
    }
    @Test(expected = PayPalConnectException.class)
    public void getUserDataByAccessTokenShouldReturnDataWithException() throws CommandNotSupportedException {

        when(commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)).thenReturn(commandFactory);
        when(commandFactory.createCommand(DefaultPayPalConnectGetUserInformationCommand.class)).thenThrow(CommandNotSupportedException.class);

        unit.getUserDataByAccessToken(ACCESS_TOKEN);

    }
}
