/**
 *
 */
package com.paypal.hybris.core.service.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_VAULT_SAVED_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE_PROCESS_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_REPLENISHMENT_FLOW;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.PayPalCommercePaymentProviderStrategy;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commerceservices.order.CommercePaymentAuthorizationStrategy;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionService;
import java.math.BigDecimal;
import java.util.Map;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultPayPalCommerceCheckoutServiceTest {

    private static final String DEFAULT_FLOW = "DEFAULT_FLOW";
    private static final String ORDER_PROCESS_FLOW = "ORDER_PROCESS_FLOW";
    private static final String TEST_INTENT = "TestIntent";
    private static final String PAYPAL_ORDER_PROCESS_FLOW = "ORDER_PROCESS_FLOW";
    private static final String PAYPAL_ORDER_ID_PLACEHOLDER = "ORDER_ID_PLACEHOLDER";
    private static final String PAYPAL_NONDECIMAL_CURRENCY = "JPY,HUF,TWD";
    private static final String CURRENCY = "USD";
    private static final String PAYPAL = "payPal";

    @Mock
    private ModelService modelService;

    @Mock
    private DefaultPayPalConfigurationService defaultPayPalConfigurationService;

    @Mock
    private SessionService sessionService;

    @Mock
    private Map<String, CommercePaymentAuthorizationStrategy> strategyMap;
    @Mock
    private PayPalPaymentService payPalPaymentService;

    @Mock
    private PayPalCommercePaymentProviderStrategy commercePaymentProviderStrategy;

    @Mock
    private CommercePaymentAuthorizationStrategy strategy;

    @Mock
    private CartModel cartModel;

    @Mock
    private CommerceCheckoutParameter parameter;

    @Mock
    private PayPalCommerceCheckoutParameter payPalCommerceCheckoutParameter;

    @Mock
    private PaymentInfoModel paymentInfoModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel;

    @Mock
    private Converter<CartModel, PayPalOrderRequestData> orderRequestDataConverter;

    @InjectMocks
    private DefaultPayPalCommerceCheckoutService unit;

    private BigDecimal authorizationAmount;

    private CartData cartData;

    private PriceData priceData;
    private PayPalOrderRequestData requestData;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(TEST_INTENT);

        authorizationAmount = BigDecimal.ONE;
        cartData = new CartData();
        priceData = new PriceData();
        requestData = new PayPalOrderRequestData();
        cartData.setTotalTax(priceData);
        cartData.setDeliveryCost(priceData);
        cartData.setTotalDiscounts(priceData);
        when(orderRequestDataConverter.convert(cartModel)).thenReturn(requestData);
    }

    @Test
    public void shouldReturnPaymentProvider() {
        when(commercePaymentProviderStrategy.getPaymentProvider(cartModel)).thenReturn(PAYPAL);

        String result = unit.getPaymentProvider(cartModel);

        Assert.assertEquals(PAYPAL, result);
        verify(commercePaymentProviderStrategy).getPaymentProvider(cartModel);
    }

    @Test
    public void shouldAuthorizePaymentAuthorizationWhenAmountNull() {
        when(cartModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.getOrDefault(eq(DEFAULT_FLOW), any())).thenReturn(strategy);

        unit.authorizePayment(parameter);

        verify(parameter).setAuthorizationAmount(any());
        verify(modelService, never()).refresh(cartModel);
        verify(strategy).authorizePaymentAmount(parameter);

    }

    @Test
    public void shouldAuthorizePaymentAuthorizationWithPayPalReplenishmentFlow() {
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCommerceCheckoutParameter.getCart()).thenReturn(cartModel);
        when(payPalCommerceCheckoutParameter.getAuthorizationAmount()).thenReturn(null);
        when(payPalCommerceCheckoutParameter.isReplenishment()).thenReturn(Boolean.TRUE);
        when(strategyMap.getOrDefault(eq(PAYPAL_REPLENISHMENT_FLOW), any())).thenReturn(strategy);

        unit.authorizePayment(payPalCommerceCheckoutParameter);

        verify(strategy).authorizePaymentAmount(payPalCommerceCheckoutParameter);
    }

    @Test
    public void shouldAuthorizePaymentAuthorizationWithPayPalHostedFieldsSavedFlow() {
        when(parameter.getCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getPaymentProvider()).thenReturn(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        when(payPalCreditCardPaymentInfoModel.isSaved()).thenReturn(Boolean.TRUE);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.getOrDefault(eq(PAYPAL_VAULT_SAVED_FLOW), any())).thenReturn(strategy);

        unit.authorizePayment(parameter);

        verify(strategy).authorizePaymentAmount(parameter);
    }

    @Test
    public void shouldAuthorizePaymentAuthorizationWithPayPalIntentCaptureProcessFlow() {
        when(parameter.getCart()).thenReturn(cartModel);
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.getOrDefault(eq(PAYPAL_INTENT_CAPTURE_PROCESS_FLOW), any())).thenReturn(strategy);

        unit.authorizePayment(parameter);

        verify(strategy).authorizePaymentAmount(parameter);
    }

    @Test
    public void shouldAuthorizePaymentAuthorizationWhenAmountNotNull() {
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(authorizationAmount);
        when(strategyMap.getOrDefault(eq(ORDER_PROCESS_FLOW), any())).thenReturn(strategy);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn("any");
        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn("any");

        unit.authorizePayment(parameter);

        verify(parameter, never()).setAuthorizationAmount(any());
        verify(strategy).authorizePaymentAmount(parameter);

    }

    @Test
    public void shouldCreatePayPalOrder() {
        cartData.setTotalPrice(priceData);
        cartData.setTotalPriceWithTax(priceData);
        priceData.setValue(BigDecimal.ONE);
        priceData.setCurrencyIso(CURRENCY);

        when(payPalCreditCardPaymentInfoModel.getIntent()).thenReturn(TEST_INTENT);
        when(payPalCreditCardPaymentInfoModel.getSubscriptionId()).thenReturn(PAYPAL_ORDER_ID_PLACEHOLDER);
        when(payPalCreditCardPaymentInfoModel.getPayPalOrderId()).thenReturn(PAYPAL_ORDER_ID_PLACEHOLDER);
        when(cartModel.getPaymentInfo()).thenReturn(payPalCreditCardPaymentInfoModel);
        when(parameter.getCart()).thenReturn(cartModel);
        when(parameter.getAuthorizationAmount()).thenReturn(null);
        when(strategyMap.getOrDefault(eq(PAYPAL_ORDER_PROCESS_FLOW), any())).thenReturn(strategy);
        when(defaultPayPalConfigurationService.getNonDecimalCurrency()).thenReturn(PAYPAL_NONDECIMAL_CURRENCY);

        unit.authorizePayment(parameter);

        verify(payPalCreditCardPaymentInfoModel).setIntent(TEST_INTENT);
        verify(modelService).refresh(cartModel);
    }

}
