/*

 */
package com.paypal.hybris.core.service.impl;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commerceservices.address.AddressErrorCode;
import de.hybris.platform.commerceservices.address.AddressFieldType;
import de.hybris.platform.commerceservices.address.AddressVerificationDecision;
import de.hybris.platform.commerceservices.address.data.AddressFieldErrorData;
import de.hybris.platform.commerceservices.address.data.AddressVerificationResultData;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.user.AddressModel;
import java.util.ArrayList;
import java.util.List;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.UUID;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;

@UnitTest
public class DefaultPayPalAddressVerificationServiceTest {

    private static final String REVIEW = "review";
    private static final String FIRST_NAME = "John";
    private static final String LAST_NAME = "Doe";
    private static final String ADDRESS_LINE_1 = "addressline1";
    private static final String ADDRESS_LINE_2 = "addressline2";
    private static final String ZIP_CODE = "1234567";
    private static final String CITY = "city";
    private static final String TITLE_CODE = "titleCode";
    private static final String MISSING = "missing";

    @Mock
    private AddressModel addressModel;

    @Mock
    private BaseStoreService baseStoreService;

    @Mock
    private CountryModel countryModel;

    @Mock
    private RegionModel regionModel;

    @InjectMocks
    private DefaultPayPalAddressVerificationService unit = new DefaultPayPalAddressVerificationService();

    private String bigLine;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        StringBuilder sb = new StringBuilder();
        sb.append("a".repeat(260));
        bigLine = sb.toString();

        when(addressModel.getRegion()).thenReturn(regionModel);
        when(addressModel.getCountry()).thenReturn(countryModel);
        when(addressModel.getRegion().getIsocode()).thenReturn(UUID.randomUUID().toString());
        when(addressModel.getCountry().getIsocode()).thenReturn(UUID.randomUUID().toString());
        when(addressModel.getPostalcode()).thenReturn(ZIP_CODE);
        when(addressModel.getTown()).thenReturn(CITY);
        when(addressModel.getLine1()).thenReturn(ADDRESS_LINE_1);
        when(addressModel.getLine2()).thenReturn(ADDRESS_LINE_2);
        when(addressModel.getLastname()).thenReturn(LAST_NAME);
        when(addressModel.getFirstname()).thenReturn(FIRST_NAME);
    }

    @Test
    public void shouldValidateAddressFieldsWithError() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult =
            new AddressVerificationResultData<>();
        when(addressModel.getFirstname()).thenReturn(null);
        when(addressModel.getLastname()).thenReturn(null);
        when(addressModel.getPostalcode()).thenReturn(UUID.randomUUID().toString());
        when(addressModel.getCountry()).thenReturn(null);
        when(addressModel.getLine1()).thenReturn(null);
        when(addressModel.getTown()).thenReturn(null);
        when(addressModel.getRegion().getIsocode()).thenReturn(null);
        int expectedErrorNumbers = 7;

        unit.validateAddressFields(acceptedResult, addressModel);

        assertEquals(expectedErrorNumbers, acceptedResult.getFieldErrors().size());
    }

    @Test
    public void shouldValidateAddressFieldsWithErrorWhenLinesMoreThanAcceptable() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult =
                new AddressVerificationResultData<>();
        when(addressModel.getLine2()).thenReturn(bigLine);
        when(addressModel.getRegion().getIsocode()).thenReturn(bigLine);
        when(addressModel.getCountry().getIsocode()).thenReturn(bigLine);
        int expectedErrorNumbers = 3;

        unit.validateAddressFields(acceptedResult, addressModel);

        assertEquals(expectedErrorNumbers, acceptedResult.getFieldErrors().size());
    }



    @Test
    public void shouldValidateAddressFieldsWithoutErrors() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult =
            new AddressVerificationResultData<>();
        int expectedErrorNumber = 0;

        unit.validateAddressFields(acceptedResult, addressModel);

        assertEquals(expectedErrorNumber, acceptedResult.getFieldErrors().size());
    }

    @Test
    public void shouldReviewAddress() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult;
        when(addressModel.getTown()).thenReturn(REVIEW);

        acceptedResult = unit.verifyAddress(addressModel);

        assertEquals(AddressVerificationDecision.REVIEW, acceptedResult.getDecision());
        assertNotNull(acceptedResult.getSuggestedAddresses());
    }

    @Test
    public void shouldAcceptAddress() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult;

        acceptedResult = unit.verifyAddress(addressModel);

        assertEquals(AddressVerificationDecision.ACCEPT, acceptedResult.getDecision());
    }

    @Test
    public void shouldRejectAddress() {
        AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult;
        when(addressModel.getFirstname()).thenReturn(null);

        acceptedResult = unit.verifyAddress(addressModel);

        assertEquals(AddressVerificationDecision.REJECT, acceptedResult.getDecision());
    }

    @Test
    public void isCustomerAllowedToIgnoreSuggestions() {
        final BaseStoreModel baseStore = mock(BaseStoreModel.class);

        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStore);
        when(baseStore.isCustomerAllowedToIgnoreSuggestions()).thenReturn(true);

        boolean result = unit.isCustomerAllowedToIgnoreSuggestions();

        assertTrue(result);
        verify(baseStoreService).getCurrentBaseStore();
        verify(baseStore).isCustomerAllowedToIgnoreSuggestions();
    }

    @Test
    public void addErrorToVerificationResult() {
        List<AddressFieldErrorData<AddressFieldType, AddressErrorCode>> errors = new ArrayList<>();

        unit.addErrorToVerificationResult(TITLE_CODE, MISSING, errors);

        assertTrue(errors.size() > 0);
    }
}
