package com.paypal.hybris.core.jobs;

import com.paypal.hybris.core.dao.PayPalPaymentInfoDao;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.model.ValidateExpiredCardsCronJobModel;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.servicelayer.data.PaginationData;
import de.hybris.platform.core.servicelayer.data.SearchPageData;
import de.hybris.platform.cronjob.enums.CronJobResult;
import de.hybris.platform.cronjob.enums.CronJobStatus;
import de.hybris.platform.servicelayer.cronjob.PerformResult;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Answers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.*;

@UnitTest
public class ValidateExpiredCardsJobPerformableTest {

    private static final int PAGE_SIZE = 1;
    private static final int NEGATIVE_EXPIRE_BETWEEN = -12;
    private static final int POSITIVE_EXPIRE_BETWEEN = 10;
    private static final String MONTH = "01";
    private static final String YEAR = "2020";
    private static final String WRONG_MONTH = "2222";

    @Mock
    private PayPalPaymentInfoDao payPalPaymentInfoDao;

    @Mock
    private ModelService modelService;

    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;

    @Mock
    CustomerModel customerModel;

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private ValidateExpiredCardsCronJobModel cronJobModel;

    @InjectMocks
    private ValidateExpiredCardsJobPerformable unit;

    private PaginationData paginationData;

    private SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData;

    private PerformResult expectedResult;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        expectedResult = new PerformResult(CronJobResult.SUCCESS, CronJobStatus.FINISHED);

        paginationData = GenericBuilder.of(PaginationData::new)
                .with(PaginationData::setCurrentPage, 0)
                .with(PaginationData::setNeedsTotal, true)
                .with(PaginationData::setPageSize, 1)
                .with(PaginationData::setHasNext,  false)
                .build();

        searchPageData = GenericBuilder.of(SearchPageData<PayPalCreditCardPaymentInfoModel>::new)
                .with(SearchPageData::setPagination, paginationData)
                .with(SearchPageData::setResults, List.of(paymentInfoModel))
                .build();
    }

    @Test
    public void shouldPerformWhenExpireBetweenLessThanZero() {
        when(cronJobModel.getStore().getPayPalPluginConfiguration().getHostedFieldsConfiguration().getExpireBetween()).thenReturn(NEGATIVE_EXPIRE_BETWEEN);
        when(cronJobModel.getPageSize()).thenReturn(PAGE_SIZE);

        when(payPalPaymentInfoDao.findAllExpiredCards(any())).thenReturn(searchPageData);

        when(paymentInfoModel.getValidToMonth()).thenReturn(MONTH);
        when(paymentInfoModel.getValidToYear()).thenReturn(YEAR);

        PerformResult actualResult = unit.perform(cronJobModel);

        assertEquals(expectedResult.getResult(), actualResult.getResult());
        assertEquals(expectedResult.getStatus(), actualResult.getStatus());

        verify(modelService, times(1)).save(paymentInfoModel);

    }

    @Test
    public void shouldPerformWhenExpireBetweenMoreThanZero() {
        when(cronJobModel.getStore().getPayPalPluginConfiguration().getHostedFieldsConfiguration().getExpireBetween()).thenReturn(POSITIVE_EXPIRE_BETWEEN);
        when(cronJobModel.getPageSize()).thenReturn(PAGE_SIZE);

        when(payPalPaymentInfoDao.findAllExpiredCards(any())).thenReturn(searchPageData);

        when(paymentInfoModel.getValidToMonth()).thenReturn(MONTH);
        when(paymentInfoModel.getValidToYear()).thenReturn(YEAR);

        when(payPalPaymentInfoDao.findAllExpiredSoon(any(), any())).thenReturn(searchPageData);
        when(paymentInfoModel.getUser()).thenReturn(customerModel);

        PerformResult actualResult = unit.perform(cronJobModel);

        assertEquals(expectedResult.getResult(), actualResult.getResult());
        assertEquals(expectedResult.getStatus(), actualResult.getStatus());

        verify(modelService, times(1)).saveAll(paymentInfoModel, customerModel);

    }

    @Test
    public void shouldPerformWhenFailedExpirationDateValidationWhenMonthWrong() {
        when(cronJobModel.getStore().getPayPalPluginConfiguration().getHostedFieldsConfiguration().getExpireBetween()).thenReturn(POSITIVE_EXPIRE_BETWEEN);
        when(cronJobModel.getPageSize()).thenReturn(PAGE_SIZE);

        when(payPalPaymentInfoDao.findAllExpiredCards(any())).thenReturn(searchPageData);

        when(paymentInfoModel.getValidToMonth()).thenReturn(WRONG_MONTH);
        when(paymentInfoModel.getValidToYear()).thenReturn(YEAR);

        when(payPalPaymentInfoDao.findAllExpiredSoon(any(), any())).thenReturn(searchPageData);
        when(paymentInfoModel.getUser()).thenReturn(customerModel);

        PerformResult actualResult = unit.perform(cronJobModel);

        assertEquals(expectedResult.getResult(), actualResult.getResult());
        assertEquals(expectedResult.getStatus(), actualResult.getStatus());

        verify(modelService, never()).save(paymentInfoModel);
        verify(modelService, never()).saveAll(paymentInfoModel, customerModel);
    }

    @Test
    public void shouldPerformWhenFailedExpirationDateValidationWhenYearNull() {
        when(cronJobModel.getStore().getPayPalPluginConfiguration().getHostedFieldsConfiguration().getExpireBetween()).thenReturn(POSITIVE_EXPIRE_BETWEEN);
        when(cronJobModel.getPageSize()).thenReturn(PAGE_SIZE);

        when(payPalPaymentInfoDao.findAllExpiredCards(any())).thenReturn(searchPageData);

        when(paymentInfoModel.getValidToMonth()).thenReturn(MONTH);
        when(paymentInfoModel.getValidToYear()).thenReturn(null);

        when(payPalPaymentInfoDao.findAllExpiredSoon(any(), any())).thenReturn(searchPageData);
        when(paymentInfoModel.getUser()).thenReturn(customerModel);

        PerformResult actualResult = unit.perform(cronJobModel);

        assertEquals(expectedResult.getResult(), actualResult.getResult());
        assertEquals(expectedResult.getStatus(), actualResult.getStatus());

        verify(modelService, never()).save(paymentInfoModel);
        verify(modelService, never()).saveAll(paymentInfoModel, customerModel);
    }


}