package com.paypal.hybris.core.executor.impl;

import com.paypal.hybris.core.service.PayPalPaymentService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.enums.OrderEntryStatus;
import de.hybris.platform.core.PK;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordercancel.OrderCancelEntry;
import de.hybris.platform.ordercancel.OrderCancelException;
import de.hybris.platform.ordercancel.OrderCancelNotificationServiceAdapter;
import de.hybris.platform.ordercancel.OrderCancelPaymentServiceAdapter;
import de.hybris.platform.ordercancel.OrderCancelRecordsHandler;
import de.hybris.platform.ordercancel.OrderCancelRequest;
import de.hybris.platform.ordercancel.OrderStatusChangeStrategy;
import de.hybris.platform.ordercancel.model.OrderCancelRecordEntryModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.List;

import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalImmediateCancelRequestExecutorTest {

    private static final String CODE = "code";

    @Mock
    private ModelService modelService;

    @Mock
    private OrderCancelPaymentServiceAdapter paymentServiceAdapter;

    @Mock
    private PayPalPaymentService paymentService;

    @Mock
    private OrderCancelRecordsHandler orderCancelRecordsHandler;

    @Mock
    private OrderStatusChangeStrategy completeCancelStatusChangeStrategy;

    @Mock
    private OrderStatusChangeStrategy partialCancelStatusChangeStrategy;

    @Mock
    private OrderCancelNotificationServiceAdapter notificationServiceAdapter;

    @Mock
    private OrderCancelRequest orderCancelRequest;

    @Mock
    private OrderCancelEntry orderCancelEntry;

    @Mock
    private OrderCancelRecordEntryModel cancelRequestRecordEntry;

    @Mock
    private AbstractOrderEntryModel orderEntry;

    @Mock
    private OrderModel order;

    @Mock
    private OrderEntryModel entryLiving;

    @InjectMocks
    private DefaultPayPalImmediateCancelRequestExecutor unit;

    private List<OrderCancelEntry> orderCancelEntries;


    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(entryLiving.getQuantityStatus()).thenReturn(OrderEntryStatus.LIVING);
        when(entryLiving.getQuantity()).thenReturn(12L);

        orderCancelEntries = List.of(orderCancelEntry);
    }

    @Test
    public void shouldProcessCancelRequestWhenHasNotLivingEntries() throws OrderCancelException {
        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(12L);

        when(orderCancelRequest.getOrder()).thenReturn(order);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);

        verify(completeCancelStatusChangeStrategy).changeOrderStatusAfterCancelOperation(cancelRequestRecordEntry, true);
        verify(paymentServiceAdapter).recalculateOrderAndModifyPayments(order);
        verify(notificationServiceAdapter).sendCancelFinishedNotifications(cancelRequestRecordEntry);
        verify(orderEntry).setQuantityStatus(OrderEntryStatus.DEAD);
        verify(modelService).save(orderEntry);
    }

    @Test
    public void shouldProcessCancelRequestWhenHasNotLivingEntriesAndCompleteCancelStatusChangeStrategyIsNull() throws OrderCancelException {
        unit.setCompleteCancelStatusChangeStrategy(null);

        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(12L);

        when(orderCancelRequest.getOrder()).thenReturn(order);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);

        verify(paymentServiceAdapter).recalculateOrderAndModifyPayments(order);
        verify(notificationServiceAdapter).sendCancelFinishedNotifications(cancelRequestRecordEntry);
        verify(orderEntry).setQuantityStatus(OrderEntryStatus.DEAD);
        verify(modelService).save(orderEntry);
    }

    @Test
    public void shouldProcessCancelRequestHasLivingEntries() throws OrderCancelException {
        when(order.getEntries()).thenReturn(List.of(entryLiving));

        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(12L);

        when(orderCancelRequest.getOrder()).thenReturn(order);

        when(order.getCode()).thenReturn(CODE);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);

        verify(partialCancelStatusChangeStrategy).changeOrderStatusAfterCancelOperation(cancelRequestRecordEntry, true);
        verify(paymentServiceAdapter).recalculateOrderAndModifyPayments(order);
        verify(notificationServiceAdapter).sendCancelFinishedNotifications(cancelRequestRecordEntry);
        verify(orderEntry).setQuantityStatus(OrderEntryStatus.DEAD);
        verify(modelService).save(orderEntry);
    }

    @Test
    public void shouldProcessCancelRequestHasLivingEntriesAndPaymentServiceAdapterAndPartialCancelStrategyAndNotificationServiceAdapterIsNull() throws OrderCancelException {
        unit.setPartialCancelStatusChangeStrategy(null);
        unit.setPaymentServiceAdapter(null);
        unit.setNotificationServiceAdapter(null);

        when(order.getEntries()).thenReturn(List.of(entryLiving));

        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(12L);


        when(orderCancelRequest.getOrder()).thenReturn(order);

        when(order.getCode()).thenReturn(CODE);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);

        verify(orderEntry).setQuantityStatus(OrderEntryStatus.DEAD);
        verify(modelService).save(orderEntry);
    }


    @Test(expected = OrderCancelException.class)
    public void shouldThrowOrderCancelExceptionWhenCancelQuantityMoreThanPreviousQuantity() throws OrderCancelException {
        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(13L);

        when(orderCancelRequest.getOrder()).thenReturn(order);
        when(order.getCode()).thenReturn(CODE);

        when(orderEntry.getPk()).thenReturn(PK.BIG_PK);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);

    }


    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenAttemptToAddOrderEntryThatBelongsToAnotherOrder() throws OrderCancelException {
        unit.setNotificationServiceAdapter(null);

        when(order.getEntries()).thenReturn(List.of(entryLiving));

        when(orderCancelRequest.getEntriesToCancel()).thenReturn(orderCancelEntries);
        when(orderCancelEntry.getOrderEntry()).thenReturn(orderEntry);
        when(orderEntry.getQuantity()).thenReturn(12L);

        when(orderCancelEntry.getCancelQuantity()).thenReturn(12L);

        when(orderCancelRequest.getOrder()).thenReturn(order);
        when(orderCancelRequest.isPartialCancel()).thenReturn(Boolean.TRUE);

        when(orderCancelRequest.getOrder()).thenReturn(order);
        when(order.getCode()).thenReturn(CODE);

        unit.processCancelRequest(orderCancelRequest, cancelRequestRecordEntry);
    }
}