/**
 *
 */
package com.paypal.hybris.core.dao.impl;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.servicelayer.data.SearchPageData;
import de.hybris.platform.servicelayer.exceptions.ModelNotFoundException;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.search.FlexibleSearchQuery;
import de.hybris.platform.servicelayer.search.FlexibleSearchService;
import de.hybris.platform.servicelayer.search.paginated.PaginatedFlexibleSearchParameter;
import de.hybris.platform.servicelayer.search.paginated.PaginatedFlexibleSearchService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.time.YearMonth;
import java.util.List;
import java.util.Optional;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalPaymentInfoDaoTest {

    private static final String PAY_PAL_ORDER_ID = "payPalOrderId";

    private static final String PK = "PK";

    private static final String SUBSCRIPTIONID = "subscriptionId";

    @Mock
    private FlexibleSearchService flexibleSearchService;

    @Mock
    private PaginatedFlexibleSearchService paginatedFlexibleSearchService;

    @Mock
    private PayPalCreditCardPaymentInfoModel payPalCreditCardPaymentInfoModel;

    @Mock
    private ModelService modelService;

    @InjectMocks
    private DefaultPayPalPaymentInfoDao unit;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldFindPaymentInfoByPayPalOrderId() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenReturn(payPalCreditCardPaymentInfoModel);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findPaymentInfoByPayPalOrderId(PAY_PAL_ORDER_ID);

        assertTrue(result.isPresent());
        assertEquals(payPalCreditCardPaymentInfoModel, result.get());
    }

    @Test
    public void shouldNotFindPaymentInfoByPayPalOrderId() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenThrow(ModelNotFoundException.class);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findPaymentInfoByPayPalOrderId(PAY_PAL_ORDER_ID);

        assertTrue(result.isEmpty());
    }

    @Test
    public void shouldFindPaymentInfoByPK() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenReturn(payPalCreditCardPaymentInfoModel);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findPaymentInfoByPK(PK);

        assertTrue(result.isPresent());
        assertEquals(payPalCreditCardPaymentInfoModel, result.get());
    }

    @Test
    public void shouldNotFindPaymentInfoByPK() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenThrow(ModelNotFoundException.class);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findPaymentInfoByPK(PK);

        assertTrue(result.isEmpty());
    }

    @Test
    public void shouldRemovePaymentInfoByPK() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenReturn(payPalCreditCardPaymentInfoModel);

        unit.removePaymentInfoByPK(PK);

        verify(modelService).remove(payPalCreditCardPaymentInfoModel);
    }

    @Test
    public void shouldNotRemovePaymentInfoByPK() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenThrow(ModelNotFoundException.class);

        unit.removePaymentInfoByPK(PK);

        verify(modelService, times(0)).remove(payPalCreditCardPaymentInfoModel);
    }

    @Test
    public void shouldFindCardBySubscriptionId() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenReturn(payPalCreditCardPaymentInfoModel);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findCardBySubscriptionId(SUBSCRIPTIONID);

        assertTrue(result.isPresent());
        assertEquals(payPalCreditCardPaymentInfoModel, result.get());
    }

    @Test
    public void shouldReturnEmptyOptionalWhenModelNotFound() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenThrow(ModelNotFoundException.class);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findCardBySubscriptionId(SUBSCRIPTIONID);

        assertTrue(result.isEmpty());
    }

    @Test
    public void shouldNotFindCardBySubscriptionId() {
        when(flexibleSearchService.searchUnique(any(FlexibleSearchQuery.class))).thenThrow(ModelNotFoundException.class);

        Optional<PayPalCreditCardPaymentInfoModel> result = unit.findPaymentInfoByPK(SUBSCRIPTIONID);

        assertTrue(result.isEmpty());
    }

    @Test
    public void shouldFindAllExpiredCards() {
        List<PayPalCreditCardPaymentInfoModel> expectedList = List.of(payPalCreditCardPaymentInfoModel);

        SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData = new SearchPageData<>();
        searchPageData.setResults(expectedList);

        when(paginatedFlexibleSearchService.<PayPalCreditCardPaymentInfoModel>search(any(PaginatedFlexibleSearchParameter.class))).thenReturn(searchPageData);

        SearchPageData<PayPalCreditCardPaymentInfoModel> result = unit.findAllExpiredCards(searchPageData);

        assertEquals(expectedList, result.getResults());
    }

    @Test
    public void shouldFindAllExpiredSoon() {
        List<PayPalCreditCardPaymentInfoModel> expectedList = List.of(payPalCreditCardPaymentInfoModel);
        SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData = new SearchPageData<>();
        YearMonth yearMonth = YearMonth.of(2024, 01);
        searchPageData.setResults(expectedList);

        when(paginatedFlexibleSearchService.<PayPalCreditCardPaymentInfoModel>search(any(PaginatedFlexibleSearchParameter.class))).thenReturn(searchPageData);

        SearchPageData<PayPalCreditCardPaymentInfoModel> result = unit.findAllExpiredSoon(yearMonth, searchPageData);

        assertEquals(expectedList, result.getResults());
    }
}
